;******************************************************************************
;*  NAME:        a_lanl_mpa
;*
;*  DESCRIPTION: example routine for access to lanl_geo module's data
;*               For use inside the PAPCO environment 
;*               (papco is compiled, required modules are loaded)
;*               
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               T1       -  start date/time in any PAPCO format, string
;*               T2       -  end date/time in any PAPCO format, string
;*         
;*  Enter a time in one of these formats: date time, date
;*  where time is in one of those formats: h:m:s, h:m
;*  and date is in one of those formats: m/d/y, doy/y, d.m.y, m/d/y, yyyymmdd
;*  where d is day and doy is day of year, h:m:s is the time
;*  (year > 1990 if you use T90 format)
;*  e.g: 1/90 14:11, 1/1990 1:1:1, 350/1991 01:00:03
;*       15.1.90 14:11, 1/15/1990 15:3:1                  
;*
;*  DATE:        March 2003
;*
;*  AUTHOR:       R. Friedel
;*
;*  CHANGES:   changed to work for mpa instead of sopa  (Arne Aasnes,
;*             LANL, april, 2003) 
;*
;* Example call line: a_lanl_mpa,SAT_ID='LANL-97A',T1='19980724',T2='19980725'
;******************************************************************************
PRO a_lanl_mpa, SAT_ID = sat_id, T1 = T1, T2 = T2, EL_ION = el_ion, FLUX = flux 

;first, we need to call papco once so that it initializes itself. This
;sets up a bunch of common blocks, structures and environmental
;variables that are needed by various modules.

papco, /set_only ;only initializes papco then exists.

IF NOT keyword_set(T1) THEN BEGIN
    message, 'No start time (T1) defined', /cont & return
ENDIF 

IF NOT keyword_set(T2) THEN BEGIN
    message, 'No end time (T2) defined', /cont & return
ENDIF 


;PAPCO sets the start and end time in a common block, which controls
;the time range for all read routines and all plot routines in all
;papco modules that are loaded.
COMMON mjdt, mjdt_start, mjdt_end

;The time format in mjdt_start, mjdt_end is modified Julian date and
;seconds of day, in a structure:
; mjdt_start = {MJD:0l, T:0.0}

;PAPCO provides a set of time conversion routines that can convert
;between string times to papco time and a variety of other formats:
;look in ../papco/papco_lib/convert_time.pro

;for the example here, if the input string is in any of the PAPCO time
;formats, use the function convert_timeToSeconds to convert to
;mjdt_start, mjdt_end. This function returns a structure containing
;the tags err, value and errtext. If err is zero, the time string was
;recognized, and the converted time is in value. If err eq -1 there
;was an error, the error message is in errtext. This errtext the also
;contains a list of the "allowed" formats...

result = convert_timeToSeconds(T1, /T90, /MJDT)
IF result.err EQ 0 THEN mjdt_start =result.value ELSE BEGIN
    message, 'Error in start time (T1)', /cont
    print,  result.errtext
    return
ENDELSE

result = convert_timeToSeconds(T2, /T90, /MJDT)
IF result.err EQ 0 THEN mjdt_end =result.value ELSE BEGIN
    message, 'Error in end time (T2)', /cont
    print,  result.errtext
    return
ENDELSE 

;if you don't like the papco time formats you can also use the time
;routines provided under
;../papco/papco_lib/cdssw/time/
;these will come in handy in any case, since a lot of the papco
;modules use the TAI format internally - international atomic time.
;There's an informative readme file in that directory too.
;Here's a quick way of converting mjdt_start, mjdt_end to TAI:
tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

;Once the time common block is set, the module routines can be called.

;In general, the module's read routine will return data for the time
;set in common mjdt. You'll need to look at that read routine to see
;what parameters it needs to be called with. This can be as little as
;nothing, to something like a Satellite ID, or a more complete
;plotinfo structure( see explanation below...). In the read routine
;you'll also find the common block that the data is returned in, in
;general it looks something like:
common lanl_mpa_data, mpa_header, mpa_data,  mpa_sat

;where  input_header is a structure containing info about the data
;       input_data is an array of structures containing the data
;investigate by using  help,input_header,/st

;This data might not be in the format you want, as it generally will
;need some calibration/conversion etc applied to it. Often this is
;done by a module's plot routine. 
;A lot (but possibly not all) modules now support "getting" data from
;the plot routine, if the keyword OUTPUT is set to 2. basically this
;calls the plot routine, and returns from it just before the actual
;plot is made, at which point all the plot's data is left in a common
;block. 
;Again, you need to look at the plot routine to see what the call is,
;and what the common block is. many plot routines now use the same
;plotinfo structure as the read routine as input. The data is returned
;in a common block that looks something like this:
;common slice_lanl_geo2, time, ymat, utitle, uytitle, ych

COMMON lanl_mpa_slice,  utitle

COMMON lanl_mpa_channels, yarr, zmat
;The same common block is also used by the slice routines...


;In case this isn't confusing enough, one can combine both of the
;above steps. PAPCO internally keeps a list of all the plotinfo
;structures for each plot it needs to make, and calls one interface
;routine when it needs to make a plot. That routine first calls the
;read routine, and then the plot routine. You can call it yourself,
;with the keyword OUTPUT set to 2. This routine is always called
;draw_module_name, and always has the same set of input parameters and
;keywords. It resides in the modules
;../papco_interface/papcoadd_module_name.pro file:
;
;pro draw_lanl_geo, panel, PlotInfo, orbitNo, $
;                   get_data_call, slice_type, IsRead, $
;                   OUTPUT=OUTPUT

;when using this routine "manually" many of the above parameters are
;not important. panel is a vector specifying the panel position of the
;plot, set it to a default (panel=[0,1,1]). PlotInfo is the
;panel's structure containing all it's plot information, that needs to
;be set (see below). orbitNo is used if the module you're using allows
;time specification by orbit number, generally that is not the case,
;just set it to zero. get_data_call, slice_type, are return
;values which are not needed further but need to be in the call. 
;IsRead needs to be set to zero, meaning the data is not read. if
;successful, it is returned set to 1, and if you call the routine
;again with isread set to 1 it won't read the data again as it is
;already in the read common block. Useful if you're re-reading the
;same data often but getting different plot data made... use with
;care, or always set to zero to force a read.
;OUTPUT is the keyword than needs to be set to 2!


;now, the explanation of the PlotInfo structure. This is a general
;structure that represents all the data needed to specify the plot
;the module needs to make. This structure contains tags used by PAPCO,
;and tags used by the user who wrote the module. It is this structure
;that is modified by the module's panel editor (the widget you get with
;all the choices for a given module). So by setting this structure
;correctly, you can control exactly what the data is that is going to
;be plotted, in exactly the same manner as you would by clicking the
;options in the widget. The only problem of course is to know how each
;option is represented in the plotinfo structure, and this is where
;things may get difficult: Because PAPCO is pretty general, THE NAME
;OF THE TAGS USED WILL BE THE SAME, BUT THE WAY THEY ARE SET WILL MEAN
;DIFFERENT THINGS IN DIFFERENT MODULES.
;So where do you get this information? PAPCO convention requires this
;info to be in the module's help file, exactly for this
;purpose. However, not all module authors write such a help file or
;keep it up-to-date... then your only backup is to actually look at the
;module's panel_editor routine (the one that makes the widget) and to
;work out manually what options are set in what way in the plotinfo
;structure. This can be painful...

;The module's help file in in its papco_interface directory:
;../module_name/papco_interface/module_name_panel.help
;this help file is displayed when you click "help" on the module's
;panel editor. 
;In this help file the PlotInfo structure information is in the
;section "DESCRIPTION OF USE of PLOTINFO STRUCTURE"


;So in general, you need to "make" a plotinfo structure, set it
;according to what data you want, and then call the draw_module_name
;routine.

;To make a plotinfo structure there's a papco routine you can use:

plotinfo = papco_getplotinfostruct()

;So for the lanl_geo module you might do something like this:

;plotinfo.ioptions(0) =  0    ;10? sec time res
;plotinfo.ioptions(2) =  1    ;get flux
;plotinfo.ioptions(3) =  0    ;SOPA/CPA - the only available choice at this time (20030320)
;plotinfo.ioptions(10) =  13  ;LANL-97A this is changed if keyword SAT_ID is set

;setting which channels you want to read is a little more tricky. Some
;modules have LOTS of channels, and we would run out of variables to
;use in plotinfo to represent them. So often a trick is used.
;the "typevector" tag is an array of 4 64 bit integers which can be
;used as a bit-wise switch, using each bit position to either select
;or not select a channel. To find out what channels are available just
;look at the panel editor widget, they are listed there. 
;lanl_geo uses a max of 120 channels, split 60 /60 between two
;variables. So, to select the first 10 channels (the SOPA electron
;channels):
;set=bytarr(120)
;set(0:9) = 1
;plotinfo.typeVector(0) = 0 & plotinfo.typeVector(1) = 0
;FOR i=0,59 DO IF set(i) THEN $
;  plotinfo.typevector(0)=plotinfo.typevector(0)+2ll^i 
;FOR i=60,119 DO IF set(i) THEN $
;  plotinfo.typevector(1)=plotinfo.typevector(1)+2ll^(i-60)

plotinfo.typevector = [0, 0, 0, 0]
IF keyword_set(EL_ION) THEN plotinfo.typevector(1) = el_ion
IF keyword_set(FLUX) THEN plotinfo.typevector(2) = flux ;see info in papcoadd_lanl_mpa.pro

;for lanl_geo the satellite is chosen by index into an array of
;satellite names. This list is also kept in a common block ,set up by 
;lanl_geo_set_commons.pro. Many modules use such common blocks to keep
;info that is needed in several places. This routine is automatically
;compiled and executed at when you compile papco, so the common blocks
;in them are set:

COMMON lanl_mpa
COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure

;so sat_names(13) in this case will be LANL-01A, and the index is set
;by ioptions(10).

IF keyword_set(SAT_ID) THEN plotinfo.typevector(0) = where(strupcase(sat_id) EQ $
                                                 strupcase(sat_names))

;Finally you're set to read / make the data!

panel = [0, 1, 1]
orbitNo = 0
;plotinfo we set above
IsRead = 0
output = 2

extra_plotinfo = {OVERPLOT:0, PLOTS_ATTHISPOSITION:0, SUBTABLE:1}


draw_lanl_mpa_spec, panel, PlotInfo, orbitNo, get_data_call, slice_type, IsRead, $
                    OUTPUT = output

;you can check if the above call was successful. All module read
;routines set an error common block that can be interrogated:

common get_error, get_err_no, get_err_msg

;get_err_no is zero if successful, non-zero if not, then get_err_msg
;contains the error message.

IF get_err_no NE 0 THEN BEGIN
    message, get_err_msg, /cont
    return
ENDIF

;if you get this far then the data was read and resides in the common
;block from the plot routine. You can interrogate this by:

;help, time, ymat, utitle, uytitle, ych
;help, utitle
;print, utitle

;time contains the time array, "standard" is to have time in seconds, starting
;from zero ALWAYS. You need to add in the start time to get the actual
;time. Some modules return a one-dimension array, others a 2D array
;containing start and end times.

;time = time+tai1
;print, ''
;print, 'Startime: '
;print,  tai2utc(time(0), /ecs)  ;prints the start time.

;ymat can be a linear array or a 2D array depending on what data you
;selected. generally the first dimension is time, the second data
;channel number.
 
;utitle, uytitle are strings that papco would have used to label the
;panel, utitle describes the data in short, uytitle is the y-axis
;label, Some other modules might also include the z\uztitle, label for
;the color bar. ych here is info for the energies of the channels.


;Now you're done, have the data you need in common blocks, and you can do
;what you want with it...


END 
