;******************************************************************************
;*  NAME:        a_lanl_mpa_pitch
;*
;*  DESCRIPTION: routine for access to lanl_geo module's data
;*               For use inside the PAPCO environment 
;*               (papco is compiled, required modules are loaded)
;*               
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               T1       -  start date/time in any PAPCO format, string
;*               T2       -  end date/time in any PAPCO format, string
;*               MJDT     -  T1 and T2 are given in mjdt format
;*               CONTROL  - structure giving all necessary information
;                           for read routine
;*    FROM CONTROL STRUCURE:
;*               SAT_ID   - index in sat_names
;*               SPECIES  - reads electron or ion data (0,1)
;*               PL_TYPE  - 
;*               PA_RES   - pitch angle resolution (deg)
;*               PA_BIN1  - 
;*               PA_BIN2  -
;*               EK1
;*               EK2
;*               
;*         
;*  Enter a time in one of these formats: date time, date
;*  where time is in one of those formats: h:m:s, h:m
;*  and date is in one of those formats: m/d/y, doy/y, d.m.y, m/d/y, yyyymmdd
;*  where d is day and doy is day of year, h:m:s is the time
;*  (year > 1990 if you use T90 format)
;*  e.g: 1/90 14:11, 1/1990 1:1:1, 350/1991 01:00:03
;*       15.1.90 14:11, 1/15/1990 15:3:1                  
;*
;*  DATE:        March 2003
;*
;*  AUTHOR:       R. Friedel
;*
;*  CHANGES:   changed to work for mpa pitch angle instead of sopa  (Arne Aasnes,
;*             LANL, July, 2003) 
;*
;* Example call line: a_lanl_mpa_pitch,SAT_ID='LANL-97A',T1='19980724',T2='19980725'
;******************************************************************************
PRO a_lanl_mpa_pitch, T1 = T1, T2 = T2, MJDT = mjdt,CONTROL = control, SAT_ID = sat_id, $
                      SPECIES = species, PL_TYPE = pl_type, PA_RES = pa_res, $
                      PA_BIN1 = ba_bin1, PA_BIN2 = pa_bin2, EK1 = ek1, EK2 = ek2

;first, we need to call papco once so that it initializes itself. This
;sets up a bunch of common blocks, structures and environmental
;variables that are needed by various modules.

papco, /set_only ;only initializes papco then exists.

IF NOT keyword_set(T1) THEN BEGIN
    message, 'No start time (T1) defined', /cont & return
ENDIF 

IF NOT keyword_set(T2) THEN BEGIN
    message, 'No end time (T2) defined', /cont & return
ENDIF 


;PAPCO sets the start and end time in a common block, which controls
;the time range for all read routines and all plot routines in all
;papco modules that are loaded.
COMMON mjdt, mjdt_start, mjdt_end

;The time format in mjdt_start, mjdt_end is modified Julian date and
;seconds of day, in a structure:
; mjdt_start = {MJD:0l, T:0.0}

;PAPCO provides a set of time conversion routines that can convert
;between string times to papco time and a variety of other formats:
;look in ../papco/papco_lib/convert_time.pro

;for the example here, if the input string is in any of the PAPCO time
;formats, use the function convert_timeToSeconds to convert to
;mjdt_start, mjdt_end. This function returns a structure containing
;the tags err, value and errtext. If err is zero, the time string was
;recognized, and the converted time is in value. If err eq -1 there
;was an error, the error message is in errtext. This errtext also
;contains a list of the "allowed" formats...

IF keyword_set(MJDT) THEN BEGIN 
   mjdt_start = T1
   mjd_end = T2
ENDIF ELSE BEGIN 
   result = convert_timeToSeconds(T1, /T90, /MJDT)
   IF result.err EQ 0 THEN mjdt_start =result.value ELSE BEGIN
      message, 'Error in start time (T1)', /cont
      print,  result.errtext
      return
   ENDELSE

   result = convert_timeToSeconds(T2, /T90, /MJDT)
   IF result.err EQ 0 THEN mjdt_end =result.value ELSE BEGIN
      message, 'Error in end time (T2)', /cont
      print,  result.errtext
      return
   ENDELSE 
ENDELSE  
;if you don't like the papco time formats you can also use the time
;routines provided under
;../papco/papco_lib/cdssw/time/
;these will come in handy in any case, since a lot of the papco
;modules use the TAI format internally - international atomic time.
;There's an informative readme file in that directory too.
;Here's a quick way of converting mjdt_start, mjdt_end to TAI:
tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

;Once the time common block is set, the module routines can be called.

;In general, the module's read routine will return data for the time
;set in common mjdt. You'll need to look at that read routine to see
;what parameters it needs to be called with. This can be as little as
;nothing, to something like a Satellite ID, or a more complete
;plotinfo structure( see explanation below...). In the read routine
;you'll also find the common block that the data is returned in, in
;general it looks something like:
common lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data,  mpa_pitch_sat

;where  mpa_pitch_header is a structure containing info about the data
;       mpa_pitch_data is an array of structures containing the data
;investigate by using  help,mpa_pitch_header,/st

;This data might not be in the format you want, as it generally will
;need some calibration/conversion etc applied to it. Often this is
;done by a module's plot routine. 
;A lot (but possibly not all) modules now support "getting" data from
;the plot routine, if the keyword OUTPUT is set to 2. basically this
;calls the plot routine, and returns from it just before the actual
;plot is made, at which point all the plot's data is left in a common
;block. 
;Again, you need to look at the plot routine to see what the call is,
;and what the common block is. many plot routines now use the same
;plotinfo structure as the read routine as input. The data is returned
;in a common block that looks something like this:
;common slice_lanl_geo2, time, ymat, utitle, uytitle, ych

COMMON lanl_mpa_pitch_slice 

COMMON lanl_mpa_pitch_output, xarr, yarr, zmat 
;This contains the processed data 
;The same common block is also used by the slice routines...

COMMON yscale,  yscl
yscl = make_array(1, 4, value = 0)


;In case this isn't confusing enough, one can combine both of the
;above steps. PAPCO internally keeps a list of all the plotinfo
;structures for each plot it needs to make, and calls one interface
;routine when it needs to make a plot. That routine first calls the
;read routine, and then the plot routine. You can call it yourself,
;with the keyword OUTPUT set to 2. This routine is always called
;draw_module_name, and always has the same set of input parameters and
;keywords. It resides in the modules
;../papco_interface/papcoadd_module_commname.pro file:
;
;pro draw_lanl_geo, panel, PlotInfo, orbitNo, $
;                   get_data_call, slice_type, IsRead, $
;                   OUTPUT=OUTPUT

;when using this routine "manually" many of the above parameters are
;not important. panel is a vector specifying the panel position of the
;plot, set it to a default (panel=[0,1,1]). PlotInfo is the
;panel's structure containing all it's plot information, that needs to
;be set (see below). orbitNo is used if the module you're using allows
;time specification by orbit number, generally that is not the case,
;just set it to zero. get_data_call, slice_type, are return
;values which are not needed further but need to be in the call. 
;IsRead needs to be set to zero, meaning the data is not read. if
;successful, it is returned set to 1, and if you call the routine
;again with isread set to 1 it won't read the data again as it is
;already in the read common block. Useful if you're re-reading the
;same data often but getting different plot data made... use with
;care, or always set to zero to force a read.
;OUTPUT is the keyword than needs to be set to 2!


;now, the explanation of the PlotInfo structure. This is a general
;structure that represents all the data needed to specify the plot
;the module needs to make. This structure contains tags used by PAPCO,
;and tags used by the user who wrote the module. It is this structure
;that is modified by the module's panel editor (the widget you get with
;all the choices for a given module). So by setting this structure
;correctly, you can control exactly what the data is that is going to
;be plotted, in exactly the same manner as you would by clicking the
;options in the widget. The only problem of course is to know how each
;option is represented in the plotinfo structure,

;The module's help file in in its papco_interface directory:
;../module_name/papco_interface/module_name_panel.help
;this help file is displayed when you click "help" on the module's
;panel editor. 


;for lanl_geo the satellite is chosen by index into an array of
;satellite names. This list is also kept in a common block ,set up by 
;lanl_geo_commons.pro. Many modules use such common blocks to keep
;info that is needed in several places. This routine is automatically
;compiled and executed at when you compile papco, so the common blocks
;in them are set:

COMMON lanl_mpa
COMMON extra, extra_PlotInfo  

; passes in papco-set keywords structure;So in general, you need to "make" a plotinfo structure, set it
;according to what data you want, and then call the draw_module_name
;routine.

;To make a plotinfo structure there's a papco routine you can use:

new_plotinfo = papco_getplotinfostruct()

IF NOT keyword_set(CONTROL) THEN begin

new_control = lanl_mpa_control
IF keyword_set(SAT_ID) THEN new_control.sc_id = where(strupcase(sat_names) EQ strupcase(sat_id)) $
ELSE new_control.sc_id = 1
IF keyword_set(EL_ION) THEN new_control.species = el_ion ; default is electrons
IF keyword_set(PA_RES) THEN new_control.pa_res = pa_res ELSE new_control.pa_res = '5'
IF keyword_set(PA_BIN1) THEN new_control.pa_bin1 = pa_bin1
IF keyword_set(PA_BIN2) THEN new_control.pa_bin2 = pa_bin2
IF keyword_set(EK1) THEN new_control.ek1 = ek1
IF keyword_set(EK2) THEN new_control.ek2 = ek2
IF keyword_set(PL_TYPE) THEN new_control.pl_type = pl_type ELSE new_control.pl_type = 1
IF keyword_set(SAT_ID) THEN new_control.sc_id = where(strupcase(sat_names) EQ strupcase(sat_id))

new_plotinfo.USR_PTR1 = ptr_new(new_control)
ENDIF ELSE new_plotinfo.USR_PTR1 = ptr_new(control)
;Finally you're set to read / make the data!

panel = [0, 1, 1]
orbitNo = 0
;plotinfo we set above
IsRead = 0
output = 2

extra_plotinfo = {OVERPLOT:0, PLOTS_ATTHISPOSITION:0, SUBTABLE:1}

draw_lanl_mpa_pitch, panel, new_PlotInfo, orbitNo, get_data_call, slice_type, IsRead, $
                    OUTPUT = output

;you can check if the above call was successful. All module read
;routines set an error common block that can be interrogated:

common get_error, get_err_no, get_err_msg

;get_err_no is zero if successful, non-zero if not, then get_err_msg
;contains the error message.

IF get_err_no NE 0 THEN BEGIN
    message, get_err_msg, /cont
    return
ENDIF

;if you get this far then the data was read and resides in the common
;block from the plot routine. You can interrogate this by:

;help, time, ymat, utitle, uytitle, ych
;help, utitle

;time contains the time array, "standard" is to have time in seconds, starting
;from zero ALWAYS. You need to add in the start time to get the actual
;time. Some modules return a one-dimension array, others a 2D array
;containing start and end times.

;time = time+tai1
;print, ''
;print, 'Startime: '
;print,  tai2utc(time(0), /ecs)  ;prints the start time.

;ymat can be a linear array or a 2D array depending on what data you
;selected. generally the first dimension is time, the second data
;channel number.

;utitle, uytitle are strings that papco would have used to label the
;panel, utitle describes the data in short, uytitle is the y-axis
;label, Some other modules might also include the z\uztitle, label for
;the color bar. ych here is info for the energies of the channels.


;Now you're done, have the data you need in common blocks, and you can do
;what you want with it...


END 
