PRO plot_legend

  COMMON papco_color_names

  pol_det = 6

  yy = [0.05, 0.05]
  xx = [0.1, 0.15]
  xstep = 0.05
  ystep = 0.02
  FOR i = 1, pol_det DO BEGIN 
     plots,xx +i*xstep, yy,color = red+i-1, /normal
     xyouts, xx(0)+i*xstep+xstep/2, yy+ystep, strcompress(string(i), /remove_all), /normal,color = black
  ENDFOR 
  
END 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; Plot the monthly efficiency ratio values, make a weighted linear fit and 
;; calculate a new set of efficiency values
;;
;; Procedure goes as follows: Run all_years for a satellite, get
;; monthly relative efficiency factors, by reading data for each day
;; in diagnostics. Then run f_factors to get out the results.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO f_factors, sat_id, WRITE_PNG = write_file,WRITE_PS = write_ps, EN_CHAN = en_chan, PLOT_COUNTS = plot_counts, $
               plot_sigma = plot_sigma, count_lim = count_lim, WRITE_FACtORS = write_factors, save_factors = save_factors, $
               plot_eff = plot_eff, ions = ions

  papco, /set_only
  
  COMMON papco_color_names, black, red, green, yellow, blue, magenta, cyan, $
                            white, burgundy, olive, dark_green, teal, $
                            royal_blue, violet, dark_grey, grey, $
                            foreground, background, color_bar_names, $
                            fixed_color_index, fixed_color_names
  IF NOT keyword_set(ions) THEN species = '0' ELSE species = '1'
  IF NOT keyword_set(en_chan) THEN en_chan = 26
  en_chan = strcompress(string(en_chan), /remove_all)
  path = '~/lanl_geo/mpa/calibration/'
  filename = path+sat_id+'en_chan_'+en_chan+'_spec'+species+'_cal_facts.idl';'_cal_facts_new.idl'
  print,'restoring '+filename
  restore, filename, /verbose
  func = 'str_line'
  lim = 0.7
  p_lim = 0.02

  eff_factors = fltarr(6)

  set_plot, 'x'
  window
  cleanplot
  !p.background = white
  !p.color = black
  nn = 0
  plotting:
  

  erase
  IF !D.name EQ 'PS' THEN factor = 2.5 ELSE factor = 1.

  IF keyword_set(count_lim) THEN BEGIN 
     valid = where(counts GT count_lim)
     time = time(valid)
     counts = counts(valid)
     b_array = b_array(valid, *)
     chisq_arr = chisq_arr(valid, *)
     ks_c99_arr = ks_c99_arr(valid, *)
  ENDIF 

   
  pyear = long(time)/100 
  pmonth = time MOD 100 
  ptime = pyear+pmonth/12.

  IF keyword_set(PLOT_COUNTS) THEN plot,ptime, counts, psym = 1,thick = 1.5*factor,charthick = 1*factor, $
    ytitle = 'Data points', xtitle = 'Year' 
  IF NOT keyword_set(PLOT_EFF) AND NOT keyword_set(plot_COUNTS)  THEN begin
     pi = replicate({fixed:0, limited:[0, 0], limits:[0.D, 0.D]}, 2)
     pi(1).fixed =1 
     !y.margin = [4, 6]
     FOR i = 1, 5 DO BEGIN 
        IF i EQ 3 THEN !y.margin = [6, 2]
        !p.multi = [6-i, 3, 2]
        IF keyword_set(count_lim) THEN  no_err = where(b_array(*, i) GT 1-lim $
                                                       and b_array(*, i) LT 1+lim AND counts GT count_lim, err_ct) ELSE $
          no_err = where(b_array(*, i) GT 1-lim and b_array(*, i) LT 1+lim, err_ct)
        IF err_ct GT 0 THEN BEGIN 
           det_eff = b_array(no_err, i)
           det_sig = ks_c99_arr(no_err, i)
           er_counts = counts(no_err)
           er_time = time(no_err)
           time_pos = indgen(err_ct)
           er_month = er_time MOD 100
           er_year = long(er_time)/100
           dec_time = er_year+er_month/12.0
           ticks = er_year(err_ct-1)-er_year(0)
           IF ticks GT 5 THEN ticks = ticks/2
           IF max(er_year) GT 2003 OR min(er_year) LT 1989 THEN stop
;        IF i EQ 2 THEN stop
        endIF ELSE stop
        start = [double(det_eff(0)), 0d] 
        rerr = det_eff*max(er_counts)/er_counts
        result = mpfitfun(func, time_pos, det_eff, rerr, start,PARINFO = pi, /quiet)
        print, 'Detector'+string(i+1)+':'+string(result(0))+string(result(1))
        plot, dec_time, det_eff,ystyle = 1, $
              xrange = [er_year(0), er_year(err_ct-1)+1], xstyle = 1,thick = 1.*factor,charthick = 1*factor,  $
              ytitle = 'Relative Efficiency', xtitle = 'Time', title = 'Detector '+strcompress(string(i+1),/remove_all), $
              xticks = ticks    ;, yrange = [result(0)-p_lim, result(0)+p_lim]
        IF keyword_set(plot_sigma) THEN BEGIN 
           errplot, dec_time, det_eff-det_sig, det_eff+det_sig
           
        ENDIF 
        oplot, dec_time, det_eff,thick = 1.5*factor, psym = 1, symsize = 0.3
;     errplot, dec_time, det_eff-det_eff*10/alog10(counts)/counts, det_eff+det_eff*10/alog10(counts)/counts, $
;              width = 1/24.
;     oplot, [er_time(0), er_time(err_ct-1)], [result(0)+er_time(0)*result(1),result(0)+er_time(err_ct-1)*result(1)]
        oplot, [er_year(0), er_year(err_ct-1)+1], [result(0), result(0)]
;        xyouts, er_year(0)+0.1*(er_year(err_ct-1)-er_year(0)+1), result(0)+0.85*p_lim, $
;                'Linear fit: '+ $
;                strmid(strcompress(string(result(0)), /remove_all), 0, 5), /data, charthick = 1*factor
        eff_factors(i) = result(0)
     ENDFOR
     FOR i = 0, 5 DO BEGIN 
        xyouts,0.07, 0.85-0.025*i, 'Detector '+strcompress(string(i+1), /remove_all)+': '+ $
               strmid(strcompress(string(new_factors(i)), /remove_all), 0, 5), /normal
        
     ENDFOR 
     eff_factors(0) = 1
     IF Sat_id EQ 'l4' THEN eff_factors(3) = 0
     IF sat_id EQ 'l7' THEN eff_factors(5) = 0
     IF sat_id EQ 'a1' THEN eff_factors(1) = 0
     new_factors = 0.95*eff_factors/max(eff_factors)  
     xyouts, 0.07, 0.88, 'Calculated efficiency factors:', /normal
     print, new_factors
     plot_title = 'Sat id: '+sat_id+'  Relative efficiency of detectors (det. 1 as reference) Energy ch.' $
                  +strcompress(string(en_chan), /remove_all)
     xyouts, 0.1, 0.97,plot_title, /normal, size = 1.1,charthick = 1.0*factor
  ENDIF 
  
  spawn, 'echo $user', idnt
  xyouts, 0.01, 0.002, 'Generated by:  '+idnt+', '+systime(0), /normal,charthick = 1*factor, size = 0.9    
  IF nn LE 1 THEN BEGIN 
     IF keyword_set(WRITE_FACTORS)   THEN BEGIN 
        IF WRITE_FACTORS EQ  1 THEN write_factors = '~/lanl_geo/mpa/calibration/' +sat_id+'_monthly_factors.txt'
        print, 'writing factors to file...'
        openw, 1, write_factors
        printf, 1, '', 'Efficiency factors'
        printf, 1, 'yyyymm  ', 'det 1   ', 'det 2   ', 'det 3   ', 'det 4   ', 'det 5   ', 'det 6   '
     ENDIF  
     ndat = n_elements(time)
  
     IF Sat_id EQ 'l4' THEN b_array(*, 3) = 0
     IF sat_id EQ 'l7' THEN b_array(*, 5) = 0
     IF sat_id EQ 'a1' THEN b_array(*, 1) = 0

     
     IF keyword_set(SAVE_FACTORS) THEN BEGIN 
        filename = '$PAPCO_BASE/papco_modules/lanl_mpa/factors/'+sat_id+'_'+species+'_monthly_factors.idl'
        save, eff_factors, filename = filename,  verbose = verbose
     ENDIF 
     fac_dummy = {str, yyyymm:0d, mn_eff_factors:fltarr(6)}
     eff_factors = replicate(fac_dummy, ndat)
     FOR i = 0, ndat-1 DO BEGIN 
        print, 'year-month:'+strtrim(string(time(i)), 2)
        monthly_factor =0.95*b_array(i, *)/max(b_array(i, *))
        eff_factors(i).yyyymm = time(i)
        eff_factors(i).mn_eff_factors = monthly_factor
        print, monthly_factor
        IF keyword_set(WRITE_FACTORS) THEN printf, 1, strmid(string(time(i)), 6, 6), $
          strmid(monthly_factor(0), 3, 8), strmid(monthly_factor(1), 3, 8), $
          strmid(monthly_factor(2), 3, 8), strmid(monthly_factor(3), 3, 8), strmid(monthly_factor(4), 3, 8), $
          strmid(monthly_factor(5), 3, 8)
        
     ENDFOR  
     IF keyword_set(WRITE_FACTORS) THEN close, 1
     IF keyword_set(WRITE_PNG)  THEN BEGIN 
        plotname = path+sat_id +'_int_cal'+'.png'
        print, 'writing image to file:'+plotname
        image = tvrd(true = 1)
        write_png, plotname, image
     ENDIF 
  ENDIF 
  IF keyword_set(PLOT_EFF) THEN BEGIN 
     !y.margin = [4, 6]
     plot_title = 'Sat id: '+sat_id+'  Efficiency of detectors.  Energy ch.' $
                  +strcompress(string(en_chan), /remove_all)
     xyouts, 0.1, 0.97,plot_title, /normal, size = 1.1,charthick = 1.0*factor 
     FOR i = 0, 5 DO   BEGIN   
        IF keyword_set(count_lim) THEN BEGIN 
           good = where(counts GT count_lim, good_cnts)
           IF good_cnts EQ 0 THEN CONTINUE
           ptime = ptime(good)
        ENDIF   ELSE good = findgen(ndat)
        
        IF i EQ 3 THEN !y.margin = [6, 2]
        !p.multi = [6-i, 3, 2]
        plot, ptime, eff_factors(good).mn_eff_factors(i), xstyle = 1, $
              yrange = [0.9*min(eff_factors(good).mn_eff_factors(i)), 1.1*max(eff_factors(good).mn_eff_factors(i))], $
              title = 'Detector '+strtrim(string(i+1), 2)   
     ENDFOR   
  ENDIF 

  nn = nn+1
  plotname = path+sat_id +'_int_cal'
  IF keyword_set(WRITE_PS) AND nn LE 1  THEN BEGIN 
     print, 'Plot is sent to PS file'
      print, plotname+'.ps'
      set_plot,'ps'
      device,/portrait,bits=8,$
      filename=plotname+'.ps', $
      xsize=18,ysize=15,xoffset=1,yoffset=4
      !p.background = 255
      !p.color = 0
      goto, plotting
   ENDIF 

  IF keyword_set(WRITE_PS) THEN DEVICE, /CLOSE_FILE
END 

;**************************************************************************
;* DESCRIPTION
;*     Wrapper routine for diagnostics. It runs through all years from
;*     a given time (must be changed in file)
;* INPUT 
;*     none (start time must be changed manually in file)
;* KEYWORD
;*     resume: reads the saved files and resumes the process from
;*             where it was interrupted 
;*     _extra: can contain order,channel,pa_low,pa_high (see diagnostics)
;* OUTPUT 
;*     saves the relative (to detector 1) efficiency factor (b_array), data
;*     points (counts) and the date (yyyymm)  
;****************************************************************************

PRO all_years, resume = resume, SAT_ID = sat_id, YEAR = year, month = month,CHANNEL = channel,SPECIES = species, _EXTRA = e
; .r /u/friedel/idl/papco/papco_modules/cluster_rapid/plot_read/gen_fit_release.pro
  COMMON output,all_b_params,  data_points, chisq_tarr, ks_c99_tarr
  
  black = 238
  path = '$PAPCO_DLNK/lanl/geo/mpa/cdf/'
;  path='/n/projects1/space/disks/data6/local/ftp/newmpa/'
  fl_name1 = '_h0_mpa_'
  fl_name2 = '01_v02.cdf'


  IF not keyword_set(channel) THEN channel = 26
  IF NOT keyword_set(SAT_ID) THEN sat_id = 'l0'
  IF NOT keyword_set(YEAR) THEN year = 1999e ELSE year = float(year)
  IF NOT keyword_set(MONTH) THEN month = 3e ELSE month = float(month)
  IF NOT keyword_set(species) THEN species = 0
  filename='~/lanl_geo/mpa/calibration/'+sat_id+'en_chan_'+strcompress(string(channel) $
             , /remove_all)+'_spec'+strcompress(string(species), /remove_all)+'_cal_facts.idl';'_cal_facts_new.idl'
;  filename = '~/lanl_geo/mpa/calibration/'+sat_id+'en_chan_'+strcompress(string(channel) $
;             , /remove_all)+'_spec'+strcompress(string(species), /remove_all)+'_cal_facts.idl'
  IF keyword_set(resume) THEN BEGIN 

     restore, filename, /verbose
     num = n_elements(time)
     year = floor(time(num-1)/100d)
     month = time(num-1)-100d*year+1
     IF month EQ 13 THEN BEGIN 
        year = year+1
        month = 1
     endIF 
  ENDIF ELSE BEGIN 
     isfile = findfile(filename,count = cnt)
     answ = ''
     IF cnt GT 0 THEN BEGIN 
        read, answ, prompt = $
 'Save file exists, do you really want to delete it and start over again? (y/n): '
        IF answ EQ  'y' THEN BEGIN 
           file_delete, filename
        ENDIF ELSE BEGIN 
           restore, filename, /verbose
           num = n_elements(time)
           year = floor(time(num-1)/100d)
           month = time(num-1)-100d*year+1
           IF month EQ 13 THEN BEGIN 
              year = year+1
              month = 1
           endIF 
        ENDELSE 
     endIF 
  ENDELSE 
  FOR year = year, 2003. DO BEGIN  
     IF year LT 1993. THEN year = 1993. ; no data before 1993
;     IF year EQ 1996. THEN year =  1997. ; no data for 1996 ; yes
;     there is
     yrstr = strmid(strcompress(string(year), /remove_all), 0, 4)
     
     FOR month = month, 12. DO BEGIN 
        mnstr = strmid(strcompress(string(month), /remove_all), 0, 2)
        IF month LT 10 THEN mnstr = '0'+strmid(mnstr, 0, 1)

        file_name = sat_id+fl_name1+yrstr+mnstr+fl_name2

;        full_path = path+yrstr+'/'+file_name
        full_path = path+yrstr+'/'+file_name
;        IF year LT 1997. THEN full_path = path+yrstr+'/'+yrstr+mnstr+'/'+file_name
;        IF year GE 1997. AND year LE 2000. THEN full_path = path+yrstr+'/'+file_name
;        IF year GT 2000. THEN full_path = path+yrstr+mnstr+'/'+file_name

        diagnostics, full_path, /compare_det,species = species,channel=channel, _EXTRA = e
        
        IF all_b_params(0) NE 0 THEN begin
           IF n_elements(B_array) eq 0 THEN BEGIN 
              B_array = transpose(all_b_params)
              counts = data_points
              time = 100.*year +month
              chisq_arr = transpose(chisq_tarr)
              ks_c99_arr = transpose(ks_c99_tarr)
           ENDIF  ELSE BEGIN 
              IF n_elements(b_array) GT 1 THEN BEGIN 
                 B_array = [B_array, transpose(all_b_params)]
                 counts = [counts, data_points]
                 time = [time, 100.*year +month]
                 chisq_arr = [chisq_arr, transpose(chisq_tarr)]
                 ks_c99_arr = [ks_c99_arr, transpose(ks_c99_tarr)]
              ENDIF 
           ENDELSE  
        ENDIF 
        save, B_array,  counts,  time,chisq_arr,ks_c99_arr,  filename = filename, verbose = verbose
;        b_array = 0 &  counts = 0 &  time = 0 &  chisq_arr = 0
        all_b_params = 0 &  data_points = 0 & chisq_tarr = 0 & ks_c99_tarr = 0
        
     ENDFOR
     month = 1. 
  ENDFOR 
  erase

  FOR i = 0, 5 DO BEGIN 
     !P.multi = [6-i, 3, 2]
     plot, B_array(*, i), title = sat_id+' '+'detector '+strcompress(string(i+1), /remove_all)
  ENDFOR
  spawn, 'echo $user', idnt
  xyouts, 0.01, 0.02, 'Generated by:  '+idnt+', '+systime(0), /normal, color = black


END 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;*  diagnostics,infile
;* 
;*  DESCRIPTION
;*     a set of routines to analyse the mpa data and intercalibrate detectors
;* INPUT 
;*     infile: name of file to read 
;*     
;* KEYWORD
;*     COMPARE_DET: starts a routine to make scatter plots of the
;*                  counts from the detectors versus one reference
;*                  detector. A line is fitted through the data to
;*                  find the relative efficiency of the detectors.
;*     FACTORS: 
;*     COVERAGE:    investigates what region of pa space the different
;*                  detectors cover
;*     FLUX: 
;*     RAW COUNTS:
;*     ORDER:       set this to the detector to be used as reference
;*                  detector
;*     CHANNEL:     Energy channel to be used (0-39)
;*     PA_HIGH:     upper pitch angle to be used (only if COMPARE_DET)
;*     PA_LOW:      lower pitch angle to be used
;*     ALL_DAY:     bins all data for one day into one data point
;*     ONE_DAY:     run calibration only for one day
;*     SPECIES:     0 for electrons, 1 for ions
;*     VERBOSE:     allows verbose output
;*     resume: reads the saved files and resumes the process from
;*             where it was interrupted 
;* OUTPUT 
;*     saves the relative (to detector 1) efficiency factor (b_array), data
;*     points (counts) and the date (yyyymm)  
;****************************************************************************
PRO diagnostics,  infile,COMPARE_DET = compare_det, FACTORS = factors, COVERAGE = coverage, $
                  FLUX = flux,  RAW_COUNTS = raw_counts, ORDER = order,  CHANNEL = channel, $
                  PA_HIGH = pa_high, PA_LOW = pa_low, ALL_DAY = all_day,  ONE_DAY = one_day, $
                  SPECIES = species, VERBOSE = verbose
  
;  .r ~/papco_user/papco_user_modules/lanl_mpa/plot_read/diagnostics
  
  papco, /set_only
  COMMON lanl_mpa
  COMMON output, all_b_params,  data_points,  chisq_tarr, ks_c99_tarr
  COMMON get_error, get_err_no, get_err_msg
  COMMON lanl_mpa_pitch_raw
  COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data
  COMMON papco_color_names, black, red, green, yellow, blue, magenta, cyan, $
                            white, burgundy, olive, dark_green, teal, $
                            royal_blue, violet, dark_grey, grey, $
                            foreground, background, color_bar_names, $
                            fixed_color_index, fixed_color_names
  
  get_err_no = 0 & get_err_msg = 'File read successfully'
  etimesg = 0

  IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0
  
  IF n_elements(infile) EQ 0 THEN $
infile ='$PAPCO_DLNK/lanl/geo/mpa_cdf/2002/a1_h0_mpa_20020109_v02.cdf'
;'/a/teapot/tempest/projects/mpa/cd_ready/1995/199504/l1_h0_mpa_19950401_v02.cdf'
; '/n/radbelt/dlnk/lanl/geo/mpa_cdf_2/200208/l7_h0_mpa_20020801_v02.cdf'
;'$PAPCO_DLNK/cdf/lanl_mpa/1998/l1_h0_mpa_19980201_v02.cdf'
; '$PAPCO_DLNK/cdf/lanl_mpa/2000/l4_h0_mpa_20000901_v02.cdf'
;'$PAPCO_DLNK/cdf/lanl_mpa/1998/l4_h0_mpa_19980201_v02.cdf'
  
  months = ['jan', 'feb', 'mar', 'apr', 'may', 'jun', 'jul', 'aug', 'sep', 'oct', 'nov', 'dec']
  sat_code = ['l9', 'l0', 'l1', 'l4', 'l7', 'a1', 'a2']
  
;  datestr = strmid(infile, 15, 8, /reverse_offset)
;  yearstr = strmid(datestr, 0, 4)
;  monthstr = strmid(datestr, 4, 2)

  slash_pos = strpos(infile, '/', /reverse_search)
  sat_name = strmid(infile, slash_pos+1, 2)
  sat_id = sat_names(where(sat_name EQ sat_code))
  
  date = strmid(infile, slash_pos+11, 8)
  yrmn = strmid(date, 0, 6)
  yearstr = strmid(yrmn, 0, 4)
  monthstr = strmid(yrmn, 4, 6)

                                ; the pitch angle limits around 90 degrees we use
  IF NOT keyword_set(PA_LOW) THEN  PA_low = 85
  IF NOT keyword_set(PA_HIGH) THEN PA_high = 95
  IF NOT keyword_set(SPECIES) THEN species = 0
  ;; rename the variables to be compatible with older version


  IF NOT keyword_set(COMPARE_DET)  THEN BEGIN
     flux, infile, /output, VERBOSE = verbose
     
     IF  keyword_set(compare_det) NE 1 AND get_err_no NE 0 THEN GOTO, out
     
     IF n_elements(mpa_pitch_data) NE 0 THEN BEGIN 
        
        dimensions = size(mpa_pitch_data.e_unbinned, /dimensions)
        n_el = dimensions(3)
        en_channels = dimensions(0)
        pol_det = dimensions(1)
        az_det = dimensions(2)

        efa = make_array(n_el, en_channels, pol_det, az_det)
        e_alpha = make_array(n_el, en_channels, pol_det, az_det)
        FOR i = 0, n_el-1 DO BEGIN 
           efa(i, *, *, *) = mpa_pitch_data(i).e_unbinned
           e_alpha(i, *, *, *) = mpa_pitch_data(i).e_mag
        endFOR 
     ENDIF 
  ENDIF 
  
  IF n_elements(pol_det) EQ 0 THEN BEGIN 
     pol_det = 6
     az_det = 24
     en_channels = 40
  ENDIF  

   plot_title = sat_name+' '+date
  
   IF not keyword_set(CHANNEL) then channel = 26  ; choosing one energy channel for now
   IF keyword_set(FLUX) OR keyword_set(COVERAGE) OR keyword_set(RAW_COUNTS) THEN !y.margin = [8, 2]   
   IF keyword_set(RAW_COUNTS) OR keyword_set(FACTORS)   THEN BEGIN 
     ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
     ;; Correcting ecounts by the geometric factor
     ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      gfactor_4dim = rebin(reform(gfactor, 1, 1, pol_det, 1), n_el, en_channels, pol_det, az_det)
      ecounts_g = ecounts/gfactor_4dim
   ENDIF  ELSE gfactor = make_array(6, value = 0)

; the data is listed from high energies to low, so to get the correct
; energy channel we have to turn it around (if electrons)
   channel_nr = channel
   channel = en_channels-1-channel
  
   desc = ''
  
   IF  keyword_set(COMPARE_DET) THEN BEGIN 
     
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Makes a scatter plot of counts in one detector vs the other
;; detectors. We choose a range of pitch angles near 90 degrees where
;; all detectors contribute, and PA asymetry should not influence the
;; result. From the linear fit of the scatter plots we calculate
;; efficiencys and normalise the most efficient channel to a factor
;; 0.95 (to agree with previous work of Michelle Thomsen)
;; 
;; Our selection criterias should be: (august 2003, after discussions
;; with M. Thomsen)
;;         1. Counts rate < 5000 cts/9 ms (before geometric factor)
;;;        2. Tperp/Tpar > 1.05
;;         3. E~1 KeV
;;
;; We want to use only the geometric factor to correct the counts
;; Make plots of conjugate pairs to investigate effect of errors in
;; magnetic field determination        
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
     
      filename = '~/lanl_geo/mpa/calibration/pol_det_compared_'+sat_name+'_en_chan_'+$
                 strcompress(string(channel_nr), /remove_all)+'_'+yrmn+'.idl'
      
      integr = 1 ; number of time steps to integrate over

      all_b_params = fltarr(pol_det)
     
      pol_counts_tot = fltarr( pol_det)
      pol_flux_tot = fltarr( pol_det)
;      az_counts_tot = fltarr( az_det)
;      az_flux_tot = fltarr( az_det)
      
      pol_counts = fltarr( pol_det)
      pol_flux = fltarr( pol_det)

;      az_counts = fltarr( az_det)
;      az_flux = fltarr( az_det)
      
;     slash_pos= strpos(infile, '/', /reverse_search)
      pos = slash_pos+17        ; position of day of month in input file
      
      first_day = strmid(infile, pos, 2)
      first_day = fix(first_day)
      IF first_day EQ 1 AND NOT keyword_set(one_day) THEN file_delete, filename, /ALLOW_NONEXISTENT

      FOR date = first_day, 31 DO BEGIN 
         IF date GE 1 AND NOT keyword_set(one_day) THEN BEGIN
            dom =strcompress(string(date), /remove_all)
            IF date LT 10 THEN dom = '0'+dom
            strput, infile, dom, pos
         ENDIF  
         
         get_err_no = 0
         ecounts = 0
         pcounts = 0
         e_alpha = 0
         p_alpha = 0
         mpa_pitch_data = 0

; read the data file for this day and goto next day if there is no
; data
         flux, infile, /output, VERBOSE = verbose, /mag_corr
         IF get_err_no NE 0 OR n_elements(mpa_pitch_data) LE  1 THEN GOTO, plot_loop
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Saving the constructed data (this is used in the lanl_mpa papco module)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
         
;         save_path = '$PAPCO_DLNK/lanl/geo/savesets/pitch/mpa/'
;         save_file = save_path+sat_id+'_'+dom+'-'+months(fix(monthstr)-1)+'-'+yearstr+'.idl'
;         save,  mpa_pitch_header, mpa_pitch_data, sat_id, filename = save_file(0), /compress

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;    Form the geometric corrected counts and pitch angle matrices
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;         
         n_el = n_elements(mpa_pitch_data) 

; expand the geometric factor to the size of ecounts
         gfactor_4dim = rebin(reform(gfactor, 1, 1, pol_det, 1), n_el, en_channels, pol_det, az_det)
         alpha = make_array(n_el, en_channels, pol_det, az_det)

         ; reforming e_mag into e_alpha 
         IF species EQ 0 THEN BEGIN  ; electrons
            FOR el = 0, n_el-1 DO alpha(el, *, *, *) = mpa_pitch_data(el).e_mag 
            counts = ecounts
            aniso = e_aniso
            max_cnt_lim=2000
            min_cnt_lim=1
         endIF ELSE BEGIN  ; protons
           FOR el = 0, n_el-1 DO alpha(el, *, *, *) = mpa_pitch_data(el).p_mag
           counts = pcounts
           aniso = p_aniso
;           channel=en_channels-channel-1
           max_cnt_lim=100
           min_cnt_lim=0
        ENDELSE 
        counts_g = counts/gfactor_4dim
; resetting the parameter matrix         
         all_b_params(*) = 0
         

         IF NOT keyword_set(ONE_DAY) THEN BEGIN                   
            ON_IOERROR,  go_on
            restore, filename, VERBOSE = verbose
            dealloc_lun,/quiet  ;frees logical units used by save/restore
            message,'saveset read:', /cont
            print, filename
            go_on:
            ON_IOERROR,  plot_loop                   
         ENDIF 
;; running through data for one day        
         FOR h = 0,n_el-1 DO BEGIN 

; resetting the medium level storage matrices
            pol_counts_tot(*) = 0
            pol_flux_tot(*) = 0

            IF keyword_set(ALL_DAY) THEN BEGIN 
               integr = n_el    ; gather all data in a bunch
               psym = 1
            ENDIF ELSE psym = 3

            FOR i =h, h+integr-1 DO BEGIN 
; resetting the basic level storage matrices.               
               pol_counts(*) = 0
               pol_flux(*) =  0

               IF i GE n_el THEN BREAK
               
;               e_alpha_test = reform(e_alpha(i, channel, *, *), pol_det, az_det)  
;               efa_test =reform(efa( i, channel, *, *), pol_det, az_det)
;               ecounts_test = reform(ecounts( i, channel, *, *), pol_det, az_det)
               
               th_alpha = alpha(i, channel, *, *)
               th_counts = counts(i, channel, *, *)
;               th_counts_g = counts_g(i, channel, *, *)
               pitch_ind = where(th_alpha GT PA_low $
                             AND th_alpha LT PA_high $
                             AND th_counts GT min_cnt_lim $
                             AND finite(th_counts) EQ 1 $
                             AND th_counts LT max_cnt_lim, hits)
               
                                ; only use the data if the b field
                                ; determination is good. If the data
                                ; has been interpolated (b_flag eq 5) that indicates
                                ; a singular value in the b field
                                ; direction.
               aniso_lim = 1.05
               IF sat_name EQ 'a1' THEN aniso = e_aniso ;aniso_lim = 1 ELSE aniso_lim = 1.05
               IF hits GT 0 AND aniso(i) GT aniso_lim  AND mpa_pitch_data(i).b_flag NE 5 THEN BEGIN 

                  inc_counts = th_counts(pitch_ind)
                  inc_angles = th_alpha(pitch_ind)
                  
              
                  which_pol_det = pitch_ind MOD pol_det 
;                  which_az_det = fix(pitch_ind/(pol_det))
         
                  pol_counts(*) = 0
                  pol_flux(*) = 0
 ;                 az_counts(*) = 0
 ;                 az_flux(*) = 0
              
;              print, 'polar detectors      :', which_pol_det
;              print, 'included pitch angles:', inc_angles
;              print, 'included counts      :', inc_counts  

                  FOR k = 0, hits-1 DO BEGIN
                     pol_flux( which_pol_det(k)) =  pol_flux( which_pol_det(k))+inc_counts(k)
                     pol_counts( which_pol_det(k)) = pol_counts( which_pol_det(k))+1
                     
;                     az_flux( which_az_det(k)) =  az_flux( which_az_det(k))+inc_counts(k)
;                     az_counts( which_az_det(k)) = az_counts( which_az_det(k))+1
                  ENDFOR

                                ; only include the data if there is
                                ; counts in all detectors!! 
                  ;exceptions because detectors are crazy:
                  IF sat_id(0) EQ 'lanl-01a' THEN pol_counts(1) = 1
                  IF sat_id(0) EQ '1994-084' THEN pol_counts(3) = 1
                  IF sat_id(0) EQ 'lanl-97a' THEN pol_counts(5) = 1
               
                  IF min(pol_counts) NE 0 THEN BEGIN 
                     IF verbose THEN print, 'Valid data for calibration at time '+string(i)
                     pol_flux_tot = pol_flux_tot+pol_flux/pol_counts
                     pol_counts_tot = pol_counts_tot+[1, 1, 1, 1, 1, 1];pol_counts
;                     az_flux_tot = az_flux_tot+az_flux
;                     az_counts_tot = az_counts_tot+az_counts
                  ENDIF ELSE IF verbose THEN print, 'Not valid data in all detectors for calibration at time '+string(i)
               ENDIF   ELSE IF verbose THEN print, 'Not valid data for calibration at time '+string(i)
           
            ENDFOR    
            h = i-1
            IF min(pol_counts_tot) NE  0  THEN BEGIN             
               pol_flux_tot = pol_flux_tot;/pol_counts_tot*min(pol_counts_tot)
              ; IF h EQ integr-1 AND n_elements(pol_flux_row) EQ 0 THEN BEGIN 
               IF  n_elements(pol_flux_row) EQ 0 THEN BEGIN
                  pol_flux_row = transpose(pol_flux_tot)
                  pol_counts_row = transpose(pol_counts_tot)
               ENDIF ELSE BEGIN 
                  pol_flux_row = [pol_flux_row,transpose(pol_flux_tot)]
                  pol_counts_row = [pol_counts_row, transpose(pol_counts_tot)]
               ENDELSE  
            ENDIF   
         ENDFOR          
         
stop

         IF NOT keyword_set(ONE_DAY) THEN  save, pol_flux_row, pol_counts_row, filename = filename ELSE date = 31
         
;; drawing up the scatter plots of the polar detectors vs detector 1
         
         plot_loop: 
         IF n_elements(pol_flux_row) EQ 0 OR date LT 31 THEN CONTINUE ;go to next iteration

         IF sat_id(0) EQ 'lanl-01a' THEN pol_flux_row(*, 1) = 0
         IF sat_id(0) EQ '1994-084' THEN pol_flux_row(*,3) = 0
         IF sat_id(0) EQ 'lanl-97a' THEN pol_flux_row(*, 5) = 0
         
         for pd = 0,5 do pol_flux_row(*,pd)=pol_flux_row(*,pd)/gfactor(pd)
         
         pos_count = 1 ; use this to get the positions of the plots correct
         detector = 0 ; note that detectors have values 0-5 here, detector 0 is the one closest to the spin axis
         IF keyword_set(order) THEN detector = order

         plotinfo = +sat_name+'_'+strmid(infile, slash_pos+11, 6)+'_det' $
                  +strcompress(string(detector+1), /remove_all)+'_PA'+strcompress(string(PA_low), /remove_all) $
                  +'-'+strcompress(string(PA_high), /remove_all)
         
         plotname = '~/lanl_geo/mpa/calibration/'+plotinfo+'_calibration.gif'
         plotinfo = sat_name+' '+strmid(infile, slash_pos+11, 6)+' PA bin:'+strcompress(string(PA_low), /remove_all) $
                  +'-'+strcompress(string(PA_high), /remove_all) 
         cleanplot
         !P.background = white
         !P.color = black
         window, title = infile
         erase
           
         x_max = max(pol_flux_row(*, detector))
         dimen = size(pol_flux_row, /dimensions)
         IF n_elements(dimen) LE 1 THEN dimen = [1, pol_det]
         
         xyouts, 0.017, 0.97, plotinfo, /normal
         XYOUTS, 0.017, 0.940, 'Fit parameters A+Bx', charsize = 0.95, /normal
         

               
         chisq_tarr = findgen(6)
         ks_c99_tarr = findgen(6)
         FOR g = 0, pol_det-1 DO BEGIN 
            IF g EQ detector THEN g = g+1
            IF sat_id(0) EQ 'lanl-01a' AND g EQ 1 THEN CONTINUE 
            IF sat_id(0) EQ '1994-084' AND g EQ 3 THEN CONTINUE 
            IF sat_id(0) EQ 'lanl-97a' AND g EQ 5 THEN CONTINUE 
            !p.multi = [pol_det-pos_count, 2, 3]
            y_max = max(pol_flux_row(*, g))
            sc_max = y_max >  x_max
            plot, [0, sc_max], [0, sc_max],/nodata, $
                  title = 'Detector '+strcompress(string(detector+1),/remove_all)+ $
                  ' vs '+strcompress(string(g+1), /remove_all)
            
            
            plots, pol_flux_row(*, detector),  pol_flux_row(*, g), PSYM = psym
            a = 0
            IF dimen(0) EQ 1 THEN BEGIN 
                b = y_max/x_max
               params = [a, b]
               chisq = !values.f_nan &  ks_c99 = [b, b]
            endIF ELSE BEGIN
               x = pol_flux_row(*, detector) & y = pol_flux_row(*, g)
               sigx = sqrt(x) & sigy = sqrt(y)
;               ind = where(x GT 0 AND y GT 0)
;               x = x(ind) & y = y(ind) & sigx = sigx(ind) & sigy = sigy(ind)
               gen_fit_release, x = x, sigma_x = sigx, y = y, sigma_y = sigy, $
                     nparms = 1,guess_slope = 1, slope = B, chisq = chisq, ks_c99 = ks_c99, /force_kappa_search
               
               params = [A, B]
               
            ENDELSE  
            print, params
            xyouts,0.017, 0.935-0.025*pos_count, 'Det '+strcompress(string(detector+1), /remove_all)+ $
                            ' vs '+strcompress(string(g+1), /remove_all)+ $
                                   '      A:'+strcompress(string(round(A)), /remove_all), /normal, charsize = 0.95
            xyouts, 0.25,  0.935-0.025*pos_count, ' B:'+strmid(strcompress(string(round(B*10000.)/10000.), $
                                            /remove_all), 0, 4), /normal, charsize = 0.95
            xyouts, 0.35,0.935-0.025*pos_count, ' chisq:'+strmid(strcompress(string(chisq), /remove_all), 0, 9), $
                                   /normal, charsize = 0.95
            x_row = [0, x_max]
            y_row = x_row*params(1)+params(0)
            oplot, x_row, y_row
            
            all_b_params(g) = params(1)
            chisq_tarr(g) = chisq &  ks_c99_tarr(g) = mean(ks_c99)
            pos_count = pos_count+1
         ENDFOR     
         all_b_params(detector) = 1
               
;               new_factors = 0.95/max(1/all_b_params)/all_b_params
               
         new_factors = 0.95*all_b_params/max(all_b_params)
         print, 'the efficiency factors are:', new_factors
         xyouts, 0.017, 0.758, 'the efficiency factors are:', /normal
        
        
         new_factors = (round(new_factors*1000.))/1000.
         FOR i = 0, pol_det-1 DO xyouts,  0.02+0.08*i, 0.72,  strmid(strcompress(string(new_factors(i))$
                                     , /remove_all), 0, 5), /normal
         data_points = dimen(0)     
         xyouts, 0.017, 0.68, 'Number of data points: ' +strcompress(string(data_points), /remove_all), /normal
      
            
         
         spawn, 'echo $user', idnt
         xyouts, 0.01, 0.01, 'Generated by:  '+idnt+', '+systime(0), size = 0.9, /normal, color = black
stop_loop:
      ENDFOR        

      IF n_elements(plotname) NE 0 THEN BEGIN 
         image = tvrd(true = 1)
         write_png, plotname, image 
         print, 'Writing '+plotname+' to file...'
      ENDIF 
   ENDIF    
     
   IF keyword_set(FACTORS) THEN BEGIN 
   
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Compare the flux in the polar detector array to see if they are
;; calibrated correctly
;;
;; From other tests we found that all detectors cover the PA's around
;; 90 deg. 
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   desc_fa = desc+'factors'
   
;; Picking pitch angle limits where all polar detectors contribute,
;; generally found to be around 90 degrees
   

   print, 'checking detector intercalibration'
   
   
   pol_counts_tot = fltarr( pol_det)
   pol_flux_tot = fltarr( pol_det)

   az_counts_tot = fltarr( az_det)
   az_flux_tot = fltarr( az_det)

   pol_counts = fltarr( pol_det)
   pol_flux = fltarr( pol_det)

   az_counts = fltarr( az_det)
   az_flux = fltarr( az_det)

   
   
;;
;; Running through every time element of data for this day to 
;;
   
   FOR i = 0,n_el-1 DO BEGIN  
      
      
      e_alpha_test = reform(e_alpha(i, channel, *, *), pol_det, az_det)  
      ecounts_test = reform(ecounts_g( i, channel, *, *), pol_det, az_det)

      ;; checking for bad azimuthal sectors...
      

      pitch_ind = where(e_alpha_test GT PA_low AND e_alpha_test LT PA_high, hits)
      
      
      IF hits GT 0 THEN BEGIN 

         inc_counts = ecounts_test(pitch_ind)
         inc_angles = e_alpha_test(pitch_ind)

         
         which_pol_det = pitch_ind MOD pol_det 
         which_az_det = fix(pitch_ind/(pol_det))
         
         pol_counts(*) = 0
         pol_flux(*) = 0
         az_counts(*) = 0
         az_flux(*) = 0
         
         print, 'polar detectors      :', which_pol_det
         print, 'included pitch angles:', inc_angles
         print, 'included counts      :', inc_counts  
         FOR k = 0, hits-1 DO BEGIN
            pol_flux( which_pol_det(k)) =  pol_flux( which_pol_det(k))+inc_counts(k)
            pol_counts( which_pol_det(k)) = pol_counts( which_pol_det(k))+1
            
            az_flux( which_az_det(k)) =  az_flux( which_az_det(k))+inc_counts(k)
            az_counts( which_az_det(k)) = az_counts( which_az_det(k))+1
         ENDFOR 

         pol_flux_tot = pol_flux_tot+pol_flux
         pol_counts_tot = pol_counts_tot+pol_counts
         az_flux_tot = az_flux_tot+az_flux
         az_counts_tot = az_counts_tot+az_counts

      ENDIF  ELSE print, 'no counts for this PA range at time '+string(i)
   ENDFOR 

   pol_flux_tot = pol_flux_tot/pol_counts_tot
   az_flux_tot = az_flux_tot/az_counts_tot
   erase
   
   !p.multi = 0
   !p.multi = [0, 2, 2, 1, 0]
   plot, pol_counts_tot, xticks = 5, xtitle = 'Polar detector', ytitle = 'Counts'
   !p.multi = [3, 2, 2, 1, 0]
   plot, pol_flux_tot, xticks = 5, $
 xtitle = 'Polar detector', ytitle = 'Mean Raw Counts'
     !p.multi = [2, 2, 2, 1, 0]
     plot, az_counts_tot, xticks = 5, xtitle = 'Azimuthal sample', ytitle = 'Counts'
     
     !p.multi = [1, 2, 2, 1, 0]
     plot, az_flux_tot, xticks = 5, $
xtitle = 'Azimuthal sample', ytitle = 'Mean Raw Counts'
     
     factors = 0.95*pol_flux_tot/max(pol_flux_tot)
     print, 'Normalisation factors are:'
     print, factors
     
     filename = sat_name +'_'+date +'_'+desc_fa+'.png'
     print, 'writing image to file:'+filename
     image = tvrd(true = 1)
     write_png, filename, image

     
  ENDIF 
  
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Find the pitch angles that are covered by all 6 polar detectors
;; This routine plots up the pitch angle coverage from the different
;; polar detectors        
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
     IF keyword_set(COVERAGE) THEN BEGIN
        desc_co = desc+'coverage'

     
     window
     erase
     !p.multi = 0
     pol_hist_arr = fltarr(6, 180)
     plot_title = sat_name+' '+date+'  '+strcompress(string(n_el),/remove_all)+' time steps'
     FOR i = 0, pol_det-1 DO BEGIN 
        pol_alpha = reform(e_alpha(*, channel, i, *),n_el* az_det)
        pol_hist = histogram(pol_alpha, min = 0, max = 180, nbins = 180) 
        pol_hist_arr(i, *) = pol_hist
;    IF i eq 0 THEN plot, pol_hist, color = red, xtitle = 'Pitch angle (bin)', ytitle = 'counts', title = plot_title $
;ELSE oplot, pol_hist, color = red+i
        

     ENDFOR


     max_val = max(pol_hist_arr)

     plot,[0, 180],[0, max_val], color = black, xtitle = 'Pitch angle (bin)', $
    ytitle = 'counts',title = plot_title,/nodata 
     FOR i = 0, pol_det-1 DO  oplot, pol_hist_arr(i, *), color = red+i 

     plot_legend
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Printing image to a png file
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
     filename = sat_name +'_'+date +'_'+desc_co+'.png'
     print, 'writing image to file:'+filename
     image = tvrd(true = 1)
     write_png, filename, image
     
  ENDIF
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; find the average flux for each polar detector and pitch angle bin
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  IF keyword_set(FLUX) THEN BEGIN
     desc_fl = desc+'flux'
     window
     erase
     !p.multi = 0
     pol_flux = fltarr(pol_det, 36)
     st_time = 0
     int = n_el
     plot_title = sat_name+' '+date+'  '+strcompress(string(n_el),/remove_all)+' time steps'
     
     FOR i = 0, pol_det-1 DO BEGIN 
       
        pol_alpha = reform(e_alpha(st_time:st_time+int-1, channel, i, *), int* az_det)
        pol_efa = reform(efa(st_time:st_time+int-1, channel, i, *), int* az_det)
        pol_hist = histogram(pol_alpha, min = 0, max = 180, nbins = 36, reverse_indices = re) 
        FOR j = 0, 35 DO BEGIN 
           IF re(j+1) GT re(j) THEN pol_flux(i, j) = $ 
           total(pol_efa(re(re(j):re(j+1)-1)))/(re(j+1)-re(j))
        ENDFOR   
        
;        IF i eq 0 THEN plot, pol_flux, color = red, xtitle = 'Pitch angle (bin)', $
; ytitle = 'flux',title = plot_title  ELSE oplot, pol_flux, color = red+i
     ENDFOR 

     max_val = max(pol_flux)

     plot,[0, 35],[0, max_val], color = black, xtitle = 'Pitch angle (bin)', $
 ytitle = 'flux',title = plot_title, /nodata 
     FOR i = 0, pol_det-1 DO  oplot, pol_flux(i, *), color = red+i
        
     
     plot_legend

     filename = sat_name +'_'+date +'_'+desc_fl+'.png'
     print, 'writing image to file:'+filename
     image = tvrd(true = 1)
     write_png, filename, image
  ENDIF  

  IF keyword_set(RAW_COUNTS) THEN BEGIN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Find the average uncorrected counts for each polar detector and
; pitch angle bin
;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
     desc_rc = desc+'raw_counts'
     window
     erase
     !p.multi = 0
     pol_raw_plot = fltarr(pol_det, 36)
     st_time = 0
     int = n_el ;integrating over the whole day
     plot_title = sat_name+' '+date+'  '+strcompress(string(n_el), /remove_all)+' time steps'
     
     FOR i = 0, pol_det-1 DO BEGIN 
        
        pol_alpha = reform(e_alpha(st_time:st_time+int-1, channel, i, *), int* az_det)
        pol_raw = reform(ecounts_g(st_time:st_time+int-1, channel, i, *), int* az_det)
        pol_hist = histogram(pol_alpha, min = 0, max = 180, nbins = 36, reverse_indices = re) 
        FOR j = 0, 35 DO BEGIN 
           IF re(j+1) GT re(j) THEN pol_raw_plot(i, j) = $ 
           total(pol_raw(re(re(j):re(j+1)-1)))/(re(j+1)-re(j))
        ENDFOR   
        
;        IF i eq 0 THEN plot, pol_raw_plot, color = red, xtitle = 'Pitch angle (bin)', $
;  ytitle = 'raw counts',title = plot_title  ELSE oplot, pol_raw_plot, color = red+i
     ENDFOR 
     max_val = max(pol_raw_plot)

     plot,[0, 35],[0, max_val], color = black, xtitle = 'Pitch angle (bin)', $
 ytitle = 'raw_counts',title = plot_title, /nodata 
     FOR i = 0, pol_det-1 DO  oplot, pol_raw_plot(i, *), color = red+i
    
 plot_legend

     filename = sat_name +'_'+date +'_'+desc_rc+'.png'
     print, 'writing image to file:'+filename
     image = tvrd(true = 1)
     write_png, filename, image
  ENDIF  


ecounts = 0 & pcounts = 0 & mpa_pitch_data = 0

out:
  


END 

;*******************************************************************
;*  PROCEDURE
;*      mpa_plot_pa,  NUMBERED = numbered, TIME = time,  ENERGY = energy,  $
;*                WRITE_FILE = write_file, SIN_FIT = sin_fit
;*
;*  DESCRIPTION
;*       This procedure plots the pitch angle data from lanl geo MPA, which
;*       must be contained in the common block lanl_sopa_pitch_data
;*       (use a_lanl_mpa_pitch to read the data)  
;*
;*  KEYWORDS:
;*       NUMBERED: plots the data points as numbers according to 
;*                 azimuthal spin phase 
;*       TIME:     index of time to be plotted
;*       ENERGY:   energy channel to be plotted (10 channels for
;*                 electrons, 12 for protons)  
;*       WRITE_FILE: set this to write image to png file
;*       SIN_FIT:  if set, draws a p(0)*sin(pa)^p(1) fit to the pa
;*                 distribution
;* 
;*  DEPENDS ON:  ~/IDL/lib/mpfit
;*               ~/IDL/lib/mpfitfun &
;*               .r ~/papco_user/papco_user_modules/lanl_geo/plot_read/lanl_geo_pitch_tools
;*       
;*  MODIFICATION HISTORY:
;*       written July 2003, Arne Aasnes at LANL
;********************************************************************
PRO mpa_plot_pa, NUMBERED = numbered, TIME = time,  ENERGY = energy,SYMLINE = symline, $
                 WRITE_PNG = write_png, SIN_FIT = sin_fit, POLY_FIT = poly_fit, $
                 SPLINE_FIT = spline_fit, SPECIES = species, cust = cust, $
                 REM_PH_EL = rem_ph_el, FILE_NAME = file_name, MAX_SCL = max_scl, NO_REM_NEG = no_rem_NEG

  COMMON papco_color_names
  COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data
  COMMON lanl_mpa               ;see lanl_mpa_set_commons for contents

  plotted_x = !x  &  plotted_y = !y &  plotted_p = !p

  nodata = -1e31
  bird = strlowcase(mpa_pitch_data(0).csat_id)
  bird_no = where(bird EQ sat_names) &  bird_no = bird_no(0)


  IF keyword_set(ENERGY) THEN en = energy ELSE en = 10 
  en = 39-en ;reverse the index because array is ordered in reverse
  IF NOT keyword_set(TIME) THEN time = 0
  IF NOT keyword_set(SPECIES) THEN species = 0 ;0 (electrons) or 1 (ions)

  energy_label =r_lanl_mpa_channels(bird_no, species)
  
  en_ch = 40;dim(0)
  pol_det = 6;dim(0)
  az_det = 24;dim(1)
  IF species EQ 0 THEN BEGIN 
     raw_mag = reform(mpa_pitch_data(time).e_mag (en, *, *), 6, 24)
     raw_counts =  reform(mpa_pitch_data(time).e_unbinned(en, *, *), 6, 24)
  ENDIF ELSE BEGIN 
     raw_mag = reform(mpa_pitch_data(time).p_mag(en, *, *), 6, 24)
     raw_counts = reform( mpa_pitch_data(time).p_unbinned(en, *, *), 6, 24)
  endELSE 
  dim = size(raw_mag, /dimensions)

  nomag = where(finite(raw_mag) EQ 0, cnt)
  IF cnt GT 0 THEN raw_counts(nomag) = nodata
  
  IF bird EQ 'lanl-01a' THEN raw_counts(1, *) = nodata
  IF bird EQ '1994-084' THEN raw_counts(3, *) = nodata
  IF bird EQ 'lanl-97a' THEN raw_counts(5, *) = nodata

; allow for negative values to be retained
  IF NOT keyword_set(NO_REM_NEG) THEN BEGIN 
     IF min(raw_counts) LE 0 THEN raw_counts(where(raw_counts LE 0)) = nodata
  ENDIF 

  raw_mag_st =  reform(raw_mag( *, *),pol_det*az_det)
;  dim = size(raw_counts, /dimensions)
  raw_counts_st =  reform(raw_counts( *, *),pol_det*az_det)


  IF keyword_set(REM_PH_EL) AND species EQ 0 AND energy LE 33 THEN BEGIN
     message, 'Filtering for photo electrons', /cont
     fact_lim = 1.3
     nodat_idx = where(raw_counts LE 0 OR finite(raw_counts) EQ 0, nodat_cnt)
     if nodat_cnt GT 0 THEN raw_counts(nodat_idx) = !values.f_nan
     
     print, string(nodat_cnt)+' invalid points before filtering'
     
     
     az_mean = fltarr(az_det)
     pol_mean = fltarr(pol_det)


     FOR j = 0, az_det-1 DO az_mean(j) = mean(raw_counts( *, j), /nan)
     az_mean_sort = az_mean(sort(az_mean))
     low_mean = mean(az_mean_sort(0:14)) ; taking the mean of the 15 azimuthal sectors with lowest counts
     bad_sectors = where(az_mean GT fact_lim*low_mean, bad)
     IF bad GT 0 THEN  FOR bb = 0, bad-1 DO  raw_counts( *, bad_sectors(bb)) = !values.f_nan
;     IF bad EQ 0 THEN BEGIN 
     
     FOR j = 0, pol_det-1 DO pol_mean(j) = mean(raw_counts( j, *), /nan)
     pol_mean_sort = pol_mean(sort(pol_mean))
     low_mean = mean(pol_mean_sort(0:4))
     bad_sectors = where(pol_mean GT fact_lim*low_mean, bad)
     IF bad GT 0 THEN  FOR bb = 0, bad-1 DO  raw_counts( bad_sectors(bb), *) = !values.f_nan
;     ENDIF
;     nodat_idx = where(raw_counts NE nodata, nodat_cnt)   
     nodat_idx = where(finite(raw_counts) EQ 0, nodat_cnt)
     print, string(nodat_cnt)+' invalid points after filtering'
     IF nodat_cnt GT 0 THEN raw_counts(nodat_idx) = nodata ;converting back to standard nodata value
  ENDIF 
no_rem_photo:
  
  IF max(raw_counts) EQ nodata THEN GOTO,  error


  cleanplot
  !p.position = 0
  !p.background = white
  !y.margin = [7, 4] & !x.margin = [12, 5]
  window, title = 'MPA Counts vs pitch angle'
  erase 

  mjd_time = tai2utc(mpa_pitch_data(time).tai)
  mjd_t = {mjd_t, mjd:mjd_time.mjd,t:mjd_time.time}
  convert_t90_to_date, mjd_t, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; making the string title with time  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  uthours = mpa_pitch_data(time).uthours
  hour = floor(uthours)
  minute = floor((uthours-hour)*60.)
  second = floor((uthours-hour-minute/60.)*60.)
  
  stryear = strcompress(string(year), /remove_all)
  IF month LT 10 THEN  strmonth = '0'+strcompress(string(month), /remove_all) $
  ELSE strmonth = strcompress(string(month), /remove_all)
  IF day_om LT 10 THEN strday_om = '0' +strcompress(string(day_om), /remove_all) $
  ELSE strday_om = strcompress(string(day_om), /remove_all) 
  strdate = stryear+strmonth+strday_om
  IF hour LT 10 THEN strhour = '0' +strmid(strcompress(string(hour), /remove_all), 0, 1)$
  ELSE strhour = strmid(strcompress(string(hour), /remove_all), 0, 2)

  IF minute LT 10 THEN strmin = '0' +strcompress(string(minute), /remove_all) $
  ELSE strmin = strcompress(string(minute), /remove_all)
  IF second LT 10 THEN strsec = '0'+strcompress(string(second), /remove_all)$
  ELSE strsec = strmid(strcompress(string(second), /remove_all), 0, 2)

  title_str = 'Mpa counts vs Pitch Angles '+mpa_pitch_data(0).csat_id+ $
        ' !C energy channel:'+strcompress(string(energy+1),/remove_all)+ $
   ' date: '+ strdate+' time:'+ $
   strhour+':'+strmin
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; plotting 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  y_max = 1.1*max(raw_counts( *, *))
  IF keyword_set(max_SCL) THEN y_max = max_scl
  plot, [0, 180],[0,y_max],/nodata, min_value = 0, $
   color = black, ytitle = 'Energy Flux ', xtitle = 'Pitch Angle (deg)', title = title_str, $
   ystyle = 9, xstyle = 9

x0 = 0.85
x1 = x0+0.03
y0 = 0.9
ydiff = 0.03
xdiff = 0.01


col = white+1
FOR h = 0, pol_det-1 DO BEGIN
   IF keyword_set(numbered) THEN $
      FOR i = 0, az_det-1 DO xyouts, raw_mag( h, i),raw_counts( h, i), $
     strcompress(string(i+1), /remove_all), color =col+h, /data, alignment = 0.5, charsize = 0.9 ELSE $
     oplot, raw_mag( h, *),raw_counts(h, *), color =col+h, psym = 1
;   plots, [x0, x1], [y0-h*ydiff+0.3*ydiff, y0-h*ydiff+0.3*ydiff], color = col+h, /normal
   xyouts, x1+xdiff, y0-h*ydiff, 'Det '+strcompress(string(h+1)), /normal, color = col+h
ENDFOR   
IF keyword_set(symline) THEN oplot, [90, 90], [0, 0.95*y_max], color = black, linestyle = 1

IF keyword_set(poly_fit) OR keyword_set(spline_fit) THEN BEGIN 
   val_idx = where(raw_counts_st NE nodata AND finite(raw_counts_st) NE 0, val_cnt)
   IF val_cnt GT 0 THEN BEGIN 
      y = raw_counts_st(val_idx)
      x = raw_mag_st(val_idx)
   endIF 
ENDIF 

IF keyword_set(sin_fit) THEN BEGIN 
   COMMON  fit_result, result, fit_count, fit_alpha
   IF keyword_set(cust) THEN cust = 1 ELSE cust = 0
   pa_fit, raw_mag_st(*), raw_counts_st(*), cust = cust
   oplot, fit_alpha, fit_count, color = black
   IF keyword_set(cust) THEN print, 'Params:'+string(result)
   print, 'Params (p(0)*sin(x*p(1))^p(2)):'+string(result(0))+string(result(1))
ENDIF 
IF keyword_set(poly_fit) THEN BEGIN 
;   x = raw_mag_st(*)
;   y = raw_counts_st(*)
   gt90 = where(x GT 90, gt90_cnt)
   IF gt90_cnt GT 0 THEN x(gt90) = 180-x(gt90)
   order = sort(x)
   x =x(order)
   y =y(order)/1000e ;polyfit likes small numbers!
   chisq_arr = fltarr(10)

   FOR i = 1, 10 DO BEGIN 
      pf_res = svdfit(x, y, i, CHISQ = chisq)
      chisq_arr(i-1) = chisq
   endFOR  
   res = min(chisq_arr, min_pos)
   pf_res = svdfit(x, y, min_pos+1, YFIT = yfit)
  

;   pf_res = poly_fit(x, y, 5, yfit)
   le90 = where(x LE 90, le90_cnt)
   IF le90_cnt GT 0 THEN BEGIN 
      x = [x(le90), 180-reverse(x(le90))]
      yfit = [yfit(le90),reverse(yfit(le90))]
   ENDIF 
   oplot, x, yfit*1000e, color = black
ENDIF 
IF keyword_set(spline_fit) THEN BEGIN 
;   x = raw_mag_st(*)
;   y = raw_counts_st(*)
   order = sort(x)
   x = x(order)
   yfac = max(y)/10
   y = y(order)/yfac ;polyfit likes small numbers!
   t = findgen(181)
;   sp_res = spline(x, y, t)*10000
;   spline_p, x, y, xx, yy, interval = 1
;   oplot, xx, yy*10000, color = black
   yerr = sqrt(max(y))
   dist = 3
   spline_smooth, x, y, yerr, dist, coefficients, smoothness, t, yplot, /plot
   oplot, t, yplot*yfac, color = black
ENDIF 
spawn, 'echo $user', idnt
xyouts, 0.01, 0.02, 'Generated by:  '+idnt+', '+systime(0), /normal, color = black
IF keyword_set(WRITE_PNG) THEN BEGIN 
   path = '~/lanl_geo/mpa/plots/'
   IF keyword_set(file_name)  THEN plotname = file_name ELSE $ 
   plotname = path+'PA_'+bird+'_'+strdate+'_'+ $
 strmid(strcompress(string(mpa_pitch_data(time).uthours),/remove_all), 0, 5)+ $
'hr'+'_en_chan'+strcompress(string(energy+1),/remove_all)+'.png'
   print, 'Writing '+plotname+' to file'
   image = tvrd(true = 1)
   write_png, plotname, image   
ENDIF 

!x =  plotted_x  &  !y = plotted_y  &  !p = plotted_p

return
error:
message, 'No data to plot', /cont

END 


