;***************************************************************************************
;* PROCEDURE: 
;*      flux,infile,VERBOSE = verbose,  OUTPUT = output
;* DESCRIPTION:
;*      this procedure calls a routine readcdf, to get the detector
;*      and spin resolution data from lanl-mpa, and calculates the
;*      pitch angle for each data point. Data is binned and returned
;*      in the common block lanl_mpa_pitch_data   
;* INPUTS: 
;*      infile   name (and path) of cdf file to read
;* OUTPUTS:
;*      none     (all in common blocks)    
;* KEYWORDS: 
;*      VERBOSE  give feedback on process
;*      OUTPUT   return the raw data before doing any binning (in
;*               common block lanl_mpa_pitch_raw)
;*      MAG_CORR removes singular points and interpolates 
;*
;* MODIFICATION HISTORY: 
;*      Written May 2003, Arne Aasnes at LANL
;*      Modified August 2003, AAA at LANL, 
;*                 no longer bins the data in pitch angle bins, this
;*                 is now done in the plot routines to increase
;*                 flexibility.  
;*      Modified October 2003, AAA at LANL
;*                 added option for magnetic field correction.
;*      Last modified: 10/31/2003
;*               
;***************************************************************************************


PRO flux,infile, VERBOSE = verbose,  OUTPUT = output, MAG_CORR = mag_corr

  COMMON lanl_mpa_pitch_data,  mpa_pitch_header,mpa_pitch_data,  mpa_pitch_sat
  COMMON mpa_pitch_time, mpa_pitch_year, mpa_pitch_month, mpa_pitch_day_om
  COMMON lanl_mpa ; set in lanl_mpa_set_commons
  COMMON lanl_mpa_pitch_raw, uthours, effece, effecp, ecounts, pcounts, $
                             gfactor, etimesg, e_aniso,p_aniso              ; output for diagnostics
  COMMON get_error, get_err_no, get_err_msg

  !quiet=1  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; GET DATA from the CDF file ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  
  IF keyword_set(VERBOSE) THEN print,infile
  ON_ERROR, 0 ;just stop

  data = readcdf(infile, VERBOSE = verbose)        ;stores all data in the structure data.var_struct
  
  IF get_err_no NE 0 THEN GOTO, out
  
  IF keyword_set(VERBOSE) THEN $
print, 'Finished reading datafile,  now calculating pitch angles...'
  ie = 40                       ; total energy channels
  ip = 6                        ; total polar channels
  is = 24                       ; total spin channels
  
  nodata = -1e31
  
; number of array

;  epoch_ind = where(tag_names(data) EQ 'EPOCH'
;  IF ind(0) EQ -1 THEN BEGIN 

; data should contain 26 structures, if there are less something is missing
  IF n_tags(data) LT 26 THEN BEGIN 
     get_err_no = 2 &  get_err_msg = 'Not enough info in data file'
     data = 0
     GOTO, out
  ENDIF 
  ileng = n_elements(data.Epoch.data)

IF n_elements(year) EQ 0 THEN BEGIN 
   datestr = strmid(infile,15, 8, /reverse_offset)
   year = strmid(datestr, 0, 4)
   month = fix(strmid(datestr, 4, 2))
   day_om = fix(strmid(datestr, 6, 2))
endif
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; B_Flag gives info about the safety of the B field determination, 
;; 0 = determined from e anisotropy
;; 1 = determined from p anisotropy
;; 2 = determined from hip anisotropy
;; 3 = error value (nodata)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
B_flag = make_array(ileng, value = 4)

 

  IF n_elements(input_sat) EQ 0 THEN BEGIN
     pos = strpos(infile, '_h0_', /reverse_search)
     IF pos NE -1 THEN input_sat = sat_names(where(strmid(infile, pos-2, 2) EQ sat_codes))

     IF pos EQ -1 THEN input_sat = strmid(infile,23, 8, /reverse_offset)
     input_sat = input_sat(0)
  ENDIF 
     
  
  
;******************************************************************
;  To calculate the ion flux from the count rate
;******************************************************************
; C/t = e1 * e2 * G * E *j
; where E is energy (eV), j is number flux (cm^-2 s^-1 sr^-1 eV^-1)
;    and t = 9ms
; j ~ C/(t*e1*e2*G*E)
; j = C/(9e-3s*e1*e2*G*E)
;******************************************************************
  
  r2d = !dpi/180.                ; d2r(!)
  
;******************************************************************
;
; Renaming the variables for shorter notation + some minor calcs
; Converting all angles to radians
;
;******************************************************************
  
  IF rstrpos(infile, '_h0_') ne -1 THEN BEGIN
     theta_phi_hip = data.thetaphi_hip.data
     theta_phi_e = data.thetaphi_e.data
  ENDIF  ELSE BEGIN 
     theta_phi_hip = data.theta_phi_hip.data
     theta_phi_e   = data.theta_phi_e.data
  ENDELSE 
  uthours = data.uthours.data
  Pcounts = data.Pcounts.data   ;raw electron counts 
  Ecounts = data.Ecounts.data   ;raw proton counts

  etimesg = data.etimesg.data   ;efficiency times geometric factor common for electrons and ions
  ece     = data.ece.data       ;center electron channel energy
  ecp     = data.ecp.data       ;center proton channel energy
  effecp = data.effecp.data     ;channel dependent efficiencies (protons)
  effece = data.effece.data     ; (electrons)

  temp_e  = data.temp_e.data    ; temperature of electrons
  temp_hip = data.temp_hip.data ; temperature of high energy protons
  
  theta_phi_hip = theta_phi_hip*r2d ; magnetic field direction in satellite coords, based on high energy proton 
  theta_phi_e = theta_phi_e*r2d ; magnetic field direction in satellite coords, based on electrons
;  Azanglp = data.Azanglp.data   ;azimuthal angle for electron sample
;  Azangle = data.Azangle.data   ;
  ephij =  data.Azangle.data*r2d
  pphij =  data.Azanglp.data*r2d
  thetaD = 0.5*(data.theta_l.data +data.theta_u.data)*r2d ;center angle of polar detector array
  
  data = 0 ; resetting the structure now that we've read out all necessary data

;; calling routine to get all efficiency factors

  yyyymm = year*100.0+month

IF NOT keyword_set(OUTPUT) THEN BEGIN 
  corr_factors = r_lanl_mpa_flux_conversion( input_sat,yyyymm)
  etimesg = corr_factors.etimesg
  gfactor = corr_factors.gfactor
  eff_factors = corr_factors.factors
ENDIF 
;  ece = corr_factors.ece
;  ecp = corr_factors.ecp
;  effecp = corr_factors.effecp
;  effece = corr_factors.effece


  pfactor = 1./9e-3/effecp;/ecp
  efactor = 1./9e-3/effece;/ece

; geometric factor of detectors
  gfactor = [3.26e-4, 4.9e-4, 5.74e-4, 5.74e-4, 4.9e-4, 3.26e-4]
  print, 'Efficiency factors: '
IF keyword_set(output) THEN print, etimesg/gfactor ELSE  print, eff_factors

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Setting up the return data structure;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  mpa_pitch_header = {mpa_pitch_header, tag1:'Time in decimals of hours from beginning of day', $
; tag2:'Binned Electron flux (cm^-2 s^-1 sr^-1 eV^-1)(time,energy,pitch angles)', $
; tag3:'Binned Proton flux (cm^-2 s^-1 sr^-1 eV^-1)(time,energy,pitch angles)', $
 tag2:'Unbinned Electron flux (cm^-2 s^-1 sr^-1 eV^-1)(time,energy,polar detectors,azimuthal detectors)', $
 tag3:'Electron Pitch angles (time,energy,polar detectors,azimuthal detectors)', $
 tag4:'Unbinned Proton flux (cm^-2 s^-1 sr^-1 eV^-1)(time,energy,polar detectors,azimuthal detectors)', $
 tag5:'Proton Pitch angles (time,energy,polar detectors,azimuthal detectors)', $
; tag6:'Center energy of electron detectors',  tag7:'Center energy of proton detectors', $
; tag8:'Pitch angle limits of data', 
 tag6:'tai time',$ ; tag7:'modified julian day time', $
 tag7:'satellite id', tag8:'magnetic field direction, in theta, phi, satellite coords', $
 tag9:'B determination flag; 0: from e, 2: from hip,3:low anisotropy, 4: error, 5: Error, but Interpolated value'} 
  dummy = {uthours:0d, $
 ; e_pitch_data:fltarr(ie, bin_nr), p_pitch_data:fltarr(ie, bin_nr), $
         e_unbinned:fltarr(ie, ip, is), e_mag:fltarr(ie, ip, is), $
         p_unbinned:fltarr(ie, ip, is), p_mag:fltarr(ie, ip, is), $
;         ece:ece, ecp:ecp, pitch_angles:pitch_lims, $
         tai:0d, csat_id:input_sat, mag_dir:fltarr(2), B_flag:0e}
  
  max_no_rec = ileng            ; we don't make the structure bigger than it got to be
  mpa_pitch_data = replicate(dummy, max_no_rec)

;mpa_pitch_data = {uthours:fltarr(ileng),  e_pitch_data:fltarr(ileng, ie, bin_nr), $
;             p_pitch_data:fltarr(ileng, ie, bin_nr), ece:fltarr(ie),ecp:fltarr(ie),  $
;             pitch_angles: fltarr(ileng, bin_nr+1), tai:fltarr(ileng), mjdt_time:fltarr(ileng), $
;            csat_id:make_array(ileng, value = input_sat)}   
  
  
;***************************************************************************;
;
; Calculating the energy flux for each detector and spin angle
;
;***************************************************************************;
  
  
   efactor_4dim = rebin(reform(efactor, 1, ie, 1, 1),ileng, ie, ip, is)
   pfactor_4dim = rebin(reform(pfactor, 1, ie, 1, 1),ileng, ie, ip, is)
   etimesg_4dim = rebin(reform(etimesg, 1, 1, ip, 1), ileng, ie, ip, is)
   ece_4dim = rebin(reform(ece, 1, ie, 1, 1), ileng, ie, ip, is)
   ecp_4dim = rebin(reform(ecp, 1, ie, 1, 1), ileng, ie, ip, is)

   
   efa = efactor_4dim*Ecounts/etimesg_4dim;*ece_4dim
   pfa = pfactor_4dim*Pcounts/etimesg_4dim;*ecp_4dim

; resetting some arrays that are no longer needed   
    efactor_4dim = 0 &  pfactor_4dim = 0 
   
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Finding the correct B field direction, based on electron or ion 
; temperature anisotropies
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   
;   par_e_temp_4dim =rebin(reform(temp_e(*, 0), ileng, 1, 1, 1), ileng, ie, ip, is)
;   perp_e_temp_4dim =rebin(reform(temp_e(*, 1), ileng, 1, 1, 1), ileng, ie, ip, is)

;; check for missing information, sometimes the center energy
;; information of a channel is missing, indicating useless data. A
;; component of the temperature can also be missing in some cases. 
   no_ece = where(ece_4dim EQ -1, nece) ;dim ie
   no_ecp = where(ecp_4dim EQ -1, necp) ;dim ie

; resetting some arrays that are no longer needed  
   ece_4dim = 0 &  ecp_4dim = 0

   no_par_e_temp = where(temp_e(*, 0) LE  0, npar_etemp) ;dim ileng,2
   no_perp_e_temp = where(temp_e(*, 1) LE  0., nperp_etemp) ;dim ileng,2

   e_aniso = temp_e(*, 0)/temp_e(*, 1) ; perpendicular over parallel
   p_aniso = temp_hip(*, 0)/temp_hip(*, 1)



   thetaB = theta_phi_e(*, 0)
   phiB = theta_phi_e(*, 1)
;   p_thetaB = thetaB
;   p_phiB = phiB
   

; find where we have good determination of field direction

   good_e_aniso = where(e_aniso GT 1.1, good_cnt)
   IF good_cnt GT 0 THEN B_flag(good_e_aniso) = 0

; if hip has better anisotropy than e, use hip

   better_p_aniso = where(p_aniso GT e_aniso+0.1 and e_aniso lt 1.1, better_p_cnt)
   IF better_p_cnt GT 0 THEN BEGIN 
      thetaB(better_p_aniso) = theta_phi_hip(better_p_aniso, 0)
      phiB(better_p_aniso) = theta_phi_hip(better_p_aniso, 1)
      B_flag(better_p_aniso) = 2
   endIF 

   bad_aniso = where(e_aniso LT 1.1 AND p_aniso LT 1.2, bad_cnt)
   IF bad_cnt GT 0 THEN B_flag(bad_aniso) = 3
   
   


;; special case for LANL-01A
sat_code = strmid(infile, strpos(infile, '/', /reverse_search)+1, 2)
;IF  sat_code EQ 'a1' THEN BEGIN 
;   thetaB = theta_phi_e(*, 0)
;   phiB = theta_phi_e(*, 1)
;   B_flag(*) = 1
;endif


   IF nperp_etemp  NE 0 THEN BEGIN
      thetaB(no_perp_e_temp) = theta_phi_hip(no_perp_e_temp, 0)
      phiB(no_perp_e_temp) = theta_phi_hip(no_perp_e_temp, 1)
;      p_thetaB(no_perp_e_temp) = theta_phi_hip(no_perp_e_temp, 0)
;      p_phiB(no_perp_e_temp) = theta_phi_hip(no_perp_e_temp, 1)
      B_flag(no_perp_e_temp) = 4
   ENDIF 
   IF npar_etemp NE 0 THEN BEGIN
;      e_alpha(no_par_e_temp) = nodata
;      efa(no_par_e_temp) = nodata
      B_flag(no_par_e_temp) = 4
   ENDIF 


   err_b_field = where(abs(thetaB) GT !dpi OR abs(phiB) GT !dpi, errB)
   IF errB GT 0 THEN BEGIN 
      thetaB(err_b_field) = nodata
      phiB(err_b_field) = nodata
      B_flag(err_b_field) = 4
   ENDIF 

;   b_flag_4dim = rebin(reform(B_flag,ileng, 1, 1, 1), ileng, ie, ip, is)
;   bad_B = where(b_flag_4dim EQ 4, bb)

IF keyword_set(MAG_CORR) AND ileng GE 3 THEN BEGIN 
   org_thetaB = thetaB
   org_phiB = phiB
   temp_nodata = -1 ; smooth does not handle the original nodata value!!
  IF min(thetaB) EQ nodata THEN thetaB(where(thetaB EQ nodata)) = temp_nodata
  IF min(phiB) EQ nodata THEN phiB(where(phiB EQ nodata)) = temp_nodata
  lim = 0.1 
  
  sm_thetaB = smooth(thetaB, 3,/edge_truncate)
  first_it_keep = where(abs(sm_thetaB - thetaB ) LT lim AND thetaB NE temp_nodata, keep_pts)
  IF keep_pts LT ileng AND keep_pts GT 2 THEN  BEGIN 
     sm_thetaB = smooth(thetaB(first_it_keep), 3, /edge_truncate)
     sm_interp_thetaB = interpol(sm_thetaB, uthours(first_it_keep), uthours)
     sec_it_keep_theta = where(abs(sm_interp_thetaB-thetaB) LT lim AND thetaB NE temp_nodata,  keep_theta_pts)
     IF keep_theta_pts GT 0 THEN thetaB = interpol(thetaB(sec_it_keep_theta), uthours(sec_it_keep_theta), uthours)
  ENDIF ELSE BEGIN 
     rem_theta_pts = 0
     keep_theta_pts = ileng
     sec_it_keep_theta = indgen(ileng)
  ENDELSE 
  sm_phiB = smooth(phiB, 3,/edge_truncate)
  first_it_keep = where(abs(sm_phiB - phiB) LT lim AND phiB NE temp_nodata, keep_pts)
  IF keep_pts LT ileng AND keep_pts GT 2 THEN BEGIN 
     sm_phiB = smooth(phiB(first_it_keep), 3, /edge_truncate)
     sm_interp_phiB = interpol(sm_phiB, uthours(first_it_keep), uthours)
     sec_it_keep_phi = where(abs(sm_interp_phiB-phiB) LT lim AND phiB NE temp_nodata,  keep_phi_pts)
     IF keep_phi_pts GT 0 THEN phiB = interpol(phiB(sec_it_keep_phi), uthours(sec_it_keep_phi), uthours)
  ENDIF ELSE BEGIN 
     rem_phi_pts = 0
     keep_phi_pts = ileng
     sec_it_keep_phi = indgen(ileng)
  ENDELSE 
;  use null_array to find the union of sec_it_rem_theta AND
;  sec_it_rem_phi, as we need to remove bad data points in either array

  null_arr = make_array(ileng, value = 1)
  IF keep_theta_pts GT 0 THEN null_arr(sec_it_keep_theta) = null_arr(sec_it_keep_theta)-1;0
  IF keep_phi_pts GT 0 THEN null_arr(sec_it_keep_phi) = null_arr(sec_it_keep_phi)-1; 0
  interpolated = where(null_arr NE  -1, bad)
  IF bad GT 0 THEN BEGIN 
     B_flag(interpolated) = 5
  endIF 
;  keep = where(null_arr EQ 0, good)
;  IF good GT 0 THEN BEGIN
;     thetaB = interpol(thetaB(keep), uthours(keep), uthours)
;     phiB = interpol(phiB(keep), uthours(keep), uthours)
;  ENDIF 
ENDIF 
   


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Calculate pitch angle
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   
   thetaD_4dim = rebin(reform(thetaD,1, 1, ip, 1), ileng, ie, ip, is) ;detector polar angel
   e_phij_4dim = rebin(reform(ephij, ileng, ie, 1, is), ileng, ie, ip, is) ;detector azimuth
   p_phij_4dim = rebin(reform(pphij, ileng, ie, 1, is), ileng, ie, ip, is)
 ;  p_thetaB_4dim = rebin(reform(p_thetaB, ileng, 1, 1, 1), ileng, ie, ip, is)
   thetaB_4dim = rebin(reform(thetaB, ileng, 1, 1, 1), ileng, ie, ip, is) ;B polar
 ;  p_phiB_4dim = rebin(reform(p_phiB, ileng, 1, 1, 1), ileng, ie, ip, is)
   phiB_4dim = rebin(reform(phiB, ileng, 1, 1, 1), ileng, ie, ip, is) ; B azimuth
   
   e_cosalpha = sin(thetaD_4dim)*cos(e_phij_4dim)*sin(thetaB_4dim)*cos(phiB_4dim) + $
                      sin(thetaD_4dim)*sin(e_phij_4dim)*sin(thetaB_4dim)*sin(phiB_4dim) + $
                      cos(thetaD_4dim)*cos(thetaB_4dim)
   
   e_alpha = acos(e_cosalpha)/r2d 
   
;; changed 08/19/2003: use the magnetic field direction derived from
;; electrons (AAA)
   p_cosalpha = sin(thetaD_4dim)*cos(p_phij_4dim)*sin(thetaB_4dim)*cos(phiB_4dim) + $
                      sin(thetaD_4dim)*sin(p_phij_4dim)*sin(thetaB_4dim)*sin(phiB_4dim) + $
                      cos(thetaD_4dim)*cos(thetaB_4dim)
   
   p_alpha = acos(p_cosalpha)/r2d 

   thetaD_4dim = 0 & e_phij_4dim =0 & p_phij_4dim = 0 & thetaB_4dim = 0 
   phiB_4dim = 0 & e_cosalpha = 0& p_cosalpha = 0
   
   IF nece NE 0 THEN BEGIN
      e_alpha(no_ece) = nodata
      efa(no_ece) = nodata
   ENDIF   
   IF necp NE 0 THEN BEGIN
      p_alpha(no_ecp) = nodata
      pfa(no_ecp) = nodata
   ENDIF 

;   IF bb GT 0 THEN BEGIN 
;      e_alpha(bad_B) = nodata
;      p_alpha(bad_B) = nodata
;      efa(bad_B) = nodata
;      pfa(bad_B) = nodata
;   ENDIF 



;   IF keyword_set(output) THEN BEGIN 
;      IF e_cnt GT 0 THEN ecounts(e_index) = 0
;      IF p_cnt GT 0 THEN pcounts(p_index) = 0

;      GOTO, out
;   ENDIF 
   IF keyword_set(VERBOSE) THEN $
  print, 'Constructing the data structure...'
   
   hr = fix(uthours) & mn = (uthours-hr)*60d
   sec = fix((mn-fix(mn))*60d) &  mn = fix(mn)
   hr = double(hr)
   mn = double(mn)
   sec = double(sec)
   
   FOR i = 0, ileng-1 DO BEGIN
      dummy.uthours = uthours(i)
      dummy.tai = UTC2TAI({YEAR:year, MONTH:month, DAY:day_om, $
                            HOUR:hr(i),  MINUTE:mn(i), SECOND:sec(i), MILLISECOND:0})
      dummy.e_mag = e_alpha(i, *, *, *)
      dummy.p_mag = p_alpha(i, *, *, *)
      dummy.e_unbinned = efa(i, *, *, *)
      dummy.p_unbinned = pfa(i, *, *, *)

      dummy.mag_dir = [thetaB(i), phiB(i)]
      dummy.B_flag = B_flag(i)
      mpa_pitch_data(i) = dummy
      
   ENDFOR 
   e_alpha = 0 &  p_alpha = 0 & efa = 0 &  pfa = 0 & B_flag = 0 
   ephij = 0 & pphij = 0 

   return
out:

mpa_pitch_data = 0
get_err_no = 1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;* WHAT'S IN THE INPUT DATA STRUCTURE??? (name, descripition, size)
;;* 
;;* ece/ecp:         center energy of detector channel (40)
;;* effece/effecp:   energy depend eff.elec/ion values (40)
;;* ecounts/pcounts  actual counts in detecor (ndat,40,6,24)
;;* etimesg          geometric factor*efficiency (6)
;;* temp_e           temperature electrons (ndat,2)
;;* temp_hip         temperature high-E p (ndat,2)
;;* theta_l/theta_u  lower/upper edge look angles (6) 
;;                                                
;;* Azanglp/Azangle  Azimuthal spin phase angle (ndat,40,24)
;;*  
;;*
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
END

