;******************************************************************************
;* PROCEDURE:     
;*      p_lanl_mpa_line, panel, bird, dataType, channel, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel    three element integer vector. Controls positioning of the plot
;*               (see papco_conventions.text in $papco)
;*      bird     Satellite number
;*      dataType Data category
;*      channel  Energy channel, only relevant if average ion or electron flux
;*               chosen
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_lanl_mpa_line, panel, bird, dataType, channel, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written August 1998, Haje Korth
;******************************************************************************
pro p_lanl_mpa_line, panel, bird, dataType, channel, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_lanl_mpa. 
COMMON lanl_mpa_data, input_header, input_data, input_sat
COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents
  
panelset,panel			;sets the panel position viewport

; restrict data to actual time range requested - this makes for faster zooming
; input data is in TAI. For plotting, always start at zero to get max
; max resolution (plot works in float only, large values get rounded! 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

; restrict data in time  
index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c GT 2 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1  & time=data.tai-tai_1

ndat=n_elements(data)  

utitle = sat_names(bird)+'!C'
 
case dataType of
     0:begin
         yray=data(*).ave_flux_e(channel)
         utitle=utitle+'Av. El. Flux!CChannel: '+STRTRIM(STRING(channel),2)
         uytitle='cm!E-2!Ns!E-1!Nsr!E-1!NeV!E-1!N'
       end
     1:begin
         yray=data(*).ave_flux_i(channel)
         utitle=utitle+'Av. Ion Flux!CChannel: '+STRTRIM(STRING(channel),2)
         uytitle='cm!E-2!Ns!E-1!Nsr!E-1!NeV!E-1!N'
       end
     2:begin
         yray=data(*).backgrd
         utitle=utitle+'Background'
         uytitle='Counts'
       end
     3:begin
         yray=data(*).sc_poten
         utitle=utitle+'S/C Potential'
         uytitle='V'
       end
     4:begin
         yray=data(*).xgeo
         utitle=utitle+'S/C X Coord.'
         uytitle='km'
       end
     5:begin
         yray=data(*).ygeo
         utitle=utitle+'S/C Y Coord.'
         uytitle='km'
       end
     6:begin
         yray=data(*).zgeo
         utitle=utitle+'S/C Z Coord.'
         uytitle='km'
       end
     7:begin
         yray=data(*).scmlat
         utitle=utitle+'S/C Mag. Lat.'
         uytitle='Deg.'
       end
     8:begin
         yray=data(*).scmlt
         utitle=utitle+'S/C Mag. LT'
         uytitle=''
       end
     9:begin
         yray=data(*).denlp
         utitle=utitle+'Density LP'
         uytitle='cm!E-3!N'
       end
    10:begin
         yray=data(*).vellp
         utitle=utitle+'Velocity LP'
         uytitle=''
       end
    11:begin
         yray=data(*).denhp
         utitle=utitle+'Density HP'
         uytitle='cm!E-3!N'
       end
    12:begin
         yray=data(*).tparhp
         utitle=utitle+'T-Par. HP'
         uytitle='eV'
       end
    13:begin
         yray=data(*).tperhp
         utitle=utitle+'T-Perp. HP'
         uytitle='eV'
       end
    14:begin
         yray=data(*).theta_p
         utitle=utitle+'Theta HP'
         uytitle='Deg.'
         trat=data(*).tperhp/data(*).tparhp
       end
    15:begin
         yray=data(*).phi_p
         utitle=utitle+'Phi HP'
         uytitle='Deg.'
         trat=data(*).tperhp/data(*).tparhp
       end
    16:begin
         yray=data(*).denhe
         utitle=utitle+'Density HE'
         uytitle='cm!E-3!N'
       end
    17:begin
         yray=data(*).tparhe
         utitle=utitle+'T-Par. HE'
         uytitle='eV'
       end
    18:begin
         yray=data(*).tperhe
         utitle=utitle+'T-Perp. HE'
         uytitle='eV'
       end
    19:begin
         yray=data(*).theta_e
         utitle=utitle+'Theta HE'
         uytitle='Deg.'
         trat=data(*).tperhe/data(*).tparhe
       end
    20:begin
         yray=data(*).phi_e
         utitle=utitle+'Phi HE'
         uytitle='Deg.'
         trat=data(*).tperhe/data(*).tparhe
       end
    21:begin
         yray=data(*).denhp* $
              (2*data(*).tperhp+data(*).tparhp)/3 * 1.6022e-4
         utitle=utitle+'Pressure HP'
         uytitle='nPa'
       end
endcase

; restrict data to valid points
index=where(yray ne -1.0,count)
if count ne 0 then begin
    yray=yray(index) & time=time(index)
endif
   
; restrict theta and phi to valid temperature ratio tper/tpar>1.05 or <0.95
if (dataType eq 14) or (dataType eq 15) or $
   (dataType eq 19) or (dataType eq 20) then begin
    index=where(trat gt 1.05,count)
    if count ne 0 then begin
        yray=yray(index) & time=time(index)
    endif
endif

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.
   
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else BEGIN
    papco_autorange, yray, yst, yen, LOG = yscl(panel(0),3),  $
        NODATA = -99, exclude=0
    yscl(panel(0),1)=yst
    yscl(panel(0),2)=yen
endelse

; use auto y-labeling routine to get "good" y-lables
papco_y_label, yst, yen, LOG=yscl(panel(0),3)

; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO.

extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
               ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

; now do the actual data plot, data only, no axis

;plot,time,yray,_extra=extra_plotPar,xstyle=5,ystyle=5

if (extra_plotPar.psym eq 0) then BEGIN
    plot,[0, 1], _extra=extra_plotPar,xstyle=5,ystyle=5, /nodata
    lanl_mpa_gap_oplot,time,yray,_extra=extra_plotPar
endif else begin  
    plot,time,yray,_extra=extra_plotPar,xstyle=5,ystyle=5
endelse

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label, panel, uytitle,/rot90

; store the coordinate information into common block for slices
plotted_x = !x & plotted_y = !y

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION

if not keyword_set(overplot) then begin
   right_side_label, panel, utitle,_extra=extra_Par
endif else begin
   IF dataType LE 1 THEN numret='!C!C!C' ELSE numret='!C!C'
   for i=0, PLOTS_ATTHISPOSITION-1 do utitle=numret+utitle
   right_side_label, panel, utitle,_extra=extra_Par
endelse

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
if KEYWORD_SET(OUTPUT) then begin
    print,'% p_lanl_mpa_line: Writing plot data out to file'
    description='mpa template sample data - single energy channel line'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
endif  
    
return
  
;jump to here if no data to plot 
noplot: 
  get_err_no = 10
  get_err_msg = 'No data to plot'
  papco_Draw_ErrorPanel, panel, 'lanl_mpa_line'+': ' + get_err_msg
  message, get_err_msg, /cont
  plotted_x = !x  &  plotted_y = !y
    
END 

;******************************************************************************
;* PROCEDURE:     
;*      lanl_mpa_gap_oplot, x, y, _EXTRA=e
;* 
;* DESCRIPTION:  
;*      This procedure plots data without connecting data point if they are
;*      further than 3 hours apart.
;*
;* INPUTS:       
;*      x,y     arrays to be plotted
;*              
;* OUTPUTS: 
;*      none
;* 
;* CALLING SEQUENCE:
;*      lanl_mpa_gap_oplot, x, y, _EXTRA=e
;*
;* MODIFICATION HISTORY:       
;*     written July 1997, Haje Korth
;*     modified August 1999, Haje Korth
;******************************************************************************
pro lanl_mpa_gap_oplot, xray, yray, _EXTRA=e

; check for psym=10 and disallow, doesn't work for plots 
if e.psym ne 0 then begin
    e.psym=0
    message,'works only for psym=0', /cont
end
    
; work out average spacing along time axis. If actual spacing encountered is
; more than three times the average, don't plot.
ndat=n_elements(xray)
diff_time=abs(xray(1:ndat-2) - xray(0:ndat-1))
av_int=total(diff_time)/(n_elements(diff_time))

; work out jumps in y. If a jump is larger than 5 times the average, don't plot
for i=0l,ndat-2 do begin
      int=abs(xray(i+1)-xray(i))
      if (int lt av_int*5) then begin
        plots,xray(i),yray(i) 
        plots,xray(i+1),yray(i+1),/continue,noclip=0,_EXTRA=e
      endif
 endfor

end
