;******************************************************************************
;* PROCEDURE:     
;*      p_lanl_mpa_pitch, panel, PlotInfo,  OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*            PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_lanl_mpa_pitch, panel, PlotInfo, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=subtable, _EXTRA=extra_par
;*
;* MODIFICATION HISTORY: 
;*        modified April 2003, Arne Aasnes at LANL
;*             allows data to be returned in the lanl_mpa common block
;*             without plotting the data when OUTPUT ge 2
;*        July 2003, A. Aa at LANL
;*             magnetic field direction is converted to SM coordinate system
;******************************************************************************
pro p_lanl_mpa_pitch, Panel,PlotInfo,   OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed for PAPCO:
  COMMON mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  COMMON yscale, yscl           ;man/auto yscaling
  COMMON zscale, zscl           ;man/auto zscaling
  COMMON shift_label, down      ;common for x-axis label shifting
  COMMON coordinateSystems, plotted_x, plotted_y  
                                ;info on coords used by mouse
  COMMON get_error, get_err_no, get_err_msg
  COMMON papco_color_names
  color_indx=[black, red, green, yellow, blue, magenta, cyan,white, $
            burgundy, olive, dark_green, teal, royal_blue, violet, dark_grey, grey]
  
; One of the common blocks should contain the plot data as returned by the
; data read procedure r_lanl_mpa. 
  COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data, mpa_pitch_sat
  COMMON lanl_mpa               ;see lanl_mpa_set_commons for contents
  COMMON lanl_mpa_pitch_slice,  utitle, uytitle, uztitle
  COMMON lanl_mpa_pitch_output,  xarr, yarr, zmat
  
  
  if keyword_set(OUTPUT) then output=OUTPUT else output=0
  IF n_elements(extra_par) EQ 0 THEN extra_par = {no_tag:0}
  IF output LT 1 THEN BEGIN 
  
     papco_Set_SubTable, SUBTABLE ;sets the papco color table index
     panelset,panel             ;sets the panel position viewport
  ENDIF 

  control = *plotinfo.USR_PTR1 

; restrict data to actual time range requested - this makes for faster zooming
; mpa_pitch data is in TAI. For plotting, always start at zero to get max
; resolution (plot works in float only, large values get rounded! 
  tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
  tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

  IF control.pl_type EQ 4 THEN blim = 4 ELSE blim = 5
  index=where((mpa_pitch_data.tai ge tai_1) AND (mpa_pitch_data.tai le tai_2) $
              AND mpa_pitch_data.b_flag LT blim $
              ,c)
  if c GT 2 then data=mpa_pitch_data(index) ELSE GOTO, noplot
  xut1=0  &  xut2=tai_2-tai_1  & time=data.tai-tai_1 &  time(0) = 0
  ndat=n_elements(data)

  dim = size(data.e_unbinned, /dimensions)

  en_channels = dim(0)
  pol_det = dim(1) ; 6 polar detectors
  az_det = dim(2)  ; 24 azimuthal samples (15 deg)

  nodata = -1e31
  

  
  
;if plot is pitch angles vs time set up yarr, if energy spectrum for
;one pitch angle, 
;assign energy channels. Read from file!

  
  energies = r_lanl_mpa_channels(control.sc_id, control.species)

  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Removal of bad data and calibration 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



IF control.pl_type NE 7 THEN BEGIN  ;no need to calibrate if plotting magnetic field
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Remove "fake" data from detectors that don't work
;; Data was put in the data files to calculate moments (for details,
;; ask M. Thomsen)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   start_filter_el = 26
   IF sat_names(control.sc_id) EQ 'LANL-01A' THEN BEGIN 
      data.e_unbinned(*, 1, *, *) = nodata
     data.e_mag(*, 1, *, *) = nodata
     data.p_unbinned(*, 1, *, *) = nodata
     data.p_mag(*, 1, *, *) = nodata
  endIF 
  IF sat_names(control.sc_id) EQ '1994-084' THEN BEGIN 
     data.e_unbinned(*, 3, *, *) = nodata
     data.e_mag(*, 3, *, *) = nodata
     data.p_unbinned(*, 3, *, *) = nodata
     data.p_mag(*, 3, *, *) = nodata
     start_filter_el = 0 ; some glitch in 1994-084 sweep routine leads to some azimuthal sweeps being of wrong energy
  endIF 
  IF sat_names(control.sc_id) EQ 'LANL-97A' THEN BEGIN 
     data.e_unbinned(*, 5, *, *) = nodata
     data.e_mag(*, 5, *, *) = nodata
     data.p_unbinned(*, 5, *, *) = nodata
     data.p_mag(*, 5, *, *) = nodata
  ENDIF 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Lower electron channels are contaminated by photo electrons
;; Try to remove these by identifying azimuthal sectors with abnormal
;; flux rates and remove these
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;control.rem_ph = 1
IF control.rem_ph EQ 1  AND control.species EQ 0 THEN BEGIN
   message, 'Filtering for photo electrons', /cont
   fact_lim = 1.75 ;randomly chosen number
   nodat_idx = where(data.e_unbinned EQ nodata, nodat_cnt)

   if nodat_cnt GT 0 THEN data.e_unbinned(nodat_idx) = !values.f_nan ; need nan values to be excluded from mean

   print, string(nodat_cnt)+' invalid points before filtering'
   az_mean = fltarr(az_det)
   pol_mean = fltarr(pol_det)
   FOR h = start_filter_el, 39 DO BEGIN 
      az_mean(*) = 0
      pol_mean(*) = 0
      FOR i = 0, ndat-1 DO BEGIN
         
         FOR j = 0, az_det-1 DO az_mean(j) = mean(data(i).e_unbinned(h, *, j), /nan)
         az_mean_sort = az_mean(sort(az_mean))
         low_mean = mean(az_mean_sort(0:14))
         bad_sectors = where(az_mean GT fact_lim*low_mean, bad)
         IF bad GT 0 THEN  FOR bb = 0, bad-1 DO  data(i).e_unbinned(h, *, bad_sectors(bb)) = !values.f_nan
         IF bad EQ 0 THEN BEGIN 
            
            FOR j = 0, pol_det-1 DO pol_mean(j) = mean(data(i).e_unbinned(h, j, *), /nan)
            pol_mean_sort = pol_mean(sort(pol_mean))
            low_mean = mean(pol_mean_sort(0:4))
            bad_sectors = where(pol_mean GT fact_lim*low_mean, bad)
            IF bad GT 0 THEN  FOR bb = 0, bad-1 DO  data(i).e_unbinned(h, bad_sectors(bb), *) = !values.f_nan
         ENDIF 
      ENDFOR  
   ENDFOR 
   nodat_idx = where(NOT finite(data.e_unbinned), nodat_cnt)
   print, string(nodat_cnt)+' invalid points after filtering'

ENDIF 
no_rem_photo:

ENDIF ; end of calibration stuff, not necessary if plotting magnetic field
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; defines the y axis values and title for the different plot options 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  utitle = sat_names(control.sc_id)
  CASE control.species OF
     0: BEGIN
        all_zmat_st = reform(data.e_unbinned, en_channels,  pol_det*az_det, ndat)
        alpha_st = reform(data.e_mag, en_channels, pol_det*az_det, ndat)
        utitle=utitle+' Electrons'

     END
     1: BEGIN
        all_zmat_st = reform(data.p_unbinned, en_channels, pol_det*az_det, ndat)
        alpha_st = reform(data.p_mag, en_channels, pol_det*az_det, ndat)
        utitle=utitle+' Ions'

     END
  ENDCASE
  
;    call function to return an array of pa bins centered around
;     90 degrees with requested resolution
  pa_rng = papco_pitch_bin(fix(control.pa_res), pa_idx= pa_idx) 
  nbins = n_elements(pa_rng(*, 0))

  all_zmat_st(where(all_zmat_st LE 0)) = !values.f_nan
  


; routine to use model Blocal and Bmin to convert from local pitch
; angles to equatorial pitch angles. Only do this if requested and not
; plotting magnetic field direction
  COMMON lanl_geo_eph, ephsat_names, eph_names,  ephyunits, lanl_geo_eph_control
  COMMON geo_eph_data, eph_header, eph_data 
  COMMON lanl_geo_eph_tick, ephtime, yray, mag_model, tick_plotinfo ;yray stores (hopefully) the Blocal/Bmin value





  CASE control.pl_type OF 
     0: BEGIN                   ; all PA @ 1 E
        yarr = pa_rng
        energy_title = strcompress(energies(control.ek1, 0), /remove_all)+ $
                    '-'+ strcompress(energies(control.ek1, 1), /remove_all)+' eV'
        en_ind = en_channels-1-control.ek1
        zmat = fltarr(nbins, ndat)
        FOR i = 0, ndat-1 DO BEGIN 
           FOR j = 0, nbins-1 DO BEGIN 
              pa_ind = where(alpha_st(en_ind, *, i) GE pa_rng(j, 0) AND alpha_st(en_ind, *, i) LT pa_rng(j, 1), pa_cnt)
              IF pa_cnt GT 0 THEN zmat(j, i) = mean(all_zmat_st(en_ind, pa_ind, i), /nan) ELSE zmat(j, i) = !values.f_nan
           endFOR 
        ENDFOR
;        IF inv_cnt GT 0 THEN FOR i = 0, inv_cnt-1 DO zmat(*, inval(i)) = mean(zmat(*, inval(i)), /NAN)
        uytitle = 'Pitch angle (deg)'
        uztitle = 'Energy Flux (cm!E-2!Ns!E-1!Nsr!E-1!N)'
        utitle = utitle+'!C'+energy_title
        yst=yarr(0,0)*0.99 & yen=yarr(nbins-1,1)+1.01
        yscl(panel(0), 3) = 0 ; overrides choice from panel, always use linear scale
                
     END 
     1: BEGIN ; all E @ 1 PA
        yarr = energies
        pitch_title = strmid(strcompress(control.pa_bin1-0.5*control.pa_res, /remove_all), 0, 3)+ $
                    '-'+ strmid(strcompress(control.pa_bin1+0.5*control.pa_res, /remove_all), 0, 3)+' deg'
;        en_ind = indgen(en_channels)
        zmat = fltarr(en_channels, ndat)
        FOR i = 0, ndat-1 DO BEGIN 
           FOR j = 0, en_channels-1 DO BEGIN 
              pa_ind = where(alpha_st(en_channels-1-j, *, i) GE control.pa_bin1-0.5*control.pa_res AND $
              alpha_st(en_channels-1-j, *, i) LT control.pa_bin1+0.5*control.pa_res, pa_cnt)
              IF pa_cnt GT 0 THEN zmat(j, i) = mean(all_zmat_st(en_channels-1-j, pa_ind, i),/nan) $    
              ELSE zmat(j, i) = !values.f_nan
           endFOR 
        ENDFOR 
        uytitle = 'Energy (eV)'
        uztitle='Energy Flux (cm!E-2!Ns!E-1!Nsr!E-1!N)'
        utitle = utitle + '!C'+pitch_title
 
        IF control.ek1 NE 0 THEN BEGIN 
           zmat = zmat(control.ek1:en_channels-1, *)
           yarr = yarr(control.ek1:en_channels-1, *)
        ENDIF  
        yst=yarr(0,0)*0.99 & yen=yarr(en_channels-1-control.ek1,1)+1.01
        yscl(panel(0), 3) = 1   ; overrides choice from panel, always use linear scales
     END 

     2:BEGIN ;all PA @ E range
        yarr = pa_rng
        energy_title = strcompress(energies(control.ek1, 0), /remove_all)+ $
                    '-'+ strcompress(energies(control.ek2, 1), /remove_all)+' eV'
        control.ek2 = control.ek2 >  control.ek1+1
        en_ind = en_channels-1-indgen(control.ek2-control.ek1)-control.ek1
        en_chan = n_elements(en_ind)
        zmat = fltarr(en_chan, nbins, ndat)
;        all_zmat_st = reform(rebin(all_zmat_st(en_ind, *, *),1, pol_det*az_det, ndat), pol_det*az_det, ndat)
        FOR i = 0, ndat-1 DO BEGIN 
           FOR j = 0, nbins-1 DO BEGIN 
              FOR k = 0, en_chan-1 DO BEGIN 
                 pa_ind = where(alpha_st(en_ind(k), *, i) GE pa_rng(j, 0) AND alpha_st(en_ind(k), *, i) LT pa_rng(j, 1), pa_cnt)
                 IF pa_cnt GT 0 THEN zmat(k, j, i) = mean(all_zmat_st(en_ind(k), pa_ind, i), /nan) ELSE $
                     zmat(k, j, i) = !values.f_nan
              ENDFOR 
           ENDFOR  
        ENDFOR
        zmat = reform(rebin(zmat, 1, nbins, ndat), nbins, ndat) ; taking the mean over the energy range

        uytitle = 'Pitch angle (deg)'
        uztitle = 'Energy Flux (cm!E-2!Ns!E-1!Nsr!E-1!N)'
        utitle = utitle+'!C'+energy_title
        yst=yarr(0,0)*0.99 & yen=yarr(nbins-1,1)+1.01
        yscl(panel(0), 3) = 0 ; overrides choice from panel, always use linear scale
     END
     3:BEGIN ; all E @ PA range
        IF control.pa_bin1 GT control.pa_bin2 THEN BEGIN 
           buff_bin = control.pa_bin2 
           control.pa_bin2 = control.pa_bin1
           control.pa_bin1 = buff_bin
        endIF 
        yarr = energies
        pitch_title = strmid(strcompress(control.pa_bin1, /remove_all), 0, 3)+ $
                    '-'+ strmid(strcompress(control.pa_bin2, /remove_all), 0, 3)+' deg'
        control.pa_bin2 = control.pa_bin2 >  control.pa_bin1 +control.pa_res
;        en_ind = indgen(en_channels)
        zmat = fltarr(en_channels, ndat)
        
        FOR i = 0, ndat-1 DO BEGIN 
           FOR j = 0, en_channels-1 DO BEGIN 
              pa_ind = where(alpha_st(en_channels-1-j, *, i) GE control.pa_bin1 AND $
              alpha_st(en_channels-1-j, *, i) LT control.pa_bin2, pa_cnt)
              IF pa_cnt GT 0 THEN zmat(j, i) = mean(all_zmat_st(en_channels-1-j, pa_ind, i),/nan)     $
              ELSE zmat(j, i) = nodata
           ENDFOR  
        ENDFOR 
        uytitle = 'Energy (eV)'
        uztitle='Energy Flux (cm!E-2!Ns!E-1!Nsr!E-1!N)'
        utitle = utitle + '!C'+pitch_title
        IF control.ek1 NE 0 THEN BEGIN 
           zmat = zmat(control.ek1:en_channels-1, *)
           yarr = yarr(control.ek1:en_channels-1, *)
        ENDIF  
        yst=yarr(0,0)*0.99 & yen=yarr(en_channels-1-control.ek1,1)+1.01
        yscl(panel(0), 3) = 1   ; overrides choice from panel, always use log scale

     END  
     4:BEGIN ; anisotropy spectrogram
        yarr = energies 
        pitch_title = strcompress(long(control.pa_bin2), /remove_all)+' deg'+ $
                    '/'+ strcompress(long(control.pa_bin1), /remove_all)+' deg'
        zmat = fltarr(en_channels, ndat)        
        FOR i = 0, ndat-1 DO BEGIN 
           FOR j = 0, en_channels-1 DO BEGIN 
              IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN $
              pa_low = where( $
              (alpha_st(en_channels-1-j, *, i) GE control.pa_bin1-control.pa_res/2. AND $
              alpha_st(en_channels-1-j, *, i) LT control.pa_bin1+control.pa_res/2.) OR $
              (alpha_st(en_channels-1-j, *, i) GE 180-control.pa_bin1-control.pa_res/2. AND $
              alpha_st(en_channels-1-j, *, i) LT 180-control.pa_bin1+control.pa_res/2.), $
              pa_low_cnt) $
              ELSE pa_low = where(alpha_st(en_channels-1-j, *, i) GT control.pa_bin1-control.pa_res/2. AND $
              alpha_st(en_channels-1-j, *, i) LT control.pa_bin1+control.pa_res/2., pa_low_cnt)
              pa_high = where(alpha_st(en_channels-1-j, *, i) GE control.pa_bin2-control.pa_res/2. AND $
              alpha_st(en_channels-1-j, *, i) LT control.pa_bin2+control.pa_res/2., pa_high_cnt)
              IF pa_low_cnt GT 0 AND pa_high_cnt GT 0 THEN $
                 zmat(j, i) = (mean(all_zmat_st(en_channels-1-j, pa_high, i),/nan))/ $
                              (mean(all_zmat_st(en_channels-1-j, pa_low, i),/nan))   $
              ELSE zmat(j, i) = nodata
           ENDFOR  
        ENDFOR 
        uytitle = 'Energy (eV)'
        IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN   uztitle = 'Anisotropy' ELSE $
         uztitle = 'Streaming'
        utitle = utitle + '!C'+pitch_title
        IF control.ek1 NE 0 THEN BEGIN 
           zmat = zmat(control.ek1:en_channels-1, *)
           yarr = yarr(control.ek1:en_channels-1, *)
        ENDIF  
        yst=yarr(0,0)*0.99 & yen=yarr(en_channels-1-control.ek1,1)+1.01
        yscl(panel(0), 3) = 01; overrides choice from panel, always use log scale
     END  
     5:BEGIN ; E range @ PA range (line plot)
        IF control.pa_bin1 GT control.pa_bin2 THEN BEGIN 
           buff_bin = control.pa_bin2
           control.pa_bin2 = control.pa_bin1
           control.pa_bin1 = buff_bin
        ENDIF
        control.ek2 = (en_channels-1) < control.ek2 >  (control.ek1) 
        en_pa_title = strcompress(energies(control.ek1, 0), /remove_all)+ $
                    '-'+ strcompress(energies(control.ek2, 1), /remove_all)+' eV' + $
                    '!C'+strmid(strcompress(control.pa_bin1, /remove_all), 0, 3)+ $
                    '-'+ strmid(strcompress(control.pa_bin2, /remove_all), 0, 3)+' deg'
       
        IF control.ek2 gt control.ek1 THEN en_ind =en_channels-1-indgen(control.ek2-control.ek1)-control.ek1 $
        ELSE en_ind = en_channels-1-control.ek1
        IF control.species EQ 1 THEN en_ind = en_channels-1-en_ind
        
        en_chan = n_elements(en_ind)
        zmat = fltarr(en_chan, ndat)
        FOR i = 0, ndat-1 DO BEGIN 
           FOR j = 0, en_chan-1 DO BEGIN 
              pa_ind = where(alpha_st(en_ind(j), *, i) GE control.pa_bin1 AND $
              alpha_st(en_ind(j), *, i) LT control.pa_bin2, pa_cnt)
              IF pa_cnt GT 0 THEN zmat(j, i) = mean(all_zmat_st(en_ind(j), pa_ind, i),/nan)    $ 
              ELSE zmat(j, i) = !values.f_nan
           ENDFOR   
        endFOR 
        zmat = reform(rebin(zmat, 1, ndat), ndat)
        uytitle = 'Energy Flux (cm!E-2!Ns!E-1!Nsr!E-1!N)'
        utitle = utitle+'!C'+en_pa_title
        uztitle = ''
     END 
     6:BEGIN                    ; anisotropy line plot
        en_pa_title = strcompress(energies(control.ek1, 0), /remove_all)+ $
                    '-'+ strcompress(energies(control.ek1, 1), /remove_all)+' eV' + $
                    '!C'+strcompress(long(control.pa_bin2), /remove_all)+' deg'+ $
                    '/'+ strcompress(long(control.pa_bin1), /remove_all)+' deg'
        ek = en_channels-1-control.ek1
        IF control.species EQ 1 THEN ek = en_channels-1-ek
        zmat = fltarr(ndat)
        FOR i = 0, ndat-1 DO BEGIN
           th_zmat = all_zmat_st(ek, *, i)
           th_alpha = alpha_st(ek, *, i)
           IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN $
                 pa_low = where((th_alpha GT control.pa_bin1-control.pa_res/2. AND $
                 th_alpha LT control.pa_bin1+control.pa_res/2.) OR $
                 (th_alpha GT 180-control.pa_bin1-control.pa_res/2. AND $
                 th_alpha LT 180-control.pa_bin1+control.pa_res/2.), pa_low_cnt)$
          ELSE pa_low = where(th_alpha GT control.pa_bin1-control.pa_res/2. AND $
                 th_alpha LT control.pa_bin1+control.pa_res/2., pa_low_cnt)

           pa_high = where(th_alpha GT control.pa_bin2-control.pa_res/2. AND $
                     th_alpha LT control.pa_bin2+control.pa_res/2., pa_high_cnt)
           IF pa_low_cnt GT 0 AND pa_high_cnt GT 0 THEN zmat(i) = mean(th_zmat(pa_high))/mean(th_zmat(pa_low)) $
           ELSE zmat(i) = nodata
        endFOR 
        uztitle = ''
        IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN   uytitle = 'Anisotropy' $
        ELSE uytitle = 'Streaming'
        utitle = utitle+'!C'+en_pa_title
        
        isdata = where(zmat ne nodata,good_cnt)
        IF good_cnt GT 0 THEN BEGIN
           zmat = zmat(isdata)
           time = time(isdata)
        ENDIF 
     END  
     7:BEGIN  ; magnetic field direction
  ; prepare a model field elevation (in GEO coordinates) to compare to the derived direction
        IF 0 THEN BEGIN
           sat_id =sat_names(control.sc_id)
           org_yscl = yscl
           a_lanl_geo_eph, t1 = mjdt_start, t2 = mjdt_end, /mjdt, sat_id = sat_id, extmod = control.external_model, mag_dyn = control.mag_dyn
           yscl = org_yscl
           
           eph_control = lanl_geo_eph_control      
           mag_model = 'MODEL_'+varprt(control.internal_model)+'_'+$
                             varprt(control.external_model)
           names = papco_make_choice_names(eph_control.mag_dyn_info)
           stdyn = strupcase(strmid(names(control.mag_dyn), 0, 4))
           mag_model = mag_model+'_'+stdyn
           tags = TAG_NAMES(eph_data(0))
           mag_idx = where(tags EQ mag_model, c)
           
           
; x,y and z components in GEO coordinates
           b1 = eph_data.(mag_idx(0)).(4)
           b2 = eph_data.(mag_idx(0)).(5)
           b3 = eph_data.(mag_idx(0)).(6)
           
           ephdat = n_elements(b1)
           elarr = fltarr(ephdat)
           phiarr = fltarr(ephdat)
           
           convert_t90_to_date, mjdt_start.t90, year, day_oy, hour, minute, second
           daysec = hour*3600.+minute*60.+second
           IF n_elements(lib_name) EQ 0 THEN lib_name = getenv('PAPCO_LIB')+'/onera/onera_desp_lib.so'
           iyr = long(year) &  outvec = dblarr(3)
           FOR i = 0, ephdat-1 DO BEGIN 
              vect = [b1(i), b2(i), b3(i)]
              daysecin = daysec+eph_data(i).tai-eph_data(0).tai 
              IF daysecin GT 86400.0 THEN BEGIN 
                 daysecin = daysecin-86400.0 &  day_oy = day_oy+1
              ENDIF
              
;        idoy = long(day_oy) &  secs = long(daysecin) & invec = double(vect) 
;        result = call_external(lib_name, 'geo2sm_', iyr,idoy,secs,invec,outvec, /f_value)
;        vect = cv_coord(from_rect = outvec, /to_sphere, /degree)
;        lanl_mpa_pitch_coordtrans, vect, from_cs = 'GEO', to_cs =
;        'SM', /sphereout, iyr= year, idoy = day_oy, secs = daysecin
              vect = cv_coord(from_rect = vect, /to_sphere, /degree)
              elarr(i) = 90.0+vect(1)
              phiarr(i) = vect(0)
           ENDFOR
;     elarr = elarr*!radeg
        ENDIF

 ; calculate the equatorial pitch
 ; angles, using the model values of local and equatorial magnetic
 ; field strength        
        IF 0 THEN BEGIN         ;control.equa_pitch and control.pl_type ne 7
           ;; get the model Blocal and Bmin for each MPA data time
           org_yscl = yscl
           a_lanl_geo_eph, t1 = mjdt_start, t2 = mjdt_end, /mjdt, extmod = control.external_model, mag_dyn = control.mag_dyn
           yscl = org_yscl

           ; get the B field strength ratio onto MPA time frame
           Brat = papco_interpol(yray, eph.data.tai, data.tai, NODATA = nodata)
           ;resize the ratio array to dimensions of alpha for fast calculations
           Brat = rebin(reform(brat, 1, 1, ndat), en_channels, pol_det*az_det, ndat)
           ;;
           ;;
           o90 = where(alpha_st GT 90.0, cnt)
           alpha_st = !radeg*acos(1-2*sin(alpha_st*!dtor)^2*Brat)/2.0
           IF cnt GT 0 THEN alpha_st(o90) = 180-alpha_st(o90)
           
        ENDIF
        
        utitle = sat_names(control.sc_id)+'!CSM coordinates'; neither electron nor proton
        isdata = where(data.mag_dir(control.mf_comp) GT nodata/3, good_cnt)
        IF good_cnt GT 0 THEN BEGIN
                      COMMON lanl_geo_data, geo_header, geo_data
           th_x = !x &  th_y = !y &  th_p = !p
           org_yscl = yscl
           a_lanl_geo, T1 = mjdt_start, T2 = mjdt_end, sat_id = sat_names(control.sc_id), /mjdt, pl_type = 0
           yscl = org_yscl
           IF get_err_no NE 0 THEN GOTO, noplot
           !x = th_x & !y = th_y &  !p = th_p
           magdir = data(isdata).mag_dir
;           convert magnetic field direction to elevation angle and
;           phi in GSM coordinates
           srad = papco_interpol(geo_data.radius, geo_data.tai, data(isdata).tai)
           slong = papco_interpol(geo_data.longitude, geo_data.tai, data(isdata).tai)
           slati = papco_interpol(geo_data.latitude, geo_data.tai, data(isdata).tai)
           thetaphi = data(isdata).mag_dir
           zmat = fltarr(good_cnt)
           time = time(isdata)

           convert_t90_to_date, mjdt_start.t90, year, day_oy, hour, minute, second
           daysec = hour*3600.+minute*60.+second
           forward_FUNCTION lat2elev,band_pass
           FOR mdc = 0, good_cnt-1 DO BEGIN
              invec = [thetaphi(1, mdc), thetaphi(0, mdc), 1]
              daysecin = daysec+time(mdc) 
              IF daysecin GT 86400.0 THEN BEGIN 
                 daysecin = daysecin-86400.0 &  day_oy = day_oy+1
              ENDIF
              outvec = invec
              lanl_mpa_pitch_coordtrans, outvec,to_CS='SM',s_c = [slong(mdc), slati(mdc), srad(mdc)], /spherein, /sphereout, iyr = year, idoy = day_oy, secs = daysecin
              IF control.mf_comp EQ 0 THEN zmat(mdc) =90+!radeg*outvec(1) ELSE zmat(mdc) = outvec(0)*!radeg

           ENDFOR
           IF 0 THEN zmat = data(isdata).mag_dir(control.mf_comp)*!radeg
           IF 0 THEN BEGIN ; if control.mag_filt??

              zmat = leefilt(zmat)
           ENDIF 
;          zmat = data(isdata).mag_dir(control.mf_comp)*!radeg 
;          time = time(isdata)
           IF n_elements(elarr) NE 0 THEN elarr = papco_interpol(elarr, eph_data.tai, data(isdata).tai)
        ENDIF 
        names = papco_make_choice_names(control.mf_comp_info)
        uytitle = names(control.mf_comp)+'(deg)'
        uztitle = ''
        yscl(panel(0), 3) = 0   ; overrides choice from panel, always use linear scale
     END      

        
  ENDCASE 
; returning nodata as the value for missing data

  val_nan = where(finite(zmat) EQ 0, nr_vn)
  IF nr_vn GT 0 THEN zmat(val_nan) = nodata
  IF max(zmat) EQ nodata THEN GOTO, noplot

 
;check if any filtering or PSD is needed - get ephemeris data first -----------
IF control.L_SELECT_RANGE OR control.L_STAR_SELECT_RANGE OR $
   control.MLT_SELECT_RANGE OR control.MLAT_SELECT_RANGE $
; OR   control.cal EQ 2 $
   THEN BEGIN

;   IF control.rem_ph EQ 0 THEN BEGIN  ; if rem_ph is set the ephemeris data is already read
 
      COMMON lanl_geo_eph, vv1, vv2, vv3, lanl_geo_eph_control
      COMMON geo_eph_data, eph_header, eph_data
      
      eph_plotinfo = papco_getplotinfostruct()
      eph_control = lanl_geo_eph_control
      eph_control.SC_ID = control.SC_ID
      eph_control.ORIG = 1      ;mag model coords
      eph_control.L_CALC = control.L_CALC
      eph_control.EXTERNAL_MODEL = control.EXTERNAL_MODEL
      eph_control.INTERNAL_MODEL = control.INTERNAL_MODEL
      eph_plotinfo.USR_PTR1 = ptr_new(eph_control)
      
      r_geo_eph, eph_plotinfo, VERBOSE=VERBOSE
;   endIF 
    IF get_err_no NE 0 THEN BEGIN 
       message,'problem getting ephemeris data, cannot filter', /cont
       goto, no_restrict
    ENDIF 
    tag_idx = where(TAG_NAMES(eph_data) EQ eph_header.model_name)
    tag_idx = tag_idx(0)

    filt_t = eph_data.TAI-tai_1
    filt_lbl = ''

ENDIF

;filter the data using the ephemeris info -------------------------------------
;Handle different time bases - set non-filtered data to nodata flag.

;filter for L
IF control.L_SELECT_RANGE THEN BEGIN
       
    filt_var = eph_data.(tag_idx).L
    v1 = control.L_SELECT_FROM_VAL
    v2 = control.L_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.L_SELECT_RANGE_FRMT)+'!C'

ENDIF

;filter for L_STAR
IF control.L_STAR_SELECT_RANGE THEN BEGIN
       
    filt_var = eph_data.(tag_idx).LSTAR
    v1 = control.L_STAR_SELECT_FROM_VAL
    v2 = control.L_STAR_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.L_STAR_SELECT_RANGE_FRMT)+'!C'

ENDIF

;filter for MLT
IF control.MLT_SELECT_RANGE THEN BEGIN

    filt_var = eph_data.(tag_idx).MLT
    v1 = control.MLT_SELECT_FROM_VAL
    v2 = control.MLT_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)

    ;convert mlt to x,y, convert back. Handles midnight crossing.
    x=cos(filt_var*!pi/12)
    y=sin(filt_var*!pi/12)
    n_mlt=atan(y/x)

    idx=where((x lt 0) and y gt 0, c) ;2nd quadrant
    if c ne 0 then n_mlt(idx)=n_mlt(idx)+!pi
    idx=where((x lt 0) and y lt 0, c) ;3nd quadrant
    if c ne 0 then n_mlt(idx)=n_mlt(idx)+!pi
    idx=where((x gt 0) and y lt 0, c) ;4nd quadrant
    if c ne 0 then n_mlt(idx)=2*!pi+n_mlt(idx)
    n_mlt=n_mlt*12/!pi

    ;now select out range needed
    if v1 gt v2 then begin ;crosses over midnight
        idx=where( (n_mlt lt v1) AND (n_mlt gt v2), c)
    endif else begin
        idx=where( (n_mlt lt v1) OR (n_mlt gt v2), c)      
    ENDELSE 

    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.MLT_SELECT_RANGE_FRMT)+'!C'
ENDIF

;filter for MLAT
IF control.MLAT_SELECT_RANGE THEN BEGIN
       
    filt_var = eph_data.(tag_idx).MLAT
    v1 = control.MLAT_SELECT_FROM_VAL
    v2 = control.MLAT_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.MLAT_SELECT_RANGE_FRMT)+'!C'

ENDIF
no_restrict:    


; build time array for plotting
timeDim=N_ELEMENTS(time)
xarr=DBLARR(timeDim-1,2)
xarr(*,0)=time(0:timeDim-2)
xarr(*,1)=time(1:timeDim-1)
timeDiff=DBLARR(timeDim-1)
FOR i=0,N_ELEMENTS(timeDiff)-1 DO timeDiff(i)=time(i+1)-time(i)
index=WHERE(timeDiff GT 171.0,count)
IF (count GT 0) THEN FOR i=0,count-1 DO xarr(index,1)=xarr(index,0)+86.0


IF output GE 1 THEN BEGIN
    message,'plot data constructed',/cont    
    IF output EQ 2 THEN return
ENDIF 




; the variables used above have the following meaning:
;       uztitle   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis)
; scaling

IF control.pl_type LT 5 THEN BEGIN  ;spectrum plot

   if (zscl(panel(0),0) eq 1) then begin
      zmin=zscl(panel(0),1) & zmax=zscl(panel(0),2)
   endif else BEGIN
      papco_autorange, zmat, zmin, zmax, LOG = zscl(panel(0),3), $
      NODATA = nodata, exclude=0
      zscl(panel(0),1)=zmin & zscl(panel(0),2)=zmax
   ENDELSE
   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
   endif else begin
      yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
   ENDELSE 

ENDIF ELSE BEGIN  ; line plot
   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
   endif else BEGIN
      papco_autorange, zmat, yst, yen, LOG = yscl(panel(0),3), $
      NODATA = nodata, exclude=0
      yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
   ENDELSE
   zmin = 0 &  zmax = 0 ; not used for line plot routine, but "must" be defined !
ENDELSE 




; use auto y-labeling routine to get "good" y-lables
papco_y_label, yst, yen, LOG=yscl(panel(0),3)
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO.
extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), ztitle:uztitle, $
               xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set locally and the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; check if we need to do plot or just return arrays to slice.
IF output EQ 3 THEN  GOTO, write_output

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!

IF control.pl_type GE 5 THEN BEGIN  ; line plot
   plot, [0, 1], [0, 1], _extra=extra_plotPar, ystyle = 5, xstyle = 5, /nodata
;   papco_gap_plot, time, zmat, nodata, xstyle=5, ystyle=5, _extra=extra_plotPar

;   plot, [yst, yen], _extra=extra_plotPar,xstyle=5,ystyle=5, /nodata
   lanl_mpa_gap_oplot, time, zmat,  xstyle = 5, ystyle = 5, _extra = extra_plotpar, nodata = nodata
   IF n_elements(elarr) NE 0 THEN lanl_mpa_gap_oplot, time, elarr,  xstyle = 5, ystyle = 5, _extra = extra_plotpar, nodata = nodata
;        linestyle:plotinfo.linestyle,
;        thick:plotinfo.thick,symsize:plotinfo.symsize}

; down sets the axis labeling level
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin  
    down=down+1    
 eNDIF
; plot y-axis label at left of plot. Use scalable routine!  
   left_side_label, panel, uytitle,/rot90, _extra = extra_plotPar

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
  if not keyword_set(overplot) then begin
    IF control.pl_type EQ 7 THEN right_side_label,panel,utitle,/rot90, _extra=extra_Par ELSE right_side_label,panel,utitle,_extra=extra_Par
  endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C!C!C'+utitle
    IF control.pl_type EQ 7 THEN right_side_label,panel,utitle,/rot90, _extra=extra_Par ELSE right_side_label,panel,utitle,_extra=extra_Par
 eNDELSE
ENDIF  ELSE BEGIN    
   zmat = rotate(zmat, 4)
   IF yscl(panel(0),3) EQ 1 THEN BEGIN ; if logaritmic yaxis
     resx=10 & resy=0.01
      papco_plot_colorspec,zmat,xarr,yarr,nodata=nodata,$
      xstyle=5, ystyle=5, _extra=extra_plotPar, /VERBOSE,resx=resx, resy=resy
   ENDIF ELSE BEGIN
      resx = 10 &  resy = 0.1
      papco_plot_colorspec,zmat,xarr,yarr,nodata=nodata,$
      xstyle=5, ystyle=5, _extra=extra_plotPar, /VERBOSE, resx=resx, resy=resy,dy=0.5
   ENDELSE
  
   papco_color_bar, _extra=extra_plotPar

; plot y-axis label at left of plot. Use scalable routine!  
   left_side_label, panel, uytitle,/rot90, _extra = extra_plotPar

; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

   right_side_label,panel,utitle,/rot90, _extra = extra_plotPar

ENDELSE 

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this routine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar     


   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
write_output:
if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='lanl_mpa_pitch data'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztitle
    rs_label=utitle
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
endif  
   
return
  
;jump to here if no data to plot 

noplot: 
message,'No data to plot',/cont
plotted_x = !x  &  plotted_y = !y  
get_err_no = 1 &  get_err_msg = 'No data available!'
END
