;******************************************************************************
;* PROCEDURE:     
;*      p_lanl_mpa_spec, panel, bird, dataType, option, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*            PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_lanl_mpa, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY: 
;*        modified April 2003, Arne Aasnes at LANL
;*             allows data to be returned in the lanl_mpa common block
;*             without plotting the data when OUTPUT ge 2
;******************************************************************************
pro p_lanl_mpa_spec, panel, bird, dataType, option, scPot, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed for PAPCO:
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_lanl_mpa. 
COMMON lanl_mpa_data, input_header, input_data, input_sat
COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents
COMMON lanl_mpa_slice,  utitle
COMMON lanl_mpa_channels, yarr, zmat

if keyword_set(OUTPUT) then output=OUTPUT else output=0

papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
panelset,panel                       ;sets the panel position viewport

n_channels=40

;assign energy channels. Read from file!
yarr = r_lanl_mpa_channels(bird, dataType)

; restrict data to actual time range requested - this makes for faster zooming
; input data is in TAI. For plotting, always start at zero to get max
; max resolution (plot works in float only, large values get rounded! 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

; restrict data in time  
index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c GT 2 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1  & time=data.tai-tai_1

ndat=n_elements(data)  

zmat=FLTARR(40, ndat)
IF dataType EQ 0 THEN zmat=data.ave_flux_e ELSE zmat=data.ave_flux_i
index=where((zmat GE 0.0) AND (zmat LE 1.0), count)

utitle = sat_names(bird)+'!C'

CASE dataType OF
    0: BEGIN
          utitle=utitle+'Electrons'
          factor=8.09e-32 
    END
    1: BEGIN
          utitle=utitle+'Ions'
          factor=2.72e-25
    END
ENDCASE

uytitle='Energy (eV)'
 
CASE option OF
     0:BEGIN
        IF (dataType EQ 0) THEN $
          zmat=zmat*ABS(data.ece4) $
        ELSE $
          zmat=zmat*ABS(data.ecp4)
        uztit='Energy Flux (cm!E-2!Ns!E-1!Nsr!E-1!N)'
       END
     1:BEGIN
        uztit='Diff. Flux (cm!E-2!Ns!E-1!Nsr!E-1!NeV!E-1!N)'
       END
     2:BEGIN
        IF (dataType EQ 0) THEN $
          zmat=zmat/ABS(data.ece4)*factor $
        ELSE $
          zmat=zmat/ABS(data.ecp4)*factor
        uztit='Phase Space Density (cm!E-6!Ns!E3!N)'
       END
ENDCASE

IF (count NE 0) THEN zmat(index)=-1
zmat=reverse(zmat) & zmat=transpose(zmat)

; build array to correct S/C potential if requested
if (scPot eq 1) then begin
     scPot_arr=data.sc_poten
     scPot_arr=ABS(scPot_arr)
ENDIF

IF output GE  2 THEN BEGIN
    message,'plot data constructed',/cont
    return
ENDIF 

; build time array for plotting
timeDim=N_ELEMENTS(time)
xarr=DBLARR(timeDim-1,2)
xarr(*,0)=time(0:timeDim-2)
xarr(*,1)=time(1:timeDim-1)
timeDiff=DBLARR(timeDim-1)
FOR i=0,N_ELEMENTS(timeDiff)-1 DO timeDiff(i)=time(i+1)-time(i)
index=WHERE(timeDiff GT 180.0,count)
IF (count GT 0) THEN FOR i=0,count-1 DO xarr(index,1)=xarr(index,0)+86.0



; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling

if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1) & zmax=zscl(panel(0),2)
endif else BEGIN
    papco_autorange, zmat, zmin, zmax, LOG = zscl(panel(0),3), $
      NODATA = -1, exclude=0
    zscl(panel(0),1)=zmin & zscl(panel(0),2)=zmax
endelse

yst=yarr(0,0)*0.99 & yen=yarr(n_channels-1,1)+1.01
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
ENDELSE 

; use auto y-labeling routine to get "good" y-lables
papco_y_label, yst, yen, LOG=yscl(panel(0),3)
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO.
extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), ztitle:uztit, $
               xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set locally and the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!
nodata=-1 & resx=20 & resy=0.05

IF yscl(panel(0),3) EQ 1 THEN BEGIN
    papco_plot_colorspec,zmat,xarr,yarr,nodata=nodata,$
      xstyle=5, ystyle=5, resx=resx, resy=resy, _extra=extra_plotPar
ENDIF ELSE BEGIN
    papco_plot_colorspec,zmat,xarr,yarr,dy=40,nodata=nodata,$
      xstyle=5, ystyle=5, resx=resx, resy=resy, _extra=extra_plotPar
ENDELSE
  
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar     

papco_color_bar, _extra=extra_plotPar
  
; Plot S/C Potential
IF (scPot EQ 1) THEN BEGIN
    plot,[0, 1], _extra=extra_plotPar,xstyle=5,ystyle=5, /nodata
    lanl_mpa_gap_oplot, time, scPot_arr, _extra={color:1,psym:0}
ENDIF

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label, panel, uytitle,/rot90

; store the coordinate information into common block
plotted_x = !x & plotted_y = !y

; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='lanl_mpa template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
endif  
   
return
  
;jump to here if no data to plot 
noplot: 
message,'No data to plot',/cont
plotted_x = !x  &  plotted_y = !y  

END
