;******************************************************************************
;*  NAME:        r_lanl_mpa
;*
;*  DESCRIPTION: Routine to read mpa files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        August 1998
;*
;*  AUTOR:       Haje Korth
;*
;******************************************************************************
pro r_lanl_mpa, currPlotInfo, VERBOSE=VERBOSE

COMMON lanl_mpa_data, input_header, input_data, input_sat
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

message,/reset & get_err_no=0 & get_err_msg='lanl_mpa read succesfully'
   
; check for env. variable pointing to data directories
if not (papco_check_data_env('MPA_DATA', PATH=path)) then return
if not (papco_check_data_env('MPA_DATA_BIN', PATH=bin_path)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0  &  input_sat=0

; Assign satellite name
sc=sat_names(currPlotInfo.typevector(0))
input_sat=currPlotInfo.typevector(0)

; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
while mjdt_end.mjd ge time.mjd do begin
    ; assemble the data file from the time range requested
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt   
    IF day_om LE 9 THEN daystr='0'+STRTRIM(STRING(day_om),2) $
       ELSE daystr=STRTRIM(STRING(day_om),2)
    datestr=daystr+'-'+STRLOWCASE(cmonth)+'-'+STRTRIM(STRING(year),2)
    fln=sc+'_'+datestr
  
    ; read one day of data
    r_lanl_mpa_onefile, path+fln, bin_path+fln, VERBOSE=VERBOSE
   
    ; check for "OPENR: Error opening file" in error string - and then get 
    ; the file remotely!
    ; comment this section out if you do not want this feature.
    ;res=strpos(!err_string,'OPENR: Error opening file')
    ;if res ne -1 then begin
    ;   c_url=papco_getenv('MPA_FTP_ARCHIVE')
    ;   c_file=fln
    ;   c_download_dir=path
    ;   papco_get_remote_data, c_url, c_file, c_download_dir
    ;   ;check if file "arrived" and then read it
    ;   if get_err_no eq 0 then r_lanl_mpa_onefile, path+fln, /VERBOSE
    ;endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
       if no_files_read eq 0 then begin   ;start the array buffer
         input_data_buf=temporary(input_data)
       endif else begin                   ;otherwise concatonate arrays
         input_data_buf=[temporary(input_data_buf),temporary(input_data)]
       endelse        
       input_data=0                     ;clear array for reuse
       no_files_read=no_files_read+1
    endif
    day_count=day_count+1
    time.mjd=time.mjd+1                  ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) THEN break
    day_count=day_count+1
ENDWHILE    
   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg='% r_lanl_mpa: '+string(no_files_read,format="(i2)")+ $
      ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
endif else begin
    get_err_no=1
    get_err_msg='% r_lanl_mpa: no files found for timerange'
    return
endelse

; remove invalid data points
idx=where(input_data.key2 EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT = c2)
if c1 ne 0 then begin
    message, 'Removing '+varprt(c2)+' invalid data points.', /cont
    input_data=input_data(idx)
ENDIF

END 

;******************************************************************************
;*  NAME:        r_lanl_mpa_onefile,fln,bin_fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln     - string of original data filename
;*               bin_fln - string of IDL binary data filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        August 1998
;*
;*  AUTOR:       Haje Korth
;******************************************************************************
pro r_lanl_mpa_onefile, fln, bin_fln, VERBOSE=VERBOSE
  
COMMON lanl_mpa_data, input_header, input_data, input_sat
COMMON get_error, get_err_no, get_err_msg
   
get_err_no=0 & get_err_msg='mpa onefile read succesfully'
   
; clear data arrays for reuse
input_header=0  &  input_data=0
   
on_ioerror, read_original
restore, bin_fln+'.idl', VERBOSE=VERBOSE
dealloc_lun,/quiet    ;frees logical units used by save/restore
message,'saveset read:', /cont
print, '  ',bin_fln+'.idl'
return

read_original:
; if the .idl save file could not be found, then try to read the original
r_lanl_mpa_onefile_original, fln, bin_fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_lanl_mpa_onefile_original,fln,bin_fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln     - string of original data filename
;*               bin_fln - string of IDL binary data filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        August 1998
;*
;*  AUTOR:       Haje Korth
;******************************************************************************
pro r_lanl_mpa_onefile_original, fln, bin_fln, VERBOSE=VERBOSE
  
COMMON lanl_mpa_data, input_header, input_data, input_sat
COMMON get_error, get_err_no, get_err_msg  
   
get_err_no=0 & get_err_msg='mpa_onefile_orignal read succesfully'
   
; clear data arrays for reuse
input_header=0  &  input_data=0
   
;------------------------------------------------------------------------------
; read the original data file
  
; define the header and data structures. Edit to suit your data!
max_no_rec=5000              ;the maximum number time records in your data
ext='.ascii_moments'                   ;file extention of your dat files
montharr=['JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL',$
          'AUG', 'SEP', 'OCT', 'NOV', 'DEC']
input_header={mpa_header,tag1:0,tag2:0}
mjdt = {MJDT, MJD:0l, T:0.0}
tai = 0.0d
dummy={key1:0l,key2:0l, uthours:0.0d, lthours:0.0d, csat_id:'', $
       cdate_st:'', xgeo:0.0d, ygeo:0.0d, zgeo:0.0d, $
       dimaglat:0.0d, scmlat:0.0d, scmlt:0.0d, flag_arr:INTARR(4), $
       spin_rate:0.0d, sc_poten:0.0d, barr:0.0d, backgrd:0.0d, $
       ece4:DBLARR(40), ecp4:DBLARR(40), e4sav:DBLARR(6), g4sav:DBLARR(6), $
       ave_flux_e:DBLARR(40), ave_flux_i:DBLARR(40), denlp:0.0d, $
       vellp:DBLARR(3), tlp:DBLARR(3), tthetalp:DBLARR(3), $
       tphilp:DBLARR(3), denhp:0.0d, velhp:DBLARR(3), thp:DBLARR(3), $
       tthetahp:DBLARR(3), tphihp:DBLARR(3), tparhp:0.0d, tperhp:0.0d, $
       theta_p:0.0d, phi_p:0.0d, denhe:0.0d, velhe:DBLARR(3), the:DBLARR(3), $
       tthetahe:DBLARR(3), tphihe:DBLARR(3), tparhe:0.0d, tperhe:0.0d, $
       theta_e:0.0d, phi_e:0.0d}   
dummy1 = create_struct(name = 'lanl_mpa_data', dummy, $
                       'mjdt_time', mjdt, 'tai', tai)
input_data=replicate(dummy1, max_no_rec)

message, 'Reading ASCII file: (slow)', /cont
print, '  '+fln+ext

;files may be gzipped. Check and unzip.
zipflag = 0
r = findfile(fln+ext, count = c)
IF c EQ 0 THEN BEGIN
    r_zip = findfile(fln+ext+'.gz', count = c)
    IF c NE 0 THEN BEGIN 
        unzipfile = strmid(r_zip, 0, strlen(r_zip)-3) & zipflag = 1
        print, 'gunzip -c '+r_zip+' >! '+ unzipfile
        spawn, 'gunzip -c '+r_zip+' >! '+ unzipfile
    ENDIF
ENDIF

i=-1
on_ioerror, out  
openr,unit,fln+ext,/get_lun
finfo=fstat(unit)
fsize=finfo.size

if (fsize gt 0) then begin
     while not eof(unit) do begin
       i=i+1
       ;read in data and store in input_data 
       READF,unit,dummy, $
          FORMAT='(2I8,2F8.4,1X,A8,1X,A11,1X,3E12.3,3E12.5,4I4,3E12.5,E12.5,40E12.5,40E12.5,6E12.5,6E12.5,40E12.5,40E12.5,E12.5,3E12.5,3E12.5,3E12.5,3E12.5,E12.5,3E12.5,3E12.5,3E12.5,3E12.5,4E12.5,E12.5,3E12.5,3E12.5,3E12.5,3E12.5,4E12.5)'
       day=FIX(STRMID(dummy.cdate_st,0,2))
       month=STRMID(dummy.cdate_st,3,3)
       year=FIX(STRMID(dummy.cdate_st,7,4))
       mjdt=convert_date_to_t90(day=day,month=month,year=year, $
                                hour=dummy1.uthours,/MJDT)
       struct_assign, dummy, dummy1

       dummy1.mjdt_time = mjdt

       hr=fix(dummy.uthours) & min=(dummy.uthours-hr)*60.0
       sec=fix((min-fix(min))*60.0) & min=fix(min)
       tai=UTC2TAI({YEAR:year, MONTH:month, DAY:day, HOUR:hr, $
                    MINUTE:min, SECOND:sec, MILLISECOND:0})
       dummy1.tai = tai

       input_data(i)=dummy1

     endwhile
     input_data=input_data(0:i)
endif

close,unit & free_lun,unit

IF zipflag THEN spawn, 'rm '+unzipfile

; end reading original data
;------------------------------------------------------------------------------
; save read data with correct time. Rename data to *.idl
save, filename = bin_fln+'.idl', input_header, input_data, input_sat, $
  /compress, /verbose  
dealloc_lun,/quiet              ;frees logical units used by save/restore
return
  
out:
; if there was an error, report it. Use the IDL error variables. Error
; 171 is for "file not fond" which is used to invoke the remote get.  
get_err_no=!ERROR
get_err_msg='r_lanl_mpa_onefile:!C '+!ERR_STRING
   
end

;******************************************************************************
;*  NAME:        r_lanl_mpa_channels
;*
;*  DESCRIPTION: Routine to read lanl_mpa energy channel assignments
;*
;*  INPUT:       sat         name of satellite
;*               species     0 elec, 1 ion
;*
;*  OUTPUT:      yray of channel start, stop, in eV
;*          
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        August 2002
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:  
;******************************************************************************
function r_lanl_mpa_channels, sat, species,  VERBOSE = VERBOSE 

COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents

IF keyword_set(VERBOSE) THEN verbose = VERBOSE ELSE verbose = 0


root = papco_getenv('PAPCO_MODULES')
path = 'lanl_mpa/plot_read/MPA_Adopted_Channel_Energies.txt'

;first, read all the channle info
openr, u, root+path, /get_lun
instr = '' &  inarr = fltarr(29) & channels = fltarr(41, 28)

readf, u, instr & instr = strcompress(instr) &  instr = strlowcase(instr)
headers = strsplit(instr, ' ', /extract)
headers = headers(1:28)

FOR i = 0, 40 DO BEGIN
     readf,u,inarr
     channels(i, *) = inarr(1:28)
ENDFOR
close, u & free_lun, u

bird = sat_names(sat)
r = strsplit(bird, '-', /extract)
sat_id = r(1)

 
   IF species EQ 0 THEN head = 'elec_'+sat_id ELSE head = 'ion_'+sat_id
   pos = strpos(headers, head)
   idx = where(pos NE -1)
   
   ch =  reverse(channels(*, idx(0)))
   nch = n_elements(ch)
   yarr = fltarr(nch-1, 2)
   
   yarr(*, 0) = ch(0:nch-2)
   yarr(*, 1) = ch(1:nch-1)

   return, yarr

END
