;******************************************************************************
;*  NAME:        r_lanl_mpa_pitch
;*
;*  DESCRIPTION: Routine to read mpa pitch data files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        May 2003
;*
;*  AUTOR:       Arne Aasnes at LANL
;*
;******************************************************************************
pro r_lanl_mpa_pitch, currPlotInfo, VERBOSE=VERBOSE

  COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data, mpa_pitch_sat
  COMMON mjdt, mjdt_start, mjdt_end
  COMMON lanl_mpa               ;see lanl_mpa_set_commons for contents
  COMMON mpa_pitch_time,  mpa_pitch_year, mpa_pitch_month, mpa_pitch_day_om
  COMMON get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no is not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoke the plot routine for this data if get_err_no = 0 , otherwise an
; empty panel with the error message get_err_msg is plotted.

  message,/reset & get_err_no=0 & get_err_msg='lanl_mpa_pitch read succesfully'
  
  control = *currPlotInfo.USR_PTR1


; check for env. variable pointing to data directories
  if not (papco_check_data_env('MPA_PITCH_DATA', PATH=path)) then return
  if not (papco_check_data_env('MPA_PITCH_DATA_BIN', PATH=bin_path)) then return

; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
  
; clear data arrays for reuse
  mpa_pitch_header=0  &  mpa_pitch_data=0  &  mpa_pitch_sat=0

; Assign satellite name
  sc=sat_names(control.sc_id)
  mpa_pitch_sat=sc

  sat_codes = ['l9', 'l0', 'l1', 'l4', 'l7', 'a1', 'a2']
  scc = sat_codes(control.sc_id)

; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
  
  time=mjdt_start  &  no_files_read=0  &  day_count=0.0
  path0 = path                  ;--cy--030804
  while mjdt_end.mjd ge time.mjd do begin
                                ; assemble the data file from the time range requested
     convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt   
    IF day_om LE 9 THEN daystr='0'+STRTRIM(STRING(day_om),2) $
       ELSE daystr=STRTRIM(STRING(day_om),2)
    IF month LE 9 THEN monthstr = '0'+STRTRIM(STRING(month),2) $
       ELSE monthstr=STRTRIM(STRING(month),2)
    datestr=daystr+'-'+STRLOWCASE(cmonth)+'-'+STRTRIM(STRING(year),2)
;    fln=sc+'_'+datestr
    
    yearstr = STRTRIM(STRING(year),2)
    path = path +yearstr+'/'
    bin_path = bin_path+yearstr+'/'
;    path2 = path+STRTRIM(string(year), 2)+monthstr+'/'
;    path3 = path1+STRTRIM(string(year), 2)+monthstr+'/'
    fln = scc+'_h0_mpa_'+yearstr+monthstr+ $
          daystr+'_v02'

    bin_fln = bin_path+fln
    fln = path+fln
    
    mpa_pitch_year = year
    mpa_pitch_month = month
    mpa_pitch_day_om = day_om
    

    ; read one day of data
    r_lanl_mpa_pitch_onefile, fln, bin_fln,currPlotInfo, VERBOSE=VERBOSE
   
    ; check for "OPENR: Error opening file" in error string - and then get 
    ; the file remotely!
    ; comment this section out if you do not want this feature.
    ;res=strpos(!err_string,'OPENR: Error opening file')
    ;if res ne -1 then begin
    ;   c_url=papco_getenv('MPA_FTP_ARCHIVE')
    ;   c_file=fln
    ;   c_download_dir=path
    ;   papco_get_remote_data, c_url, c_file, c_download_dir
    ;   ;check if file "arrived" and then read it
    ;   if get_err_no eq 0 then r_lanl_mpa_pitch_onefile, path+fln, /VERBOSE
    ;endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
       if no_files_read eq 0 then begin   ;start the array buffer
         mpa_pitch_data_buf=temporary(mpa_pitch_data)
       endif else begin                   ;otherwise concatonate arrays
         mpa_pitch_data_buf=[temporary(mpa_pitch_data_buf),temporary(mpa_pitch_data)]
       endelse        
       mpa_pitch_data=0                     ;clear array for reuse
       no_files_read=no_files_read+1
    endif
;    day_count=day_count+1
    time.mjd=time.mjd+1                  ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) THEN break
    day_count=day_count+1
ENDWHILE    

IF n_elements(mpa_pitch_data_buf) LE 1 THEN get_err_no = 1

; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg='% r_lanl_mpa_pitch: '+string(no_files_read,format="(i2)")+ $
      ' files read succesfully'
    ;clear arrays and load buffer into output array
    mpa_pitch_data=0  &  mpa_pitch_data=mpa_pitch_data_buf  &  mpa_pitch_data_buf=0
endif else begin
    get_err_no=1
    get_err_msg='% r_lanl_mpa_pitch: no files found for timerange'
    return
endelse

; remove invalid data points
;idx=where(mpa_pitch_data.key2 EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT = c2)
;if c1 ne 0 then begin
;    message, 'Removing '+varprt(c2)+' invalid data points.', /cont
;    mpa_pitch_data=mpa_pitch_data(idx)
;ENDIF

END 

;******************************************************************************
;*  NAME:        r_lanl_mpa_pitch_onefile,fln,bin_fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures mpa_pitch_header and
;*               mpa_pitch_data
;*
;*  INPUT:       fln     - string of original data filename
;*               bin_fln - string of IDL binary data filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        May 2003
;*
;*  AUTOR:       Arne Aasnes at LANL
;******************************************************************************
pro r_lanl_mpa_pitch_onefile, fln, bin_fln,currPlotInfo, VERBOSE=VERBOSE
  
COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data, mpa_pitch_sat
COMMON get_error, get_err_no, get_err_msg

control = *currPlotInfo.USR_PTR1
get_err_no=0 & get_err_msg='mpa onefile read succesfully'
   
; clear data arrays for reuse
mpa_pitch_header=0  &  mpa_pitch_data=0
   
on_ioerror, read_original

IF control.mag_corr EQ 0 THEN extra = '' ELSE IF control.mag_corr EQ 1 THEN extra = '.mag_corr'

restore, bin_fln+extra+'.idl', VERBOSE=VERBOSE
dealloc_lun,/quiet    ;frees logical units used by save/restore
message,'saveset read:', /cont
print, '  ',bin_fln+extra+'.idl'
return

read_original:
; if the .idl save file could not be found, then try to read the original
r_lanl_mpa_pitch_onefile_original, fln, bin_fln,currPlotInfo, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_lanl_mpa_pitch_onefile_original,fln,bin_fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln     - string of original data filename
;*               bin_fln - string of IDL binary data filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        May 2003
;*
;*  AUTOR:       Arne Aasnes at LANL
;******************************************************************************
pro r_lanl_mpa_pitch_onefile_original, fln, bin_fln,currPlotInfo, VERBOSE=VERBOSE
  
COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data, mpa_pitch_sat
COMMON get_error, get_err_no, get_err_msg  

 
control = *currPlotInfo.USR_PTR1  
get_err_no=0 & get_err_msg='mpa_pitch_onefile_original read succesfully'
   
; clear data arrays for reuse
mpa_pitch_header=0  &  mpa_pitch_data=0
   
;------------------------------------------------------------------------------
; read the original data file
  
; define the header and data structures. Edit to suit your data!
max_no_rec=5000              ;the maximum number time records in your data
ext='.cdf'                   ;file extension of your dat files
montharr=['JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL',$
          'AUG', 'SEP', 'OCT', 'NOV', 'DEC']
mjdt = {MJDT, MJD:0l, T:0.0}
tai = 0.0d


;message, 'Reading cdf file: (slow)', /cont
;print, '  '+fln+ext


r = findfile(fln+ext, count = c)
i=-1
;on_ioerror, out  

;reading cdf data file, calculating pitch angles and putting the data in mpa_pitch_data
IF c EQ 1 THEN flux, fln+ext, mag_corr = control.mag_corr ELSE BEGIN 
   print, '>> '+fln+ext+' can"t be found!!'
   GOTO, out
ENDELSE 

;dim = n_elements(mpa_pitch_data) 
IF get_err_no NE 0 THEN GOTO, out


; end reading original data
;------------------------------------------------------------------------------
; save read data with correct time. Rename data to *.idl
IF control.mag_corr EQ 0 THEN extra = '' ELSE IF control.mag_corr EQ 1 THEN extra = '.mag_corr'
save, filename = bin_fln+extra+'.idl', mpa_pitch_header, mpa_pitch_data, mpa_pitch_sat, $
  /compress, /verbose  
dealloc_lun,/quiet              ;frees logical units used by save/restore

return
  
out:
; if there was an error, report it. Use the IDL error variables. Error
; 171 is for "file not fond" which is used to invoke the remote get.  

IF !error NE 0 THEN get_err_no = !error ELSE $
get_err_no = 1
get_err_msg='r_lanl_mpa_pitch_onefile:!C '+'data file not found!'

   
END 

;******************************************************************************
PRO m_lanl_mpa_cdf_links,  UPDATE = UPDATE, YEAR = YEAR

sat_codes = ['l9', 'l0', 'l1', 'l4', 'l7', 'a1', 'a2']

base_dir = '$PAPCO_DLNK/lanl/geo/mpa_cdf/'

;make one large list of all files with their full paths. Use several
;spawn calls to the find function.

fln_template = '??_h0_mpa_????????_v02.cdf'

cdf_dirs = ['/n/radbelt/data4/tmp/lanl/geo/mpa_cdf/', $
            '/n/space/data4/mpa/cCDF/', $
            '/n/tempest/projects/mpa/cd_ready/']

FOR i = 0, n_elements(cdf_dirs)-1 DO BEGIN
    message, 'Looking in:', /info & print, '  ',cdf_dirs(i)
    spawn, "find "+cdf_dirs(i)+" -name '"+fln_template+"' -print", result
    IF i EQ 0 THEN cdf_files = result ELSE cdf_files = [cdf_files, result]
ENDFOR 

IF keyword_set(YEAR) THEN BEGIN
    y1 = year(0) & y2 = year(1)
ENDIF ELSE BEGIN
    y1 = 1994 & y2 = 2003
ENDELSE 

;got through years selecting out files, make dirs per year under base_dir
FOR i = y1, y2 DO BEGIN
    year = varprt(i)
    result = strpos(cdf_files, year)
    idx = where(result NE -1, c)
    IF c EQ 0 THEN contrinue 
    year_cdf_files = cdf_files(idx)
    
    ;see if directory needs to be made
    papco_create_dir, base_dir+year, /NOASK
    
    ;make links for all the files for this year
    FOR j = 0, c-1 DO BEGIN
        papco_sepdir, year_cdf_files(j), dir, file, ext
        dest = base_dir+year+'/'+file+ext
        print, 'ln -s '+year_cdf_files(j)+' '+dest
        spawn, 'ln -s '+year_cdf_files(j)+' '+dest
    ENDFOR
    
ENDFOR     

END

;******************************************************************************
PRO m_lanl_mpa_cdf_missing, YEAR = YEAR

base_dir = '$PAPCO_DLNK/lanl/geo/mpa_cdf/'

fln_template = '??_h0_mpa_????????_v02.cdf'

IF keyword_set(YEAR) THEN BEGIN
    y1 = year(0) & y2 = year(1)
ENDIF ELSE BEGIN
    y1 = 1994 & y2 = 2003
ENDELSE 

FOR i = y1, y2 DO BEGIN
    year = varprt(i)
    message, 'looking for missing files for year: '+year, /info
    spawn, "find "+base_dir+year+" -name '"+fln_template+"' -print", result
    dates = strmid(result, 48, 8)

    ;look for missing files this year
    utc1 = str2utc(year+'-01-01')
    utc2 = str2utc(year+'-12-31')
    utc = utc1
    FOR j = utc1.mjd, utc2.mjd DO BEGIN
        utc.mjd = j
        date = anytim2cal(utc, form = 8, /date)
        idx = where(dates EQ date, c)
        IF c EQ 0 THEN BEGIN
            message, 'mpa cdf for date: '+date+' missing', /info
        ENDIF 
    ENDFOR     

ENDFOR 

END
