;+
; NAME:
;     READCDF
;
; IDENT:
;     $Id: readcdf.pro,v 1.4 2004/05/04 15:07:52 friedel Exp $
;
; PURPOSE:
;     Read in the contents of a CDF file, and return them in a data structure.
;
; AUTHOR:
;     Ed Santiago
;
; CALLING SEQUENCE:
;     data = READCDF( filename [,/Quiet] )
;
; INPUTS:
;     filename	- name of CDF file to read
;
; KEYWORDS:
;     /Quiet	- Read silently, without emitting progress reports.
;
; OUTPUTS:
;     A data structure containing all the elements in the CDF.
;
; SIDE EFFECTS:
;
; EXAMPLE:
;
;     IDL> data = readcdf('1994-084_05-APR-1998.CDFSAVE')
;     Reading:    Theta_l[6]
;     Reading:    Theta_u[6]
;     ....
;     Reading: theta_phi_[2][971]
;     IDL> help, data, /st
;     ** Structure <81e7d7c>, 24 tags, length=52280200, refs=1:
;        THETA_L         STRUCT    -> <Anonymous> Array[1]
;        THETA_U         STRUCT    -> <Anonymous> Array[1]
;        ...
;        THETA_PHI_E     STRUCT    -> <Anonymous> Array[1]
;     IDL> help, data.theta_phi_e, /st
;     ** Structure <81be43c>, 8 tags, length=7824, refs=2:
;        NAME            STRING    'theta_phi_e'
;        FIELDNAM        STRING    'polar,azim ang e symm axis'
;        VALIDMIN        FLOAT     Array[2]
;        VALIDMAX        FLOAT     Array[2] 
;        SCALEMIN        FLOAT     Array[2]
;        SCALEMAX        FLOAT     Array[2]
;        UNITS           STRING    'deg'
;        DATA            FLOAT     Array[971, 2]
;     IDL>
;-
FUNCTION readcdf, filename, Verbose=verbose, TimeSeries=timeseries
  COMPILE_OPT StrictArr
  
  COMMON get_error, get_err_no, get_err_msg
  
                                ; Perform sanity checks on our input parameters
  IF N_Elements(verbose) EQ 0 THEN verbose = 0
  
                                ; Open the file, and get the inquiry data from it.

;*****
  print,filename
;*****
  ON_IOERROR, nofile
  cdfid = cdf_open(filename)
  glob = cdf_inquire( cdfid )
  ON_IOERROR, null 

; if anything goes wrong, return nothing
  catch, Error_status
  IF error_status NE 0 THEN BEGIN 
     PRINT, 'Error index:', error_status
     print, 'Error messaage:', !error_state.msg
     return, 0
  ENDIF 
                                ; CDF files contain "attributes", such as a nice description of the
                                ; field, valid minimum and maximum values, units, etc.  Find out how
                                ; many attributes are available for this CDF, and use that info later.
  inq = CDF_Inquire(cdfid)
  IF inq.Natts EQ 0 THEN GOTO, nofile
  attrlist = [ {attrlist, attr_no:0L, name:'', maxZentry:0L} ]
  FOR attr=0, inq.NAtts-1 DO BEGIN
     CDF_Control, cdfid, ATTRIBUTE=attr, GET_ATTR_INFO=attr_info
     IF attr_info.maxZentry GT 0 THEN BEGIN
        CDF_AttInq, cdfid, attr, name, scope, maxE, maxZ
        attrlist = [ attrlist, {attrlist, attr, name, attr_info.maxZentry} ]
     ENDIF
  ENDFOR
  
                                ;
                                ; MAIN LOOP: first loop over the non-Z variables, then the Z variables.
                                ; Read them all in, and keep adding them to a structure.
                                ;
  FOR z=0,1 DO BEGIN
                                ; Loop over the number of this type (Z, non-Z) variables
     nvars = (z ? glob.nZvars : glob.nvars)
     FOR var=0, nvars-1 DO BEGIN
        info = CDF_VarInq(cdfid, var, Z=z)
        
        is_string = (info.datatype eq 'CDF_CHAR')
        
                                ; Z variables have a "dim" field... non-Z variables have a global dim
        dim = (z ? info.dim : glob.dim)

                                ; Since it takes so long to read these files, tell the user where we are
        IF verbose THEN BEGIN
           Print, format='("Reading: ",A12,$)',info.name
           IF dim[0] NE 0 THEN BEGIN
              Print, format='("[",I0,$)', dim[0]
              order = N_Elements(dim)
              FOR i=1,order-1 DO Print, format='("x",I0,$)', dim[i]
              Print, format='("]",$)'
           ENDIF
        ENDIF
        
                              ; Read in all the given variable's attributes
        field = Create_Struct( 'name', info.name )
        FOR attr=1,N_Elements(attrlist)-1 DO BEGIN
           Catch, err_sts
           IF err_sts EQ 0 THEN BEGIN
              CDF_AttGet, cdfid, attrlist[attr].attr_no, info.name, value
              field = Create_Struct(field, attrlist[attr].name, value)
           ENDIF
           Catch, /Cancel
        ENDFOR
        
                                ; Read in the data values.
                                ; NOVARY records are easy -- there's just one of them.  Slurp it in.
                                ; For VARY records, we need to figure out how many of them there are.
        IF info.recvar EQ 'NOVARY' THEN BEGIN
           Saved_Quiet=!Quiet	; ESM kludge: avoid the "PAD" warning msgs
           !Quiet=1
           CDF_VarGet, cdfid, info.name, data, Z=z, String=is_string
           !Quiet=Saved_Quiet
        ENDIF ELSE BEGIN
                                ; Find out how many of them there are, and read them all in.
           Saved_Quiet=!Quiet	; ESM kludge: avoid the "PAD" warning msgs
           !Quiet=1
           CDF_Control, cdfid, GET_VAR_INFO=varinfo, VARIABLE=info.name
           
                                ; It's possible to get a -1 here.  Deal with it
           IF varinfo.maxrec LT 0 THEN BEGIN
              IF verbose THEN Print, '[nope]'
              GOTO, nextvar
           ENDIF
           
           IF verbose THEN Print, format='("[",I0,"]",$)',varinfo.maxrec+1
           CDF_VarGet, cdfid, info.name, tmp, Z=z, Rec_Count = varinfo.maxrec+1,$
          String=is_string
           !Quiet=Saved_Quiet
           
                                ; Reform what we got into something nicer.  The CDF routines
                                ; return an array with the record number as the last index,
                                ; that is, something like Epoch[1,900], PCounts[24,6,40,900].
                                ; Since record number (ie, time) is the natural first index, we 
                                ; would rather have something that looks like PCounts[t, *, *, *]
                                ; than PCounts[*, *, *, t].  We would also rather have one-by-N
                                ; arrays (eg, Epoch[1,900]) converted to vectors (Epoch[900]).
                                ; The two simple-looking lines below take care of that.  We
                                ; use IDL's TRANSPOSE() function to move the last array index
                                ; into first position and keep the other indices the same.
                                ; A nice side-effect of TRANSPOSE() is that it will convert
                                ; one-by-N arrays into straight vectors.
                                ;
                                ; Yes, it's ridiculous to spend this much comment space on
                                ; one simple function, but it took me so long to get this
                                ; figured out that I reckon this should help someone later.
           s = size(tmp)
           IF varinfo.maxrec EQ 0 THEN BEGIN
              IF s[0] EQ 0 THEN tmp = [tmp] ELSE tmp = reform(tmp,[s[1:s[0]],1])
              s = size(tmp)
           ENDIF                ; maxrec == 0
           IF s[0] LE 1 THEN BEGIN
              data = [tmp]
           ENDIF ELSE BEGIN
              data = transpose(tmp, [s[0]-1, indgen(s[0]-1)])
                                ; sigh... preserve unary dimensions
              IF s[0] GT 2 THEN data = reform(data, [s[s[0]], s[1:s[0]-1]])
           ENDELSE
        ENDELSE
        IF verbose THEN Print, ''
        
                                ; We now have a nicely formatted field.  Create a data
                                ; structure and add all the fields as we read them in.
        field = Create_Struct(field, 'data', data)
        IF N_Elements(mystruct) EQ 0 THEN					$
        mystruct = Create_Struct(             info.name, field )	$
      ELSE								$
        mystruct = Create_Struct( mystruct,   info.name, field )
nextvar:
     ENDFOR                     ; nvars
  ENDFOR                        ; Z=0,1
  
                                ; Done!  Close the CDF and return what we made.
  CDF_Close, cdfid
  

                                ;
                                ; If /TimeSeries is set, convert to an array of structs.
                                ;
                                ; This is hairier than it sounds.  We do it in two passes:
                                ;
                                ;  * First, we define a new data structure, with one dimension (time)
                                ;    removed.  A 1D vector will be collapsed to a scalar, and any
                                ;    array of more dimensions will be shunted down one.  For instance,
                                ;    an array [100, 3, 4, 5] will become [3, 4, 5].
                                ;
                                ;    Note that it is imperative that the first dimension (time) be
                                ;    the same for every element.  If anything is different, we barf.
                                ;
                                ;  * Second, generate NPOINTS (e.g., 100 in the case above) instances
                                ;    of the data structure, and fill it in from the data we read in.
                                ;
  IF Keyword_Set(timeseries) THEN BEGIN
     
     tags = Tag_Names(mystruct)
   
                                ; Pass 1: define a data structure.
     FOR i=0L,N_Tags(mystruct)-1 DO BEGIN
        s = esmsize(mystruct.(i), 'data')
        
        IF N_Elements(npoints) EQ 0 THEN npoints = s[1]
        IF npoints NE s[1] THEN MESSAGE, 'foo'
        IF s[0] LE 1 THEN BEGIN
           st = mystruct.(i).data[0]
        ENDIF ELSE BEGIN
           st = Make_Array(dim=s[2:s[0]], type=s[s[0]+1])
           st = reform(st, s[2:s[0]]) ; Sigh... preserve unary dimensions
        ENDELSE
        
        IF N_Elements(newst) EQ 0 THEN BEGIN
           newst = Create_Struct(       tags[i], st)
        ENDIF ELSE BEGIN
           newst = Create_Struct(newst, tags[i], st)
        ENDELSE
     ENDFOR                     ; i (tags)

                                ; Pass 2: make NPOINTS instances of the data structure, and fill it in
     newst = Replicate(newst, npoints)
     FOR i=0L,N_Tags(mystruct)-1 DO BEGIN
        s = size(mystruct.(i).data)
        
                                ; Scalars are easy, but arrays get hairy: we need to transpose them
                                ; so the time dimension moves from first to last.
        IF s[0] LE 1 THEN BEGIN
           newst.(i) = mystruct.(i).data
        ENDIF ELSE BEGIN
           new_order = [ IndGen(s[0]-1)+1, 0 ]
           tmp = transpose(mystruct.(i).data, new_order)
                                ; Sigh... preserve unary dimensions
           s1  = esmsize(mystruct.(i), 'data')
           tmp = reform(tmp, [s1[2:s1[0]],s1[1]])
           newst.(i) = tmp
        ENDELSE
     ENDFOR
     
     mystruct = newst
  ENDIF

  
  RETURN, mystruct
  
nofile:
  print, !error_state.msg
  get_err_no = 1
  ger_err_msg = 'readcdf:!C '+'data file not found!'
  return, 0
  
END
