;******************************************************************************
;*  NAME:        r_mbi
;*
;*  DESCRIPTION: Routine to read mbi files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_mbi, VERBOSE=VERBOSE

  common mbi_data, input_header, input_data
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

  get_err_no=0  &  get_err_msg='mbi read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('MBI_DATA', path=PATH)) then return
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here the data is stored in files per year.
  no_files_read=0
  
  convert_t90_to_Date, mjdt_start, start_year, day_oy, hour, minute, second, $
    month, cmonth, day_om, /MJDT  
 
  convert_t90_to_Date, mjdt_end, end_year, day_oy, hour, minute, second, $
    month, cmonth, day_om, /MJDT  
    
  year=start_year
  while end_year ge year do begin
     
    ;assemble the data file from the time range requested "mnit1983.new"
    fln=string(year,format="('mnit',i4.4)")
     
    ;read one day of data
    r_mbi_onefile, path+fln, VERBOSE=VERBOSE
  
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin      
      c_url=papco_getenv('MBI_FTP_ARCHIVE')
      c_file=fln+'.idl'  &  c_download_dir=path
      papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      if get_err_no eq 0 then begin
        r_mbi_onefile, path+fln, VERBOSE=VERBOSE
      endif else begin
        get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
        return
      endelse  
    endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse        
      input_data=0                      ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    year=year+1                 ; next year
  endwhile         
   
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
  endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    return
  endelse      
   
; restrict data to actual time range requested
  ut1=mjdt_start.mjd + mjdt_start.t / 86400.0d
  ut2=mjdt_end.mjd + mjdt_end.t / 86400.0d
  index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
  if count ne 0 then input_data=input_data(index)

end

;******************************************************************************
;*  NAME:        r_mbi_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_mbi_onefile, fln, VERBOSE=VERBOSE
  
  common mbi_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0  &  get_err_msg='mbi onefile read succesfully'
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
  on_ioerror, read_original
  restore,fln+'.idl', VERBOSE=VERBOSE
  dealloc_lun,/quiet   ;frees logical units used by save/restore
  message,'idl saveset read succesfully: '+fln+'.idl', /cont
  return

  read_original:       ;if the .idl file is not found, try to read the original
  r_mbi_onefile_original, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_mbi_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_mbi_onefile_original, fln, VERBOSE=VERBOSE
  
  common mbi_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0  &  get_err_msg='mbi_onefile_orignal read succesfully'
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
; define the header and data structures. Edit to suit your data!
  max_no_rec=25000             ;the maximum number time records in your data
  ext='.new'                   ;file extention of your dat files
  input_header={mbi_header,title:'Aroral Midnight Boundary Index'}
  dummy1={year:0, doy:0, sec:0l, invlat:0.0}
  dummy2={mbi_dat, time:0.0d, year:0, doy:0, sec:0l, invlat:0.0}
  input_data=replicate(dummy2, max_no_rec)
  count=0
  
  message,'Reading original ASCII data - slow: '+fln+ext,/cont

  on_ioerror, out  
  openr,unit,fln+ext,/get_lun
  while not eof(unit) do begin   ;read in data and store in input_data
    readf,unit,dummy1
    result=convert_date_to_t90(doy=dummy1.doy, year=dummy1.year, $
                               seconds=dummy1.sec, /MJDT)
    input_data(count ).time= result.mjd + double(result.t / 86400.0d)
    for i=1,4 do input_data(count).(i)=dummy1.(i-1)
    count=count+1
  endwhile
  
  input_data=input_data(0:count-1)
  
; end of user written section
;------------------------------------------------------------------------------
  
; save read data with correct time. Rename data to *.idl
  save,filename=fln+'.idl',input_header,input_data, VERBOSE=VERBOSE
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING
   
end
