;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  noaa_eph_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION noaa_eph_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON noaa_eph_panel_data, widgetData
COMMON noaa_all

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
panel_base=papco_WIDGET_BASE(title=aPlotInfo.panelKind, row=1, $
                             GROUP_LEADER=group, /MODAL)
panel_left=widget_base(panel_base, column=1)
panel_right=widget_base(panel_base,column=1)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_left, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)

;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>

;--- add the widgets for choosing an satellite ----------------------------
sat_base=WIDGET_BASE(panel_left, row = 1, /frame)
sat_chosee_base = WIDGET_BASE(sat_base, column = 1)
lbl_satName=CW_FIELD(sat_chosee_base, title='Satellite ID chosen:', $
                     value=aPlotInfo.filename, $
                     xsize = 10)
lb_satNames=WIDGET_LIST(sat_base, ysize=5, xsize=10, $
                                  UVALUE='lb_satNames')
label_list_base=WIDGET_LABEL(sat_chosee_base, $
                             value='satellites for the time chosen ->');

;-- add widgets for choosing ephemers coords ----------------------------
eph_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
eph_base=CW_BGROUP(eph_outer, eph_names, ROW=1, /EXCLUSIVE, $
                   LABEL_TOP='Ephemeris Coordinates', $
                   UVALUE='spb_eph', $
                   SET_VALUE=aplotinfo.typevector(1))

mag_model_outer = widget_base(panel_left, COLUMN=1, /ALIGN_CENTER, /FRAME)

mag_model_base = widget_base(mag_model_outer, ROW=1, /ALIGN_CENTER)
int_model_list = [ $
    'DGRF/IGRF 1945-2000, reference field', $
    'Jensen&Cain (1962), epoch 1960', $
    'Jensen&Cain (1962), epoch 1960, SAA corr.', $
    'GSFC 12/66', $
    'GSFC 12/66, epoch 1970, SAA corr.', $
    'Dipole magnetic field', $
    'DGRF/IGRF 1945-2000, Kluge algorithm']
spb_int = CW_BGROUP(mag_model_base, int_model_list, COLUMN = 1, /EXCLUSIVE, $
                    LABEL_TOP='Internal Magnetic Field Model', $
                    UVALUE='spb_int', $
                    SET_VALUE=aplotinfo.ioptions(0))

ext_model_list = [ $
    'none', $
    'Mead & Fairfield [1975]',  $
    'Tsyganenko short [1987]', $ 
    'Tsyganenko long [1987]',  $
    'Tsyganenko [1989c]',  $
    'Olson & Pfitzer quiet [1977]',  $
    'Olson & Pfitzer dynamic [1988]',  $
    'Tsyganenko [1996]',  $
    'Ostapenko & Maltsev [1997]' ]
spb_ext = CW_BGROUP(mag_model_base, ext_model_list, COLUMN = 1, /EXCLUSIVE, $
                    LABEL_TOP='External Magnetic Field Model', $
                    UVALUE='spb_ext', $
                    SET_VALUE=aplotinfo.ioptions(1))

;--- provide choices for coordinates ------------------------------------------
coord_names = ['X', 'Y', 'Z', 'Magnitude'] & set = bytarr(4)
toggle = aPlotInfo.ioptions(2) 
for i=0,3 do if (toggle and 2ll^i) eq 2ll^i then set(i)=1 else set(i)=0
coord_base=CW_BGROUP(mag_model_outer,coord_names, COLUMN=4, /NONEXCLUSIVE, $
                     UVALUE='spb_coord', $
                     LABEL_LEFT = 'Model Magnetic Field  Coordinates:', $
                     SET_VALUE=set) 

;--- buttons to chose other options --------------------------------------------
opti_base=WIDGET_BASE(panel_left, ROW=1, /FRAME)
  
average_names=['seg.','av.']
average_base=CW_BGROUP(opti_base, average_names, /FRAME, ROW=1, /EXCLUSIVE, $
                       UVALUE='spb_average', $
                       SET_VALUE=aPlotInfo.options(8))
  
reload_names=['Relaod', 'Adopt']
reload_base=CW_BGROUP(opti_base, reload_names, /FRAME, ROW=1, /EXCLUSIVE, $
                      UVALUE='spb_reload', $
                      SET_VALUE=aPlotInfo.options(9))

;--- widget to chose variable to select range for ------------------------------
vars_outer_base=WIDGET_BASE(panel_left, /FRAME, COLUMN=1) 
  
vars_names = ['select L', 'select MLT', 'select MagLat']
vars_base=CW_BGROUP(vars_outer_base, vars_names, ROW=1, /EXCLUSIVE, $
                    UVALUE='spb_vars', SET_VALUE=aPlotInfo.options(4))

;--- make sliders selecting ranges - not all of them mapped! -------------------
select_outer_base=WIDGET_BASE(vars_outer_base, COLUMN=1, /align_center)
select_base=WIDGET_BASE(select_outer_base, /align_center)
  
;--- sliders to select the L-range to be plotted -------------------------------
l_base=WIDGET_BASE(select_base, column=1, /align_center) 
  
l_rangeNames=['no L range', 'select L range']
l_range_base=CW_BGROUP(l_base, l_rangeNames, ROW=1, /EXCLUSIVE, $
                       UVALUE='spb_l_range', $
                       SET_VALUE=aPlotInfo.typevector(3) < 1)  
  
l_label_base=WIDGET_BASE(l_base, row=1, /align_center)
l_low_label= WIDGET_LABEL(l_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(2),format="(f4.1)"),35,/center), $
                             UVALUE='l_low_label')
l_high_label=WIDGET_LABEL(l_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(3),format="(f4.1)"),35,/center), $
                             UVALUE='l_high_label')

l_slider_base=WIDGET_BASE(l_base, row=1, /align_center)
; --- Low L-value --------------------------------------------------------------
l_low=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=250, $
                  title=strformat('Low L-value',35,/center), $
         	  VALUE=(((aPlotInfo.options(2) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_low', MINIMUM=0, MAXIMUM=110)	      
; --- High L-value -------------------------------------------------------------
l_high=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=250, $
                  title=strformat('High  L-value',35,/center), $
		  VALUE=(((aPlotInfo.options(3) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_high', MINIMUM=0, MAXIMUM=110)
  
;--- sliders to select the MLT-range to be plotted -----------------------------
mlt_base=WIDGET_BASE(select_base, column=1, /align_center) 
  
mlt_rangeNames=['no MLT range', 'select MLT range']
mlt_range_base=CW_BGROUP(mlt_base, mlt_rangeNames, ROW=1, /EXCLUSIVE, $
                           UVALUE='spb_mlt_range', $
                           SET_VALUE=aPlotInfo.options(7))
  
mlt_label_base=WIDGET_BASE(mlt_base, row=1, /align_center)
mlt_from_label= WIDGET_LABEL(mlt_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(2),format="(f4.1)"),35,/center), $
                             UVALUE='mlt_from_label')
mlt_to_label=WIDGET_LABEL(mlt_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(3),format="(f4.1)"),35,/center), $
                             UVALUE='mlt_to_label')    
  
mlt_slider_base=WIDGET_BASE(mlt_base, row=1, /align_center)
; --- from MLT-value --------------------------------------------------------
mlt_from=WIDGET_SLIDER(mlt_slider_base, /DRAG, /SUPPRESS_VALUE, xsize=250,$
                  title=strformat('From MLT',35,/center), $
         	  VALUE=(aPlotInfo.options(5) * 2) < 49 > 0, $
                  UVALUE='mlt_from', MINIMUM=0, MAXIMUM=48)	      
  
; --- to MLT-value ----------------------------------------------------------
mlt_to=WIDGET_SLIDER(mlt_slider_base, /DRAG, /SUPPRESS_VALUE, xsize=250, $
                  title=strformat('To MLT',35,/center), $
		  VALUE=(aPlotInfo.options(6) * 2) < 49 > 0, $
                  UVALUE='mlt_to', MINIMUM=0, MAXIMUM=48)  
   
;--- sliders to select the MLAT-range to be plotted ---------------------------
mlat_base=WIDGET_BASE(select_base, column=1, /align_center) 
                                
                                 ;still add in this option...

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    noaa_eph. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;     widgets=papco_panel_AddFile_Widgets(panel_left, aPlotInfo)
;     additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
;     widgets=papco_panel_Addcolorbar_Widgets(panel_left, aPlotInfo)
;     additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(panel_right, aPlotInfo, 0, /vert)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(panel_right, aPlotInfo, /vert)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
;     widgets=papco_panel_AddZScale_Widgets(panel_right, aPlotInfo,/VERTICAL)
;     additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_right, Action, aPlotInfo, $
                                     /vert, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the noaa_eph example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'noaa_eph_panel.help', $
                 ioptions:aPlotInfo.ioptions, $
                 options:aPlotInfo.options, $
                 typevector:aPlotInfo.typevector, $
                 l_base:l_base, $
                 l_low:l_low, $
                 l_high:l_high, $
                 l_low_label:l_low_label, $
                 l_high_label:l_high_label, $
                 mlt_base:mlt_base, $
                 mlt_from:mlt_from, $
                 mlt_to:mlt_to, $
                 mlt_from_label:mlt_from_label, $
                 mlt_to_label:mlt_to_label, $
                 mlat_base:mlat_base, $
                 satname:aPlotInfo.filename, $                    
                 lbl_satName:lbl_satName, $
                 lb_satNames:lb_satNames, $
                 eph_outer:eph_outer}
widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData
WIDGET_CONTROL, panel_base, HOURGLASS=1
noaa_eph_refresh, /ALL
WIDGET_CONTROL, panel_base, HOURGLASS=0
papco_xmanager, 'noaa_eph_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
returnData={typevector:widgetData.typevector, $
            options:widgetData.options, $
            ioptions:widgetData.ioptions, $
            filename:WidgetData.satname}
additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO noaa_eph_refresh
;* 
;* DESCRIPTION:  
;*      refresh rotuine for noaa_eph_panel, called by event handler
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO noaa_eph_refresh, ALL = ALL, $
                      COORDS = COORDS, SATID = SATID, SATLIST=SATLIST, $
                      L_LIMIT=L_LIMIT, PLOT_STYLE = PLOT_STYLE, $
                      LOW_L=LOW_L, HIGH_L=HIGH_L, $ 
                      MLT=MLT, RANGE_MLT=RANGE_MLT

COMMON noaa_eph_panel_data, widgetData
COMMON noaa_list, satnames

IF keyword_set(ALL) THEN BEGIN
    COORDS = 1
    SATID=1
    L_LIMIT=1
    LOW_L=1
    HIGH_L=1
    MLT=1
    RANGE_MLT=1
    PLOT_STYLE=1
    SATLIST=1
ENDIF

IF keyword_set(COORDS) THEN BEGIN
    IF widgetData.ioptions(2) EQ 0 THEN BEGIN ;no mag field coords chosen
        widget_control, widgetData.eph_outer,  sensitive = 1
    ENDIF ELSE BEGIN
        widget_control, widgetData.eph_outer,  sensitive = 0
    ENDELSE 
END

IF KEYWORD_SET(SATLIST) THEN BEGIN
    satnames=noaa_satlist()
    WIDGET_CONTROL,WidgetData.lb_satNames, SET_VALUE=[satnames]
ENDIF

IF KEYWORD_SET(SATID) THEN BEGIN
    WIDGET_CONTROL,WidgetData.lbl_satName, SET_VALUE=WidgetData.satName
ENDIF

IF keyword_set(PLOT_STYLE) THEN BEGIN 
    bases = [widgetData.l_base, widgetData.mlt_base,widgetData.mlat_base]
    CASE widgetData.options(4) OF 
        0: map = [1, 0, 0]
        1: map = [0, 1, 0]
        2: map = [0, 0, 1]
    ENDCASE
    FOR i = 0, n_elements(map)-1 DO $
      widget_control, bases(i), MAP = map(i)
ENDIF 

if keyword_set(L_LIMIT) then BEGIN
    bases = [widgetData.l_low, widgetData.l_high, $
             widgetData.l_low_label, widgetData.l_high_label]
    CASE widgetData.typeVector(3) OF
        0: sensitive = [0, 0, 0, 0]
        1: sensitive = [1, 1, 1, 1]
    ENDCASE
    FOR i = 0, n_elements(sensitive)-1 DO $
      WIDGET_CONTROL, bases(i), SENSITIVE=sensitive(i)
ENDIF

IF keyword_set(LOW_L) THEN BEGIN 
    high_l=widgetData.options(3) > (widgetData.options(2)+0.1)
    widgetData.options(3)=high_l
    WIDGET_CONTROL, widgetData.l_high, $
      SET_VALUE=(widgetData.options(3) -1) * 10
    
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)
ENDIF   

IF keyword_set(HIGH_L) THEN BEGIN 
    low_l=widgetData.options(2) < (widgetData.options(3)-0.1)
    
    widgetData.options(2)=low_l
    WIDGET_CONTROL, widgetData.l_low, $
      SET_VALUE=(widgetData.options(2) -1) * 10
    
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)    
ENDIF 

IF keyword_set(RANGE_MLT) THEN BEGIN 
    bases = [widgetData.mlt_from, widgetData.mlt_to, $
             widgetData.mlt_from_label, widgetData.mlt_to_label]
    CASE widgetData.options(7) OF
        0: sensitive = [0, 0, 0, 0]
        1: sensitive = [1, 1, 1, 1]
    ENDCASE
    FOR i = 0, n_elements(sensitive)-1 DO $
      WIDGET_CONTROL, bases(i), SENSITIVE=sensitive(i)
ENDIF

if keyword_set(MLT) then begin
    label=string(widgetData.options(5),format="(f4.1)")
    WIDGET_CONTROL, widgetData.mlt_from_label, $
      SET_VALUE=strformat(label,35,/center)
    label=string(widgetData.options(6),format="(f4.1)")
    WIDGET_CONTROL, widgetData.mlt_to_label, $
      SET_VALUE=strformat(label,35,/center)
ENDIF   

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO noaa_eph_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO noaa_eph_panel_event, event
  
COMMON noaa_eph_panel_data, widgetData
COMMON noaa_list, satnames

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF
    'lb_satNames' : IF N_ELEMENTS(size(satnames)) GE 1 THEN BEGIN
        WidgetData.satName=satnames(event.index)
        noaa_eph_refresh, /SATID
    END 
   'spb_average'   : widgetData.options(8)=event.value
   'spb_reload'    : widgetData.options(9)=event.value

    'spb_vars' : begin
        widgetData.options(4)=event.value
        noaa_eph_refresh, /PLOT_STYLE
    end
    'spb_l_range' : begin
        widgetData.typeVector(3)=event.value
        noaa_eph_refresh, /L_LIMIT
    end  
    'spb_mlt_range' : begin
        widgetData.options(7)=event.value
        noaa_eph_refresh, /RANGE_MLT
    end  
    'l_low'    : begin
        widgetData.options(2)=event.value / 10.0 + 1.0
        noaa_eph_refresh, /LOW_L
    end
    'l_high'   : begin
        widgetData.options(3)=event.value / 10.0 + 1.0
        noaa_eph_refresh, /HIGH_L
    end  
    'mlt_from'    : begin
        widgetData.options(5)=event.value / 2.0 
        noaa_eph_refresh, /MLT
    end  
    'mlt_to'   : begin
        widgetData.options(6)=event.value / 2.0 
        noaa_eph_refresh, /MLT
    END    
    'spb_eph' : begin
        widgetData.typevector(1)=event.value
        noaa_eph_refresh, /COORDS
    END  
    'spb_int' : begin
        widgetData.ioptions(0)=event.value
    END  
    'spb_ext' : begin
        widgetData.ioptions(1)=event.value
    END  
    'spb_coord':BEGIN
        widget_control, event.id, get_value = set
        widgetData.ioptions(2) = 0
        for i=0,n_elements(set)-1 DO IF set(i) THEN $
          widgetData.ioptions(2) = widgetData.ioptions(2)+2ll^i
        noaa_eph_refresh, /COORDS
    END 
    else: BEGIN
        widgetData=papco_panel_Event(event, widgetData)
    END
ENDCASE 
    
END
