;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  noaa_flux_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the PAPCO_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $PAPCO_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      switch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $PAPCO_XX/PAPCO_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  noaa_flux_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON noaa_flux_panel_data, widgetData
COMMON noaa_all ;commons for module - see m_noaa_commons.pro for details

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
panel_base=papco_WIDGET_BASE(title=aPlotInfo.panelKind, column=1, $
                             GROUP_LEADER=group, /MODAL)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionality that PAPCO supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by PAPCO_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)

;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

;--- make toggle bases for the channels of the different NOAA birds.-----------
ch_outer_base=WIDGET_BASE(panel_base, COLUMN=1, /align_center)
ch_base=WIDGET_BASE(ch_outer_base, /align_center)  

base_NOAA12 = WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names = canal_NOAA12 + ' ' +gammep_NOAA12 & n_ch=n_elements(energy_names)
energy_set = bytarr(n_ch) 
FOR i=0l,n_ch-1 DO $
  IF (aPlotInfo.ioptions(0) AND 2l^i) EQ 2l^i THEN energy_set(i)=1 $
    ELSE energy_set(i)=0
mpb_NOAA12 = CW_BGROUP(base_NOAA12, energy_names, COLUMN=4, /NONEXCLUSIVE, $
                       LABEL_TOP='Select energy channels (NOAA-12)', $
                       UVALUE='mpb_NOAA12', SET_VALUE=energy_set)    

base_NOAA15 = WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names = canal_NOAA15 + ' ' +gammep_NOAA15 & n_ch=n_elements(energy_names)
energy_set = bytarr(n_ch) 
FOR i=0l,n_ch-1 DO $
  IF (aPlotInfo.ioptions(1) AND 2l^i) EQ 2l^i THEN energy_set(i)=1 $
    ELSE energy_set(i)=0
mpb_NOAA15 = CW_BGROUP(base_NOAA15, energy_names, COLUMN=4, /NONEXCLUSIVE, $
                       LABEL_TOP='Select energy channels (NOAA-15)', $
                       UVALUE='mpb_NOAA15', SET_VALUE=energy_set)    

base_NOAA16 = WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names = canal_NOAA16 + ' ' +gammep_NOAA16 & n_ch=n_elements(energy_names)
energy_set = bytarr(n_ch) 
FOR i=0l,n_ch-1 DO $
  IF (aPlotInfo.ioptions(2) AND 2l^i) EQ 2l^i THEN energy_set(i)=1 $
    ELSE energy_set(i)=0
mpb_NOAA16 = CW_BGROUP(base_NOAA16, energy_names, COLUMN=4, /NONEXCLUSIVE, $
                       LABEL_TOP='Select energy channels (NOAA-16)', $
                       UVALUE='mpb_NOAA16', SET_VALUE=energy_set) 


;--- add the widgets for choosing an satellite----------------------------------
sat_base=WIDGET_BASE(panel_base, column=3, /frame)
lbl_satName=CW_FIELD(sat_base, title='Satellite ID:', $
                         value=aPlotInfo.filename)
lb_satNames=WIDGET_LIST(sat_base, ysize=5, xsize=12, $
                        UVALUE='lb_satNames')
label_list_base=WIDGET_LABEL(sat_base, $
                          value='satellites for the time chosen')

;--- buttons to chose other options --------------------------------------------
opti_base=WIDGET_BASE(panel_base, ROW=1, /FRAME)
  
average_names=['seg.','av.']
average_base=CW_BGROUP(opti_base, average_names, /FRAME, ROW=1, /EXCLUSIVE, $
                       UVALUE='spb_average', $
                       SET_VALUE=aPlotInfo.options(8))
  
reload_names=['Relaod', 'Adopt']
reload_base=CW_BGROUP(opti_base, reload_names, /FRAME, ROW=1, /EXCLUSIVE, $
                      UVALUE='spb_reload', $
                      SET_VALUE=aPlotInfo.options(9))

spec_names=['line', 'L-spec']
spec_base=CW_BGROUP(opti_base, spec_names, /FRAME, ROW=1, /EXCLUSIVE, $
                    UVALUE='spb_spec', $
                    SET_VALUE=aPlotInfo.options(10))

;--- widget to chose variable to select range for ------------------------------
vars_outer_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1) 
  
vars_names = ['select L', 'select MLT', 'select MagLat']
vars_base=CW_BGROUP(vars_outer_base, vars_names, ROW=1, /EXCLUSIVE, $
                    UVALUE='spb_vars', SET_VALUE=aPlotInfo.options(4))

;--- make sliders selecting ranges - not all of them mapped! -------------------
select_outer_base=WIDGET_BASE(vars_outer_base, COLUMN=1, /align_center)
select_base=WIDGET_BASE(select_outer_base, /align_center)
  
;--- sliders to select the L-range to be plotted -------------------------------
l_base=WIDGET_BASE(select_base, column=1, /align_center) 
  
l_rangeNames=['no L range', 'select L range']
l_range_base=CW_BGROUP(l_base, l_rangeNames, ROW=1, /EXCLUSIVE, $
                       UVALUE='spb_l_range', $
                       SET_VALUE=aPlotInfo.typevector(3) < 1)  
  
l_label_base=WIDGET_BASE(l_base, row=1, /align_center)
l_low_label= WIDGET_LABEL(l_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(2),format="(f4.1)"),35,/center), $
                             UVALUE='l_low_label')
l_high_label=WIDGET_LABEL(l_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(3),format="(f4.1)"),35,/center), $
                             UVALUE='l_high_label')

l_slider_base=WIDGET_BASE(l_base, row=1, /align_center)
; --- Low L-value --------------------------------------------------------------
l_low=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=300, $
                  title=strformat('Low L-value',35,/center), $
         	  VALUE=(((aPlotInfo.options(2) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_low', MINIMUM=0, MAXIMUM=110)	      
; --- High L-value -------------------------------------------------------------
l_high=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=300, $
                  title=strformat('High  L-value',35,/center), $
		  VALUE=(((aPlotInfo.options(3) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_high', MINIMUM=0, MAXIMUM=110)
  
;--- sliders to select the MLT-range to be plotted -----------------------------
mlt_base=WIDGET_BASE(select_base, column=1, /align_center) 
  
mlt_rangeNames=['no MLT range', 'select MLT range']
mlt_range_base=CW_BGROUP(mlt_base, mlt_rangeNames, ROW=1, /EXCLUSIVE, $
                           UVALUE='spb_mlt_range', $
                           SET_VALUE=aPlotInfo.options(7))
  
mlt_label_base=WIDGET_BASE(mlt_base, row=1, /align_center)
mlt_from_label= WIDGET_LABEL(mlt_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(2),format="(f4.1)"),35,/center), $
                             UVALUE='mlt_from_label')
mlt_to_label=WIDGET_LABEL(mlt_label_base, /align_center, $
VALUE=strformat(string(aPlotInfo.options(3),format="(f4.1)"),35,/center), $
                             UVALUE='mlt_to_label')    
  
mlt_slider_base=WIDGET_BASE(mlt_base, row=1, /align_center)
; --- from MLT-value --------------------------------------------------------
mlt_from=WIDGET_SLIDER(mlt_slider_base, /DRAG, /SUPPRESS_VALUE, xsize=300,$
                  title=strformat('From MLT',35,/center), $
         	  VALUE=(aPlotInfo.options(5) * 2) < 49 > 0, $
                  UVALUE='mlt_from', MINIMUM=0, MAXIMUM=48)	      
  
; --- to MLT-value ----------------------------------------------------------
mlt_to=WIDGET_SLIDER(mlt_slider_base, /DRAG, /SUPPRESS_VALUE, xsize=300, $
                  title=strformat('To MLT',35,/center), $
		  VALUE=(aPlotInfo.options(6) * 2) < 49 > 0, $
                  UVALUE='mlt_to', MINIMUM=0, MAXIMUM=48)  
   
;--- sliders to select the MLAT-range to be plotted ---------------------------
mlat_base=WIDGET_BASE(select_base, column=1, /align_center) 
                                
                                 ;still add in this option...
  
;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $PAPCO_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the widget is always reinitialized to its last state.

; The next four PAPCO panel widgets will be added only if you want to
; support that functionality, and if they make sense for your noaa_flux.

;--- add the  PAPCO panel widget for choosing the plot-style for line plots ---
widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widget for choosing the y-axis-ranges ----------------
widgets=PAPCO_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_panel_AddZScale_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the noaa_flux example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'noaa_flux_panel.help', $
                    base_NOAA12:base_NOAA12, $
                    base_NOAA15:base_NOAA15, $
                    base_NOAA16:base_NOAA16, $
                    mpb_NOAA12:mpb_NOAA12, $
                    mpb_NOAA15:mpb_NOAA15, $
                    mpb_NOAA16:mpb_NOAA16, $
                    typevector:aPlotInfo.typevector, $
                    options:aPlotInfo.options, $
                    ioptions:aPlotInfo.ioptions,$
                    l_base:l_base, $
                    l_low:l_low, $
                    l_high:l_high, $
                    l_low_label:l_low_label, $
                    l_high_label:l_high_label, $
                    mlt_base:mlt_base, $
                    mlt_from:mlt_from, $
                    mlt_to:mlt_to, $
                    mlt_from_label:mlt_from_label, $
                    mlt_to_label:mlt_to_label, $
                    mlat_base:mlat_base, $
                    satname:aPlotInfo.filename, $                    
                    lbl_satName:lbl_satName, $
                    lb_satNames:lb_satNames}
widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
PAPCO_Panel_Realize, panel_base, widgetData
WIDGET_CONTROL, panel_base, HOURGLASS=1
noaa_flux_refresh, /ALL 
WIDGET_CONTROL, panel_base, HOURGLASS=0
papco_xmanager, 'noaa_flux_panel', panel_base

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
returnData={typevector:widgetData.typevector, $
            options:WidgetData.options,$
            ioptions:WidgetData.ioptions,$
            filename:WidgetData.satname}
additionalRData=PAPCO_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO los_alamos_refresh, ALL=ALL, SATID=SATID, SATLIST=SATLIST
;* 
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      if set (e.g. /SATID), only the widgets influenced by the information
;*      that the keyword's name describes, are updated.
;*      If the keyword /ALL is set, all of the window is updated.
;*      
;*      SATELLITE ID
;*      ALL
;*
;* CALLING SEQUENCE:
;*      have a look the the event-routine los_alamos_Panel_event
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R. Friedel
;******************************************************************************
PRO noaa_flux_refresh, ALL=ALL, SATID=SATID, SATLIST=SATLIST, $
                       CHANNEL = CHANNEL, $
                       L_LIMIT=L_LIMIT, PLOT_STYLE = PLOT_STYLE, $
                       LOW_L=LOW_L, HIGH_L=HIGH_L, $ 
                       MLT=MLT, RANGE_MLT=RANGE_MLT
                   
COMMON noaa_flux_panel_data, WidgetData
COMMON noaa_list,satnames

IF KEYWORD_SET(ALL) THEN BEGIN
    SATID=1 & L_LIMIT=1 & LOW_L=1 & HIGH_L=1
    MLT=1 & RANGE_MLT=1
    SATLIST=1 & PLOT_STYLE=1
    CHANNEL = 1
ENDIF

IF KEYWORD_SET(SATLIST) THEN BEGIN
    satnames=noaa_satlist()
    WIDGET_CONTROL,WidgetData.lb_satNames, SET_VALUE=[satnames]
ENDIF

IF KEYWORD_SET(SATID) THEN BEGIN
    WIDGET_CONTROL, WidgetData.lbl_satName, SET_VALUE=WidgetData.satName
ENDIF

IF KEYWORD_SET(CHANNEL) THEN BEGIN
    bases = [widgetData.base_NOAA12, widgetData.base_NOAA15, $
             widgetData.base_NOAA16]
    CASE WidgetData.satName OF 
        'NOAA_N12': map = [1, 0, 0]
        'NOAA_N15': map = [0, 1, 0]
        'NOAA_N16': map = [0, 0, 1]
        ELSE: map = [0, 0, 0]
    ENDCASE
    FOR i = 0, n_elements(map)-1 DO widget_control, bases(i), MAP = map(i)
ENDIF

IF keyword_set(PLOT_STYLE) THEN BEGIN 
    bases = [widgetData.l_base, widgetData.mlt_base,widgetData.mlat_base]
    CASE widgetData.options(4) OF 
        0: map = [1, 0, 0]
        1: map = [0, 1, 0]
        2: map = [0, 0, 1]
    ENDCASE
    FOR i = 0, n_elements(map)-1 DO widget_control, bases(i), MAP = map(i)
ENDIF 

if keyword_set(L_LIMIT) then BEGIN
    bases = [widgetData.l_low, widgetData.l_high, $
             widgetData.l_low_label, widgetData.l_high_label]
    CASE widgetData.typeVector(3) OF
        0: sensitive = [0, 0, 0, 0]
        1: sensitive = [1, 1, 1, 1]
    ENDCASE
    FOR i = 0, n_elements(sensitive)-1 DO $
      WIDGET_CONTROL, bases(i), SENSITIVE=sensitive(i)
ENDIF

IF keyword_set(LOW_L) THEN BEGIN 
    high_l=widgetData.options(3) > (widgetData.options(2)+0.1)
    widgetData.options(3)=high_l
    WIDGET_CONTROL, widgetData.l_high, $
      SET_VALUE=(widgetData.options(3) -1) * 10
    
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)
ENDIF   

IF keyword_set(HIGH_L) THEN BEGIN 
    low_l=widgetData.options(2) < (widgetData.options(3)-0.1)
    
    widgetData.options(2)=low_l
    WIDGET_CONTROL, widgetData.l_low, $
      SET_VALUE=(widgetData.options(2) -1) * 10
    
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)    
ENDIF 

IF keyword_set(RANGE_MLT) THEN BEGIN 
    bases = [widgetData.mlt_from, widgetData.mlt_to, $
             widgetData.mlt_from_label, widgetData.mlt_to_label]
    CASE widgetData.options(7) OF
        0: sensitive = [0, 0, 0, 0]
        1: sensitive = [1, 1, 1, 1]
    ENDCASE
    FOR i = 0, n_elements(sensitive)-1 DO $
      WIDGET_CONTROL, bases(i), SENSITIVE=sensitive(i)
ENDIF

if keyword_set(MLT) then begin
    label=string(widgetData.options(5),format="(f4.1)")
    WIDGET_CONTROL, widgetData.mlt_from_label, $
      SET_VALUE=strformat(label,35,/center)
    label=string(widgetData.options(6),format="(f4.1)")
    WIDGET_CONTROL, widgetData.mlt_to_label, $
      SET_VALUE=strformat(label,35,/center)
ENDIF   

END 
;******************************************************************************
;* PROCEDURE:     
;*      PRO noaa_flux_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO noaa_flux_panel_event, event

COMMON noaa_flux_panel_DATA, widgetData
COMMON noaa_list,satnames
 
WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'PAPCO_panel_Event', which handles the events generated
; by the PAPCO-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

CASE uval OF

    'lb_satNames' : IF N_ELEMENTS(size(satnames)) GE 1 THEN BEGIN
         WidgetData.satName=satnames(event.index)
         noaa_flux_refresh, /SATID, /CHANNEL
    END 

    'mpb_NOAA12' : BEGIN
        WIDGET_CONTROL, widgetData.mpb_NOAA12, GET_VALUE = channel_event
        widgetData.ioptions(0)=0
        for i=0l,n_elements(channel_event)-1 do if channel_event(i) then $
           widgetData.ioptions(0)=widgetData.ioptions(0)+2l^i  
    END

    'mpb_NOAA15' : BEGIN
        WIDGET_CONTROL, widgetData.mpb_NOAA15, GET_VALUE = channel_event
        widgetData.ioptions(1)=0
        for i=0l,n_elements(channel_event)-1 do if channel_event(i) then $
           widgetData.ioptions(1)=widgetData.ioptions(1)+2l^i  
    END

    'mpb_NOAA16' : BEGIN
        WIDGET_CONTROL, widgetData.mpb_NOAA16, GET_VALUE = channel_event
        widgetData.ioptions(2)=0
        for i=0l,n_elements(channel_event)-1 do if channel_event(i) then $
           widgetData.ioptions(2)=widgetData.ioptions(2)+2l^i  
    END

   'spb_average'   : widgetData.options(8)=event.value
   'spb_reload'    : widgetData.options(9)=event.value
   'spb_spec'      : widgetData.options(10)=event.value

    'spb_vars' : begin
        widgetData.options(4)=event.value
        noaa_flux_refresh, /PLOT_STYLE
    end
    'spb_l_range' : begin
        widgetData.typeVector(3)=event.value
        noaa_flux_refresh, /L_LIMIT
    end  
    'spb_mlt_range' : begin
        widgetData.options(7)=event.value
        noaa_flux_refresh, /RANGE_MLT
    end  
    'l_low'    : begin
        widgetData.options(2)=event.value / 10.0 + 1.0
        noaa_flux_refresh, /LOW_L
    end
    'l_high'   : begin
        widgetData.options(3)=event.value / 10.0 + 1.0
        noaa_flux_refresh, /HIGH_L
    end  
    'mlt_from'    : begin
        widgetData.options(5)=event.value / 2.0 
        noaa_flux_refresh, /MLT
    end  
    'mlt_to'   : begin
        widgetData.options(6)=event.value / 2.0 
        noaa_flux_refresh, /MLT
    END    

    'pb_done' : BEGIN
        uval=''
        if WidgetData.satname eq '' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a satellite name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no satellite ID')
        endif else uval='pb_done'
    END

   'pb_done_and_cont' : BEGIN
        uval=''
        if WidgetData.satname eq '' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a satellite name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no satellite ID')
        endif else uval='pb_done_and_cont'
    END

    'pb_cancel' : BEGIN
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
    END

    ELSE : begin
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
    END

ENDCASE

if (uval eq 'pb_done') or (uval eq 'pb_done_and_cont') then $
            WidgetData=PAPCO_Panel_event(event,WidgetData)

END
