;******************************************************************************
;* FILE:
;*    $PAPCO_XX/PAPCO_templates/noaa_overview_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_XX/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by 
;*    $PAPCO_XX/papco_panel.pro (a collection of standard PAPCO widgets)
;*    
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ (under $PAPCO_XX/PAPCO_doc)
;*    papco_conventions.text - which conventions were used when developing ?
;*    papco_variables.text   - what structures and global variables
;                              are used by which file ?
;*
;* COMMON-BLOCKS:
;*    PAPCO_PANEL_TEMPLATE_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_XX/papco_panel.pro for a
;*    description). 
;*       {PAPCO_PANEL_TEMPLATE_DATA, 
;*	     ; imimi_chemsine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_DOC/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  PAPCO_PANEL_TEMPLATE, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*       a=PAPCO_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=noaa_overview_panel(a)
;*       which calls up the window.
;*       Interrogating a with help,a,/str shows the contents. This is a good
;*       way of testing your own panel editor. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;*     September 2000, modified by S. Bourdarie for cassini mimi_chems
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  noaa_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the PAPCO_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $PAPCO_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      switch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commchemste between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $PAPCO_XX/PAPCO_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  noaa_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON noaa_overview_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
       column=1, $
       GROUP_LEADER=group)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionality that PAPCO supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by PAPCO_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
;  widget to chose the plot 
;
   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=4) 
canal = ['0P1','0P2','0P3','0P4','0P5','0E1','0E2','0E3' $
        ,'90P1','90P2','90P3','90P4','90P5','90E1','90E2','90E3' $
	,'P6','P7','P8']
gammep = ['30 keV - 80 keV','80 keV - 240 keV','240 keV - 800 keV' $
        ,'800 keV - 2500 keV','> 2500 keV' $
	,'30 keV - 2500 keV','100 keV - 2500 keV','300 keV - 2500 keV' $
        ,'30 keV - 80 keV','80 keV - 240 keV','240 keV - 800 keV' $
        ,'800 keV - 2500 keV','> 2500 keV' $
	,'30 keV - 2500 keV','100 keV - 2500 keV','300 keV - 2500 keV' $
	,'> 16 MeV','> 35 MeV','> 70 MeV']


   typeNames=canal + ' ' +gammep
   X_base = WIDGET_BASE(type_base, /FRAME, column=5, /EXCLUSIVE) 


   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0,N_ELEMENTS(typeNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(X_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))
    WIDGET_CONTROL, spb_type(aPlotInfo.typevector(0)), SET_BUTTON=1
;
;--- now, add the widgets for choosing an satellite----------------------------
   sat_base=WIDGET_BASE(panel_base, column=3, /frame)
   lbl_satName=CW_FIELD(sat_base, $
                         title='Satellite ID:', $
                         value=aPlotInfo.filename)
   lb_satNames=WIDGET_LIST(sat_base, $
                        ysize=5, xsize=12, $
                        UVALUE='lb_satNames')
   label_list_base=WIDGET_LABEL(sat_base, $
                          value='satellites for the time chosen')
;
; --- Plot Switches -----------------------------------------------------------
   switch_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1)

   smooth_base=WIDGET_BASE(switch_base, /FRAME, ROW=1, /EXCLUSIVE)

   smoothNames=['no smooth', $
              'smooth vertically', $
              'smooth horizontally', $
              'smooth both']
   spb_smoo=lonarr(N_ELEMENTS(smoothNames))

   FOR i=0, N_ELEMENTS(smoothNames)-1 DO $
      spb_smoo(i)=WIDGET_BUTTON(smooth_base, $
			VALUE=smoothNames(i), $
			UVALUE='spb_smoo'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_smoo(aPlotInfo.ioptions(0)), SET_BUTTON=1

   fill_base=WIDGET_BASE(switch_base, /FRAME, ROW=1, /EXCLUSIVE)

   fillNames=['no fill', $
              'fill missing data']
   spb_fill=lonarr(N_ELEMENTS(fillNames))

   FOR i=0, N_ELEMENTS(fillNames)-1 DO $
      spb_fill(i)=WIDGET_BUTTON(fill_base, $
			VALUE=fillNames(i), $
			UVALUE='spb_fill'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_fill(aPlotInfo.ioptions(1)), SET_BUTTON=1
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $PAPCO_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the widget is always reinitialized to its last state.

; The next four PAPCO panel widgets will be added only if you want to
; support that functionality, and if they make sense for your noaa_overview.

;--- add the papco panel widget for choosing the color bar for color plots ----
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the noaa_overview example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpfile:'noaa_overview_panel.help', $
                    typevector:aPlotInfo.typevector, $
                    ioptions:aPlotInfo.ioptions, $
                    satname:aPlotInfo.filename, $                    
                    lbl_satName:lbl_satName, $
                    lb_satNames:lb_satNames, $
                    spb_type:spb_type}
   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData
   WIDGET_CONTROL, panel_base, HOURGLASS=1
   noaa_overview_panel_refresh, /ALL
   WIDGET_CONTROL, panel_base, HOURGLASS=0


; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'noaa_overview_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
   returnData={typevector:widgetData.typevector,$
                 ioptions:widgetData.ioptions,$
		 filename:WidgetData.satname}
   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO los_alamos_refresh, ALL=ALL, SATID=SATID, SATLIST=SATLIST
;* 
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      if set (e.g. /SATID), only the widgets influenced by the information
;*      that the keyword's name describes, are updated.
;*      If the keyword /ALL is set, all of the window is updated.
;*      
;*      SATELLITE ID
;*      ALL
;*
;* CALLING SEQUENCE:
;*      have a look the the event-routine los_alamos_Panel_event
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R. Friedel
;******************************************************************************
PRO noaa_overview_panel_refresh, ALL=ALL, SATID=SATID, SATLIST=SATLIST
                   
   COMMON NOAA_OVERVIEW_PANEL_DATA, WidgetData
   COMMON noaa_list,satnames



   IF KEYWORD_SET(ALL) THEN BEGIN
      SATID=1
      SATLIST=1
   ENDIF

   IF KEYWORD_SET(SATLIST) THEN BEGIN
       satnames=noaa_overview_satlist()
       WIDGET_CONTROL,WidgetData.lb_satNames, $
         SET_VALUE=[satnames]
   ENDIF

   IF KEYWORD_SET(SATID) THEN BEGIN
      WIDGET_CONTROL,WidgetData.lbl_satName, $
         SET_VALUE=WidgetData.satName
   ENDIF

END 


;******************************************************************************
;* PROCEDURE:     
;*      PRO noaa_overview_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO noaa_overview_panel_event, event

   COMMON noaa_overview_panel_DATA, widgetData
   COMMON noaa_list,satnames

   WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'PAPCO_panel_Event', which handles the events generated
; by the PAPCO-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

   IF strleft(uval, 8) EQ 'spb_type' THEN begin
      choice=fix(strfrom(uval, 8))
      uval = 'spb_type'
   endif
   if strleft(uval, 8) EQ 'spb_smoo' THEN BEGIN
      smooth_choice=fix(strfrom(uval, 8)) 
      uval=strleft(uval, 8)
   endif

   if strleft(uval, 8) EQ 'spb_fill' THEN BEGIN
      fill_choice=fix(strfrom(uval, 8)) 
      uval=strleft(uval, 8)
   endif

   CASE uval OF
      'spb_type' : BEGIN
        WidgetData.typevector(0)=choice
      END

      'lb_satNames' : IF N_ELEMENTS(size(satnames)) GE 1 THEN BEGIN
         WidgetData.satName=satnames(event.index)
         noaa_overview_panel_refresh, /SATID
      END 

      'spb_smoo' : BEGIN
          widgetData.ioptions(0)=smooth_choice
      END

      'spb_fill' : BEGIN
          widgetData.ioptions(1)=fill_choice
      END

     'pb_done' : BEGIN
        uval=''
        if WidgetData.satname eq '' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a satellite name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no satellite ID')
        endif else uval='pb_done'
      END

     'pb_done_and_cont' : BEGIN
        uval=''
        if WidgetData.satname eq '' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a satellite name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no satellite ID')
        endif else uval='pb_done_and_cont'
      END

      'pb_cancel' : BEGIN
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
      END

      ELSE : begin
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
      END

   ENDCASE

   if (uval eq 'pb_done') or (uval eq 'pb_done_and_cont') then $
            WidgetData=PAPCO_Panel_event(event,WidgetData)


END

