;******************************************************************************
;* file papcoadd_noaa
;*      papcoadd_noaa_eph
;*      papcoadd_noaa_flux
;*      papcoadd_noaa_lspec
;*      papcoadd_noaa_overview
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for noaa. 
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_noaa,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this noaa to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      noaas.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_noaa,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified September 2000, S. Bourdarie for cassini mimi
;******************************************************************************
pro addplottype_noaa,plottypes_base,buttonGeometry,pdm_noaa

;example for simple pull-down for sub-types from the noaa module:
MenuDesc = [ { CW_PDMENU_S,   3, strcenter('NOAA-TIROS',10) }, $ 
             { CW_PDMENU_S, 0, ' configure noaa-tiros ' }, $
             { CW_PDMENU_S, 0, ' ephemeris' }, $
             { CW_PDMENU_S, 0, ' flux' }, $
             { CW_PDMENU_S, 0, ' lspec (rf)' }, $
             { CW_PDMENU_S, 2, ' overview (seb)' } ]
pdm_noaa = CW_PDMENU(plottypes_base, MenuDesc, UVALUE='pdm_noaa')

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_noaa
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the PAPCO_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the noaa.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_noaa_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_noaa
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_noaa,event

common get_error, get_err_no, get_err_msg

; The case statement here needs an entry for every submenu type you
; added in  the first procedure in this file (addplottype_polar_cammice).
; For each submenu entry, you will need to specify the panelEditor to use.  
  
; The FIRST time you use this module you will have to configure it,
; i.e. set the data paths etc. Once you have done this PAPCO will write
; a polar_cammice.config file into your papco_user_XX/papco_modules_defaults
; directory. 
; So here we check if this polar_cammice.config file exists, and if not,
; we diallow the execution of any other option apart from "configure
; polar_cammice". In this way the user is forced to configure the module
; before using it the first time!
  
r_module_config, strlowcase('noaa'), config_data, msg
if (get_err_no ne 0) and (event.value ne 1) then begin
    message, 'You need to configure the module first', /cont
    papco_module_configure, 'noaa'
    return  
ENDIF 

CASE Event.Value OF 
    1: BEGIN
        papco_module_configure,'noaa'
    END 
    2: BEGIN
        panelKind='noaa_eph'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct()  ; returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]       ; set some tags to desired defaults
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
    END 
    3: BEGIN
        panelKind='noaa_flux'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct()  ; returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]       ; set some tags to desired defaults
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
    END
    4: BEGIN
        panelKind='noaa_lspec'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct()  ; returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]       ; set some tags to desired defaults
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
    END
    5: BEGIN
        panelKind='noaa_overview'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct()  ; returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]       ; set some tags to desired defaults
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
    END
    else: BEGIN 
        message,'Nothing defined for this submenu entry', /cont
        return
    END   
ENDCASE

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_noaa, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by noaa orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_noaa, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_noaa_eph, plottable
plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END

pro plottable_noaa_flux, plottable
plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END

pro plottable_noaa_lspec, plottable
plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END
pro plottable_noaa_overview, plottable
plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_noaa, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that polar_cammice is capable of
;*      returning data from it's plot routine into a standard PAPCO
;*      array structure. Used for the scratch pad utility module.
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_polar_cammice.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                                1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_noaa, canget
;*
;* MODIFICATION HISTORY:       
;*      written November 1997, Reiner Friedel
;******************************************************************************
pro canget_noaa_eph, canget
canget=  1      ; 1 -> plot routine can return just data 
                ; 0 -> feature disabled
END
pro canget_noaa_flux, canget
canget=  1      ; 1 -> plot routine can return just data 
                ; 0 -> feature disabled
END
pro canget_noaa_lspec, canget
canget=  1      ; 1 -> plot routine can return just data 
                ; 0 -> feature disabled
END
pro canget_noaa_overview, canget
canget=  1      ; 1 -> plot routine can return just data 
                ; 0 -> feature disabled
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_noaa, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this noaa. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_noaa, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_noaa_eph, plotInfo, descriptStr
descriptStr=descriptStr+'[eph]'
END 

pro descriptor_noaa_flux, plotInfo, descriptStr
descriptStr=descriptStr+'[flux]'
END

pro descriptor_noaa_lspec, plotInfo, descriptStr
descriptStr=descriptStr+'[lspec]'
END

pro descriptor_noaa_overview, plotInfo, descriptStr

canal = ['0P1','0P2','0P3','0P4','0P5','0E1','0E2','0E3' $
        ,'90P1','90P2','90P3','90P4','90P5','90E1','90E2','90E3' $
	,'P6','P7','P8']
gammep = ['30 keV - 80 keV','80 keV - 240 keV','240 keV - 800 keV' $
        ,'800 keV - 2500 keV','> 2500 keV' $
	,'30 keV - 2500 keV','100 keV - 2500 keV','300 keV - 2500 keV' $
        ,'30 keV - 80 keV','80 keV - 240 keV','240 keV - 800 keV' $
        ,'800 keV - 2500 keV','> 2500 keV' $
	,'30 keV - 2500 keV','100 keV - 2500 keV','300 keV - 2500 keV' $
	,'> 16 MeV','> 35 MeV','> 70 MeV']

typeNames=canal + ' ' +gammep 
descriptStr=descriptStr+'['+typeNames(plotInfo.typevector(0))+']'
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_noaa, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the PAPCo core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of PAPCO functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure PAPCO_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 switches. Currently
;*                      Three are supported (this may be extended later)
;*                        1 = mouse zooms (utility slice, always possible)
;*                        2 = mouse slices
;*                        4 = mouse calls level zero
;*                        8 = mouse views slice (polar)  
;*                       16 = mouse views ephemeris
;*                       32 = mouse writes paneldata (utility slice)
;*                       64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      draw_noaa, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_noaa_eph, panelVector, currPlotInfo, orbitNo, $
                                get_data_call, slice_type, IsRead, $
                                OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

;ephemeris info is in flux files ".dat" files. Use that.
get_data_call=''

; set up which mouse functions to allow
slice_type=1+8+16

adopt = currPlotInfo.options(9)
IF adopt THEN IsRead = IsRead ELSE IsRead=0

IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_noaa_flux, currPlotInfo

    ; check the read for success.
    IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1 
ENDIF ELSE message, 'Adopting data in memory', /cont

IF IsRead THEN BEGIN
    p_noaa_eph, panelVector, currPlotInfo, OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo
ENDIF

END 

;******************************************************************************
pro draw_noaa_flux, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead, $
                                 OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

get_data_call=''

; set up which mouse functions to allow
slice_type=1+8+16

adopt = currPlotInfo.options(9)
IF adopt THEN IsRead = IsRead ELSE IsRead=0

style = currPlotInfo.options(10) ;0 - line plor, 1 - lspec plot
CASE style OF
    0:plot_call = 'p_noaa_flux_line'
    1:plot_call = 'p_noaa_lspec'
ENDCASE 

IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_noaa_flux, currPlotInfo
    ; check the read for success.
    IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1 
ENDIF ELSE message, 'Adopting data in memory', /cont

IF IsRead THEN call_procedure, plot_call, $
  panelVector, currPlotInfo, OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo

END

;******************************************************************************
pro draw_noaa_lspec, panelVector, currPlotInfo, orbitNo, $
                                  get_data_call, slice_type, IsRead, $
                                  OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

get_data_call=''

; set up which mouse functions to allow
slice_type=1+8+16

adopt = currPlotInfo.options(9)
IF adopt THEN IsRead = IsRead ELSE IsRead=0

IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_noaa_lspec, currPlotInfo
    ; check the read for success.
    IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1 
ENDIF ELSE message, 'Adopting data in memory', /cont

IF IsRead THEN p_noaa_lspec, panelVector, currPlotInfo, $
  OUTPUT=OUTPUT, /NO_MAKE, _EXTRA=extra_PlotInfo

END

;******************************************************************************
pro draw_noaa_overview, panelVector, currPlotInfo, orbitNo, $
                        get_data_call, slice_type, IsRead, $
                        OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure
COMMON PLOT_COMPOSER, widgetData


; set up which mouse functions to allow
slice_Type=1+2+8+16+PAPCO_slice_value('USER multiple Vert Slices')$
                +PAPCO_slice_value('USER multiple Horiz Slices')$
		+PAPCO_slice_value('USER ISOMAP')

result = strpos(currPlotInfo.filename,'done')
if result eq -1 then sat = currPlotInfo.filename $
else begin
    sat = strmid(currPlotInfo.filename,0,result)
    widgetData.plotInfos(panelVector(0)).filename=sat
endelse

get_data_call='r_noaa_overview,'+"'"+sat+"'"+','+string(currPlotInfo.typeVector(0))   ; data read call used by slices

IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_noaa_overview,sat,currPlotInfo.typeVector(0)
    ; check the read for success.
    IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1 
ENDIF

IF IsRead THEN BEGIN
    p_noaa_overview_spec, panelVector, sat, $
      currPlotInfo.typeVector(0), currPlotInfo.ioptions, $
      OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo
    IsRead = 0
ENDIF

END

