;******************************************************************************
;* PROCEDURE:     
;*      p_noaa_eph, panel, PlotInfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      PlotInfo    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_noaa_eph_line, panel, PlotInfo, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_noaa_eph, panel, PlotInfo, OUTPUT=OUTPUT, OVERPLOT=OVERPLOT, $
                PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
COMMON yscale, yscl                             ;man/auto yscaling
COMMON shift_label, down                        ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  ;info on coords

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_noaa_eph. 
COMMON noaa_flux_data, input_header, input_data
COMMON noaa_eph_tick, time, yray, tick_plotinfo
COMMON noaa_all

COMMON papco_color_names
colors = [red, blue, dark_green, black]
  
IF keyword_set(OUTPUT) THEN output = OUTPUT ELSE output = 0

;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1

sat = PlotInfo.filename
mag =  plotinfo.ioptions(2)
type = plotinfo.typevector(1) 
L_select = PlotInfo.typevector(3)
L_range = [PlotInfo.options(2), PlotInfo.options(3)]
MLT_select = PlotInfo.options(7)
MLT_range = [PlotInfo.options(5), PlotInfo.options(6)]
average = PlotInfo.options(8)

utitle='noaa_eph '+sat

;time between two points (needed not to plot missing data)
tmin = 12.0

;work out the structure index for the mag coords chosen!
;by now they "Should" exist...
intmod = PlotInfo.ioptions(0)
extmod = PlotInfo.ioptions(1)

new_struct_name='MODEL_'+varprt(intmod)+'_'+varprt(extmod)
new_struct=create_struct(name='UNILIBR', 'L',0.0,'MLT',0.0,'MLAT',0.0, $
                           'B_GSM', fltarr(4)) 
tags=tag_names(input_data(0))
index=where(tags eq new_struct_name, c)
IF c EQ 0 THEN message, 'Can not find mag coords in data'
new_idx=index(0)

;select data to plot
Re=6371.0
;set bad data flag
nodata = -999.9

IF mag EQ 0 THEN BEGIN          ;ephemeris coords

    utitle=utitle+'!C'+eph_names(PlotInfo.typevector(1))

    CASE type OF 
        0:BEGIN
            yray = data.ALT
        END 
        1:BEGIN
            yray = data.LONG
        END 
        2:BEGIN
            yray = data.LAT
        END 
        3:BEGIN                 ;GEO local time calc here
            ut_hours = (time /3660.0) MOD 24.0
            yray = (ut_hours + data.LONG/15.0) MOD 24
        END 
        4:BEGIN
            yray = data.(new_idx).L
            idx = where(yray LE 0, c)
            IF c NE 0 THEN yray(idx) = nodata
            utitle=utitle+'!C'+new_struct_name
        END 
        5:BEGIN
            yray = data.(new_idx).MLT
            utitle=utitle+'!C'+new_struct_name
        END 
        6:BEGIN
            yray = data.(new_idx).MLAT
            utitle=utitle+'!C'+new_struct_name
        END 
    ENDCASE 

    ;IF c NE 0 THEN yray(idx) = nodata
    uytitle = eph_yunits(PlotInfo.typevector(1))

ENDIF ELSE BEGIN                ;model mag field coords
    utitle=utitle+'!CMODEL_0_5!CMag. Field, GSE'
    uytitle = 'nT'
    yray = data.(new_idx).B_GSM
    ;IF c NE 0 THEN yray(*, idx) = nodata
ENDELSE    

rst = ''  ;right side lable for L. MLT, etc. strings

IF L_select then BEGIN       ; select L-range if that is selected
    noaa_l_select, yray, L_range, nodata,  model = new_struct_name
    rst = rst+string(L_range(0), L_range(1), $
                   format="(f4.1,' < L < ',f4.1)")+'!C'
ENDIF

IF MLT_select then BEGIN   ; select LT-range if is selected 
    noaa_mlt_select, yray, MLT_range, nodata,  model = new_struct_name
    rst = rst+string(MLT_range(0), MLT_range(1), $
                     format="('LT ',f4.1,' to ',f4.1)")+'!C'
ENDIF
 
IF output EQ 2 THEN return

panelset,panel	         		;sets the panel position viewport

if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else begin
    papco_ylim, time, yray, yst, yen, LOG=yscl(panel(0),3), BAD_DATA=nodata
    yr = yen-yst
    yst = yst-yr*0.05 & yen = yen+yr*0.05
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
endelse

; use auto y-labeling routine to get "good" y-lables
papco_y_label, yst, yen, LOG=yscl(panel(0),3)  

extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
               ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

; now do the actual data plot, data only, no axis
IF mag EQ 0 THEN BEGIN          ;ephemeris coords
    noaa_gap_oplot, tmin, time, yray, NODATA = NODATA, AVERAGE=average, $
      _extra=extra_plotPar
   ;papco_gap_plot,time, yray, nodata, _extra=extra_plotPar, xstyle=5, ystyle=5
ENDIF ELSE BEGIN 
    coords = plotinfo.ioptions(2) & set = bytarr(4)
    FOR i = 0, 3 DO IF (coords and 2ll^i) eq 2ll^i THEN set(i)=1 else set(i)=0
    idx2 = where(set EQ 1, n_ch) 
    IF n_ch EQ 0 THEN GOTO, noplot
    in_label = ['X (gsm)', 'Y (gsm)', 'Z (gsm)', 'Magnitude']
    in_label = in_label(idx2)
    for i = 0, n_ch-1 DO BEGIN
        IF n_ch NE 1 then extra_plotPar.color = colors(i)
        papco_gap_plot, time, yray(idx2(i), *), nodata, $
              xstyle=5, ystyle=5, _extra=extra_plotPar
    ENDFOR 
    ; plot channel info inside of panel.
    papco_in_side_label, panel, in_label, NCOL = colors, _extra=extra_plotPar
ENDELSE

; "down" sets the axis labeling level. 
down=0 

; now do the actual data plot, data only, no axis
;plot,time, yray, _extra=extra_plotPar, xstyle=5, ystyle=5

; draw y-axis label
left_side_label, panel, uytitle, /ROT90

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; down sets the axis labeling level
if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin

endif

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
if not keyword_set(overplot) then begin
    right_side_label,panel,utitle,_extra=extra_Par
endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
endelse
   
;right side labels for L, MLT, etc. ranges
right_side_label,panel, rst, POS=1,_extra=extra_Par

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
if KEYWORD_SET(OUTPUT) then begin
    print,'% p_noaa_eph_line: Writing plot data out to file'
    description='noaa_eph template sample data - single energy channel line'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
ENDIF   

return

;jump to here if no data to plot 
noplot: 
message,'No data to plot', /cont
plotted_x = !x  &  plotted_y = !y

end 
