;******************************************************************************
;* PROCEDURE:     
;*      p_noaa_flux_line, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mimi_chems.
;*              see p_crres_mimi_chems for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_noaa_flux_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written September 2000, Sebastien Bourdarie, at ONERA/DESP
;******************************************************************************
pro p_noaa_flux_line, panel, PlotInfo, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; The following common blocks are needed:

COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
COMMON papco_color_names

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_noaa_flux. 
COMMON noaa_flux_data, input_header, input_data
COMMON noaa_all ;commons for module - see m_noaa_commons.pro for details
COMMON time, xut1,xut2

FORWARD_FUNCTION mlt_average
IF keyword_set(OUTPUT) THEN output=OUTPUT else output=0

sat = PlotInfo.filename
channel = PlotInfo.ioptions
L_select = PlotInfo.typevector(3)
L_range = [PlotInfo.options(2), PlotInfo.options(3)]
MLT_select = PlotInfo.options(7)
MLT_range = [PlotInfo.options(5), PlotInfo.options(6)]
average = PlotInfo.options(8)

; make color array
couleur_arr = [black, red, green, yellow, blue, magenta, cyan, $
               white, burgundy, olive, dark_green, teal, $
               royal_blue, violet, dark_grey, grey]

; restrict data to actual time range requested - this makes for faster zooming
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
IF c NE 0 THEN data=input_data(index) ELSE BEGIN
    message, 'No data in timerange', /cont & GOTO, noplot
ENDELSE 
xut1=0  &  xut2=tai_2-tai_1  &  data.time=data.tai-tai_1 & time=data.time

;time between two points (needed not to plot missing data)
tmin = 12.0

CASE sat OF
    'NOAA_N12': BEGIN
        channel = PlotInfo.ioptions(0)
        typeNames=canal_NOAA12+ ' ' +gammep_NOAA12
    END 
    'NOAA_N15': BEGIN
        channel = PlotInfo.ioptions(1)
        typeNames=canal_NOAA15+ ' ' +gammep_NOAA15
    END 
    'NOAA_N16': BEGIN
        channel = PlotInfo.ioptions(2)
        typeNames=canal_NOAA16+ ' ' +gammep_NOAA16    
    END 
ENDCASE

utitle  = sat+'-sem!C'
uytitle = 'counts/s'
nodata  = -999.9

; make index of channel to be plotted - from binary number in "channel"
n_ch = n_elements(typeNames) & to_plot=bytarr(n_ch)
FOR i=0l,n_ch-1 DO IF (channel AND 2l^i) EQ 2l^i THEN to_plot(i)=1
plot_index=where(to_plot EQ 1, nplot)
IF nplot EQ 0 THEN BEGIN 
    message, 'No channels selected', /cont  &  GOTO, noplot
ENDIF
utitle1 = typeNames

; select plot channel
yray = data.data(plot_index)
idx = where(yray LE 0, c)
IF c NE 0 THEN yray(idx) = nodata

rst = ''  ;right side lable for L. MLT, etc. strings

IF L_select then BEGIN       ; select L-range if that is selected
    noaa_l_select, yray, L_range, nodata
    rst = rst+string(L_range(0), L_range(1), $
                   format="(f4.1,' < L < ',f4.1)")+'!C'
ENDIF

IF MLT_select then BEGIN   ; select LT-range if is selected 
    noaa_mlt_select, yray, MLT_range, nodata
    rst = rst+string(MLT_range(0), MLT_range(1), $
                     format="('LT ',f4.1,' to ',f4.1)")+'!C'
ENDIF

IF output EQ 2 THEN return

panelset,panel	         		;sets the panel position viewport
   
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else BEGIN
    papco_autorange, yray, yst, yen, $
      log=yscl(panel(0),3), exclude=0, nodata=nodata
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
endelse

extra_plotPar = {yrange:[yst,yen], xrange:[xut1,xut2], $
                 ylog:yscl(panel(0),3), xtickformat:'noticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)

; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    

couleur_start = extra_plotPar.color
couleur_index = where(couleur_start eq couleur_arr)
IF couleur_index(0) eq -1 then couleur_index(0) = 0
couleur = extra_plotPar.color
IF nplot EQ 1 THEN BEGIN 
    noaa_gap_oplot, tmin, time, yray, NODATA = NODATA, AVERAGE=average, $
      _extra=extra_plotPar
ENDIF ELSE BEGIN 

    FOR i = 0, nplot-1 DO BEGIN 
        yvect = yray(i, *)
        extra_plotPar.color = couleur
        noaa_gap_oplot, tmin, time, yvect, NODATA = NODATA, AVERAGE=average, $
          _extra=extra_plotPar
        couleur = couleur_arr(couleur_index(0)+i+1)
    ENDFOR 

ENDELSE 

; store the coordinate information into common block for slices
plotted_x = !x & plotted_y = !y

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label, panel, uytitle, /rot90 

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION

   extra_Par.color = couleur_start
   couleur = extra_Par.color
   if not keyword_set(overplot) then begin
     extra_Par.color = 1
     right_side_label,panel,utitle,_extra=extra_Par
     utitle2 = ''
     for i=0,nplot-1 do begin
        extra_Par.color = couleur
        utitleS = utitle2 + '!C' + utitle1(plot_index(i))
        right_side_label,panel,utitleS,_extra=extra_Par
        utitle2 = utitle2 + '!C'
        couleur = couleur_arr(couleur_index(0)+i+1)
     endfor
   endif else begin
     for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle1
     right_side_label,panel,utitle,_extra=extra_Par
   endelse

   if not keyword_set(overplot) then begin
     extra_Par.color = 1
     right_side_label,panel,utitle,_extra=extra_Par
   endif else begin
     for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
     right_side_label,panel,utitle,_extra=extra_Par
   endelse

;right side labels for L, MLT, etc. ranges
right_side_label,panel, rst, POS=1,_extra=extra_Par

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  

if KEYWORD_SET(OUTPUT) then BEGIN 
    print,'% p_noaa_flux_line: Writing plot data out to file'
    print,'% NOT IMPLEMENTED !!'
ENDIF 
   
return

noplot: 
plotted_x = !x  &  plotted_y = !y

END  


;******************************************************************************
;* PROCEDURE:     
;*      ylim_io,time,yray,yst,yen
;* 
;* DESCRIPTION:  
;*      This procedure search for min and max in a log scale in array yray
;*
;* INPUTS:  
;*      time     
;*      yray     array to be plotted
;*              
;* OUTPUTS: 
;*      yst,yen   min and max
;* 
;* CALLING SEQUENCE:
;*      ylim_io,time,yray,yst,yen
;*
;* MODIFICATION HISTORY:       
;*     written September 2000, Sebastien Bourdarie, at ONERA/DESP
;******************************************************************************
pro ylim_io,time,yray,yst,yen

common time, xut1,xut2

index=where((time ge xut1) and (time le xut2) and (yray gt 0.),count)

if count gt 2 then begin
    yen=max(yray(index),min=yst)
    yst = 10.^floor(ALOG10(yst)) & yen = 10.^ceil(ALOG10(yen))
endif else begin
      yen=10. & yst=1.
endelse

end


;******************************************************************************
;* PROCEDURE:     
;*      noaa_flux_gap_oplot, x, y, _EXTRA=e
;* 
;* DESCRIPTION:  
;*      This procedure plots data without connecting data point if they are
;*      further than 5 min. apart. Also ignores nodata intervals
;*      if average is set, net data between nodata intervals are
;*      averaged togther and plotted at center time
;*
;* INPUTS: 
;*      step    time step between two measurements      
;*      x,y     arrays to be plotted
;*              
;* OUTPUTS: 
;*      none
;* 
;* CALLING SEQUENCE:
;*      noaa_flux_gap_oplot, x, y, _EXTRA=e
;*
;* MODIFICATION HISTORY:       
;*     written September 2000, Sebastien Bourdarie, at ONERA/DESP
;******************************************************************************
pro noaa_gap_oplot, step, x, y, NODATA = NODATA, AVERAGE = AVERAGE, $
                         _EXTRA=e

; now do the actual data plot, data only, no axis
plot, x, y, xstyle = 5, ystyle = 5, /nodata, _extra=e

; make structure for gap plot
extra_plotPar = create_struct({PSYM:0}, {SYMSIZE:0}, {COLOR:0L}, $
                              {THICK:1}, {LINESTYLE:0})
STRUCT_ASSIGN, e, extra_plotPar

; work out jumps in y. If a jump is larger than step, don't plot
ndat=n_elements(x)

n=0 & sn=0 & sd=0 & st=0 & c=0
FOR i = 0l, ndat-1 DO BEGIN
    IF y(i) eq  NODATA then BEGIN
        IF keyword_set(AVERAGE) then BEGIN
            if c ne 0 then begin
                if sn eq 0 then begin
                    plots, st/c, sd/c, _EXTRA=extra_plotPar
                    sn=sn+1
                endif else begin
                    plots, st/c, sd/c, _EXTRA=extra_plotPar, /CONTINUE, NOCLIP=0
                endelse  
            endif  
        endif  
        n=0 & sd=0 & st=0 & c=0
        goto,next
    ENDIF   
    ; got here if first data point found
    if n eq 0 then BEGIN 
        if keyword_set(AVERAGE) then BEGIN 
            st=st+x(i) & sd=sd+y(i)
            c=c+1
        endif else BEGIN  
            plots, x(i), y(i), _EXTRA=extra_plotPar
        ENDELSE   
        n=n+1
    endif else BEGIN  
        if keyword_set(AVERAGE) then begin
            st=st+x(i) & sd=sd+y(i)
            c=c+1        
        endif else BEGIN  
            plots, x(i), y(i), _EXTRA=extra_plotPar, /CONTINUE, NOCLIP=0
        ENDELSE   
    ENDELSE   
    next:                        
ENDFOR 

END 

;------------------------------------------------------------------------------
; routine to limit data in L range. mark everything outside as nodata
PRO noaa_l_select, yray, L_range, nodata,  MODEL = MODEL
  
COMMON noaa_flux_data, input_header, input_data

IF keyword_set(MODEL) THEN model = MODEL ELSE model ='MODEL_0_0'

tags = tag_names(input_data)
idx = where(tags EQ model, c)

IF c EQ 0 THEN message, 'Mag coords not found'

L_igrf = input_data.(idx(0)).L

index= where( (L_igrf LT L_range(0)) OR (L_igrf GT L_range(1)), c)

IF c ne 0 THEN yray(index) = nodata

END  

;------------------------------------------------------------------------------
; routine to limit data in MLT range. mark everything outside as nodata
pro noaa_mlt_select, yray, mlt_range, nodata,  MODEL = MODEL
    
COMMON noaa_flux_data, input_header, input_data

IF keyword_set(MODEL) THEN model = MODEL ELSE model ='MODEL_0_0'

tags = tag_names(input_data)
idx = where(tags EQ model, c)

IF c EQ 0 THEN message, 'Mag coords not found'

MLT_igrf = input_data.(idx(0)).MLT
lt_from = mlt_range(0) & lt_to = mlt_range(1)

IF lt_to EQ lt_from THEN return 
IF lt_from LT lt_to THEN $ 
    index= where( (MLT_igrf LT lt_from) OR  (MLT_igrf GT lt_to) ,c) $
ELSE $
    index= where( (MLT_igrf GT lt_to) AND (MLT_igrf LT lt_from) ,c)
  
IF c NE 0 THEN yray(*, index) = nodata
      
END  
