;******************************************************************************
;* PROCEDURE:     
;*     p_lanl_gps_lspec, panel, type, OUTPUT=OUTPUT, $
;*                       OVERPLOT=OVERPLOT,$
;*                       PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*                       SUBTABLE=SUBTABLE, _EXTRA=extra_par
;* 
;* MODIFICATION HISTORY:       
;*     written November 2002, Reiner Friedel, at LANL
;******************************************************************************
pro p_noaa_lspec, panel, plotinfo, OUTPUT=OUTPUT, NO_MAKE = NO_MAKE, $
                  OVERPLOT=OVERPLOT, $
                  PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                  SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end       ;common time limit in mjdt
COMMON yscale, yscl		        ;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
COMMON shift_label, down                ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_noaa_flux. 
COMMON noaa_flux_data, input_header, input_data
COMMON noaa_lspec_data, lspec_header, lspec_data
COMMON noaa_slice1, var_name, sat, time, yray, data
COMMON noaa_all ;commons for module - see m_noaa_commons.pro for details

FORWARD_FUNCTION mlt_average
IF keyword_set(OUTPUT) THEN output=OUTPUT else output=0

; restrict data to actual time range requested - this makes for faster zooming
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

;setup variables from panel editor.
sat = PlotInfo.filename
L_select = PlotInfo.typevector(3)
L_range = [PlotInfo.options(2), PlotInfo.options(3)]
MLT_select = PlotInfo.options(7)
MLT_range = [PlotInfo.options(5), PlotInfo.options(6)]
average = PlotInfo.options(8)

CASE sat OF
    'NOAA_N12': BEGIN
        channel = PlotInfo.ioptions(0)
        typeNames=canal_NOAA12+ ' ' +gammep_NOAA12
    END 
    'NOAA_N15': BEGIN
        channel = PlotInfo.ioptions(1)
        typeNames=canal_NOAA15+ ' ' +gammep_NOAA15
    END 
    'NOAA_N16': BEGIN
        channel = PlotInfo.ioptions(2)
        typeNames=canal_NOAA16+ ' ' +gammep_NOAA16    
    END 
ENDCASE

utitle  = sat+'-sem!C'
uytitle = 'L-value' 
uztitle = 'counts/s'
nodata  = -999.9  

; make index of channel to be plotted - from binary number in "channel"
; use first set channel only for Lspec plots
n_ch = n_elements(typeNames) & to_plot=bytarr(n_ch)
FOR i=0l,n_ch-1 DO IF (channel AND 2l^i) EQ 2l^i THEN to_plot(i)=1
plot_index=where(to_plot EQ 1, nplot)
IF nplot EQ 0 THEN BEGIN 
    message, 'No channels selected', /cont  &  GOTO, noplot
ENDIF
var_name = typeNames(plot_index(0))
utitle = utitle+var_name

IF keyword_set(NO_MAKE) THEN BEGIN ;data is in read files, do not male lspec
    
    index=where((lspec_data.tai(0) ge tai_1) AND (lspec_data.tai(1) le tai_2),c)
    IF c NE 0 THEN data=lspec_data(index) ELSE BEGIN
        message, 'No data in timerange', /cont & GOTO, noplot
    ENDELSE 

    xut1=0  &  xut2=tai_2-tai_1

    time=transpose(data.tai) - tai_1
    zmat =  transpose(data.data)
    yarr = lspec_header.yarr

ENDIF ELSE BEGIN 

    index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
    IF c NE 0 THEN data=input_data(index) ELSE BEGIN
        message, 'No data in timerange', /cont & GOTO, noplot
    ENDELSE 

    xut1=0  &  xut2=tai_2-tai_1  &  data.time=data.tai-tai_1 & time=data.time

    ; select plot channel
    yray = data.data(plot_index(0))
    idx = where(yray LE 0, c)
    IF c NE 0 THEN yray(idx) = nodata

    rst = ''  ;right side lable for L. MLT, etc. strings

    IF L_select then BEGIN       ; select L-range if that is selected
        noaa_l_select, yray, L_range, nodata
        rst = rst+string(L_range(0), L_range(1), $
                       format="(f4.1,' < L < ',f4.1)")+'!C'
    ENDIF

    IF MLT_select then BEGIN   ; select LT-range if is selected 
        noaa_mlt_select, yray, MLT_range, nodata
        rst = rst+string(MLT_range(0), MLT_range(1), $
                         format="('LT ',f4.1,' to ',f4.1)")+'!C'
    ENDIF

    ;now make the L-arrays to plot
    noaa_make_l_array, yarr, t_cut, zmat, nodata
    time=t_cut

ENDELSE 

; check if we need to do plot or just return data!.
IF output EQ 2 THEN begin
    message,'data constructed', /cont  &  return
ENDIF      

panelset,panel

;set z-scaling
if (zscl(panel(0),0) eq 1) then begin
    zst=zscl(panel(0),1)  &  zen=zscl(panel(0),2)
endif else begin
    papco_autorange, yray, zst, zen, $
      log=zscl(panel(0),3), exclude=2, nodata=-99 
    zscl(panel(0),1)=zst  &  zscl(panel(0),2)=zen
endelse   

;set y-scaling
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
endif else begin
    yst=1.0 & yen=12.0 
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
endelse         

; Overriding values by setting keywords is possible and allowed but discouraged
; since then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zst,zen], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
               ztitle:uztitle, $
               xtickformat:'noticks', ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)

; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar
   
; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
;resx, resy are in data coordinates. If the axis is logarithmic, 
;resx, resy need to be too.
resx=1000  &  resy=0
papco_plot_colorspec, zmat, time, yarr, nodata=nodata, resx=resx, resy=resy, $
                      xstyle=5, ystyle=5, _extra=extra_plotPar
papco_color_bar, _extra=extra_plotPar

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel,uytitle,/rot90 
; plot label at right of plot. Use scalable routine!    
right_side_label, panel, utitle, /ROT90, _extra=extra_Par

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
IF KEYWORD_SET(OUTPUT) THEN BEGIN
    time = time+tai_1
    message,'Writing plot data out to file',/cont
    description='noaa data'
    time_label='TAI'
    channels=channels  & n_ch=n_elements(channels)
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=-99
    n_bins=n_elements(l_arr) & lbin=0.1 & yarr=fltarr(n_bins,2)
    yarr(*,0)=l_arr(*)-lbin/2 & yarr(*,1)=l_arr(*)+lbin/2  
    t = (time(*, 0)+time(*, 1))/2.0
    papco_write_data, t, zarr, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, z_label=z_label, nodata=nodata, $
                      RS_LABEL=RS_LABEL,channels= channels
ENDIF   
      
return

noplot: 
plotted_x = !x  &  plotted_y = !y

end 

;------------------------------------------------------------------------------
; Make Time versus L array for data, splitting up per pass through rad
; belt.
; If SAVE is set (used for monthly data files) then save to file
PRO noaa_make_l_array, yarr, t_cut, zarr, nodata,  SAVE = SAVE

COMMON mjdt, mjdt_start, mjdt_end       ;common time limit in mjdt
COMMON noaa_slice1, var_name, sat, time, yray, data

ndat=n_elements(time)

;get L info. L's are negative when very large / undefined.
lv = data.MODEL_0_0.L
idx = where(lv LE 0, c)
IF c NE 0 THEN lv(idx) = 999.9

;get mlt info
maglt = data.MODEL_0_0.MLT

;select number of L cuts. NOAA period is 105 min, about 26 min per L-cut.
;i.e. one cut every 1500 seconds or so to be safe. 
n_cuts=long((time(ndat-1)-time(0)) / 1500.0)
;select L-bins 
lbin=0.2 & lrange=[1.0, 12.0]
n_bins= ( lrange(1) - lrange(0) ) / lbin
yarr = fltarr(n_bins, 2)
yarr(*, 0) = (findgen(n_bins) * lbin) + lrange(0)
yarr(*, 1) = (findgen(n_bins) * lbin) + lrange(0)+lbin

l_arr=(yarr(*, 0)+yarr(*, 1))/2.0

;make Lspec arrays. 
zarr=fltarr(n_cuts,n_bins)  &  t_cut=dblarr(n_cuts, 2)
mlt_av = fltarr(n_cuts)
  
message,'Making L v time array (slow)',/cont

;split up the data into L-cuts and populate zarr, t_cut
apo_idx=0l & per_idx=0l & n_cut=0l

FOR i=2l,ndat-3l DO BEGIN 
    ;check for perigee
    if ( lv(i-2) GE lv(i) ) AND ( lv(i) LE lv(i+2) ) AND $
       ( lv(i) LT 1.2) THEN per_idx=i          
    ;check for apogee
    if ((lv(i-2) LE lv(i) ) AND ( lv(i) GE lv(i+2))) THEN apo_idx=i
    
    ;if a cut is found, set the indices
    if per_idx*apo_idx ne 0 then begin
        ;print, '->',n_cut, per_idx, lv(per_idx), apo_idx, lv(apo_idx), $
        ;  abs(apo_idx-per_idx)
        if per_idx lt apo_idx then begin
            from=per_idx & to=apo_idx & per_idx=0 
        endif else begin
            from=apo_idx & to=per_idx & apo_idx=0
        ENDELSE

        ;check for time range of cut
        IF time(to)-time(from) GT 1500 THEN continue

        ;fold data from l_range into L-bins. Preserve bad data flags!

        ;first take out bad L's, bad data
        data_l=lv(from:to)  &  data_c=yray(from:to) & data_mlt = maglt(from:to)
        idx = where( (data_l NE 999.9) AND (data_c GT 0), c)
        IF c NE 0 THEN BEGIN
            data_l = data_l(idx)
            data_c = data_c(idx)
        ENDIF

        ;make L-indices of data_l accorning to binning required
        l_idx = fix(((data_l-lrange(0))/lbin) < (n_bins-1))
        
        ;now loop though all possible L-bins and average together
        FOR j = 0, n_bins-1 DO BEGIN
            idx = where(l_idx EQ j, c)
            IF c NE 0 THEN zarr(n_cut, j) = total(data_c(idx))/c $
              ELSE zarr(n_cut, j) = nodata
        ENDFOR     

        t_cut(n_cut, 0) = time(from)
        t_cut(n_cut, 1) = time(to)
    
        ;increment cut counter
        n_cut=n_cut+1  
    ENDIF 
ENDFOR  

zarr=zarr(0:n_cut-1,*)
t_cut=t_cut(0:n_cut-1, *)

save = 1

IF keyword_set(SAVE) THEN BEGIN 

    COMMON noaa_lspec_data, input_header, input_data

    ; check env. variable pointing to data directory
    if not (papco_check_data_env('NOAA_DATA', PATH=path)) then return    
   
    tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})  
    t = TAI2UTC(tai_1, /EXTERNAL )
    date = string(t.YEAR, t.MONTH, t.DAY, format = "(i4.4,i2.2,i2.2)")
    year = strmid(date, 0, 4)

    r = strsplit(var_name, ' ', /extract) & var = r(0)
    filename = path+sat+'/sem/'+year+'/map/'+ $
      date+'_'+sat+'_'+var+'_lspec.idl'

    input_header = create_struct(name = sat + '_lspec_head', $
                                 {sat:sat}, {var:var}, {yarr:yarr})

    data = fltarr(n_bins)
    t_tai = t_cut + tai_1
    
    dummy = create_struct(name = sat + '_lspec', $
                          {tai:dblarr(2)}, {data:data})
    input_data = replicate(dummy, n_cut)
    input_data.tai = transpose(t_tai)
    input_data.data = transpose(zarr)

    r_noaa_lspec_onefile, filename, /SAVE 

ENDIF     

END 

;------------------------------------------------------------------------------
FUNCTION mlt_average, mlt

c = n_elements(mlt)

mlt_x=cos(mlt*15*!PI/180.0)
mlt_y=sin(mlt*15*!PI/180.0)

mlt_av_x=total(mlt_x) / c
mlt_av_y=total(mlt_y) / c

mlt_av=atan(mlt_av_y/mlt_av_x)*180/(!PI*15)
        
case 1 of  
    ((mlt_av_x ge 0) AND (mlt_av_y ge 0)): $
            mlt_av=mlt_av       ; 1st quadrant
    ((mlt_av_x lt 0) AND (mlt_av_y ge 0)): $
            mlt_av=mlt_av+12.0  ; 2nd quadrant
    ((mlt_av_x lt 0) AND (mlt_av_y lt 0)): $
            mlt_av=mlt_av+12.0  ; 3rd quadrant
    ((mlt_av_x ge 0) AND (mlt_av_y lt 0)): $
            mlt_av=24+mlt_av    ; 4th quadrant
ENDCASE 

return, mlt_av

END
