;******************************************************************************
;* PROCEDURE:     
;*      p_noaa_lspec_spec, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_PLOTINFO.switch (three element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_noaa_lspec, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written June 1997, S. BOURDARIE
;*     modified September 1999, S. BOURDARIE
;******************************************************************************
pro p_noaa_lspec_spec, panel, sat, type, switch_pl, $
                          OVERPLOT=OVERPLOT, $
                          PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                          SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_noaa_lspec. 
COMMON noaa_lspec_data, input_header, input_data

papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
panelset,panel		             ;sets the panel position viewport

n_ek=n_elements(input_header.yarr)-1
yarr =fltarr(n_ek, 2)
yarr(*,0)=input_header.yarr(0:n_elements(input_header.yarr)-2)
yarr(*,1)=input_header.yarr(1:n_elements(input_header.yarr)-1)

yminarr=input_header.yarr(0:n_elements(input_header.yarr)-2)
ymaxarr=input_header.yarr(1:n_elements(input_header.yarr)-1)

; for color plots, the y-axis normally refers to a channel. You have to 
; re-scale the data array to the range required before passing it to the
; color plot routine, also the ymin and ymax arrays.

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

; in this example we have a single test array, so no case statement is needed.

; set the begin end end times. papco convention is to use running secconds
; starting with the no of seconds since the begiing of the start day.
time=input_data.time
xut1=float(mjdt_start.mjd) + mjdt_start.t / 86400.0      
xut2=float(mjdt_end.mjd) + mjdt_end.t / 86400.0

ymat=input_data.ymat
ymat=transpose(ymat)
nodata = 1.00000e-20
uztit=input_header.unit
uytitle=input_header.ytitle
utitle=sat+' '+input_header.comment1 +$
          '!C' + input_header.comment2

yst=yminarr(0)
yen=ymaxarr(n_elements(ymaxarr)-1)

; restrict data to actual time range requested - this makes for faster zooming

;   index=where((time ge xut1) and (time le xut2),count)
;   if count ne 0 then begin
;      ymat=ymat(index,*)
;      time=time(index)
;   endif

;if flag set to filling
if (switch_pl(1) eq 1 or switch_pl(1) eq 3) then fill_lav, ymat
;set flag for vertical smoothing if needed
if (switch_pl(0) eq 1) or (switch_pl(0) eq 3) then smooth_vert, ymat, 3
;do horizontal smoothing if needed
if (switch_pl(0) eq 2) or (switch_pl(0) eq 3) then smooth_horiz, ymat, 5

if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1) & zmax=zscl(panel(0),2)
endif else BEGIN
    papco_autorange,ymat,zmin,zmax, $
      log=zscl(panel(0),3), exclude=0, nodata=nodata
    zscl(panel(0),1)=zmin & zscl(panel(0),2)=zmax
ENDELSE

if (yscl(panel(0),0) eq 1) then BEGIN 
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
endelse

; Overriding values by setting keywords is possible and allowed but discouraged
; since then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
extra_plotPar={yrange:[yst, yen], xrange:[xut1, xut2], zrange:[zmin,zmax], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
               ztitle:uztit, $
               xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}
               
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)

; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    

; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
;resx, resy are in data coordinates. If the axis is logarithmic, 
;resx, resy need to be too.
resx=0  &  resy=0
papco_plot_colorspec, ymat, time, yarr, nodata=nodata, resx=resx, resy=resy, $
                      xstyle=5, ystyle=5, _extra=extra_plotPar
papco_color_bar, _extra=extra_plotPar

right_side_label,panel,utitle,/rot90

; store the coordinate information into common block
plotted_x = !x  &  plotted_y = !y

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label, panel, uytitle, /rot90 

END 
