;****************************************************************************
;*  NAME:        r_noaa_flux
;*
;*  DESCRIPTION: Routine to read noaa_flux files. Adheres to
;*               PAPCO conventions: Error common block
;*                                  Uses MJDT time
;*                                  PAPCO read philosophy
;*
;*  INPUT:       none (uses common block time from PAPCO)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        September 2000
;*
;*  AUTHOR:       S. Bourdarie
;****************************************************************************
pro r_noaa_flux, plotinfo, VERBOSE=VERBOSE

COMMON noaa_flux_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. Papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; empty panel with the error message get_err_msg is plotted.

get_err_no=0 & get_err_msg='noaa_flux read succesfully'

; check env. variable pointing to data directory
if not (papco_check_data_env('NOAA_DATA', PATH=path)) then return

; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   

sat = PlotInfo.filename

; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
  
      
while mjdt_end.mjd ge time.mjd do begin
     
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt   
    fln = sat+'/sem/'+string(year,format="(i4.4)")+'/data/*' $
           +string(year,day_oy,format="(i4.4,i3.3)")+'.dat.gz'

    found=findfile(path+fln,count=nfs)
    if nfs ne 0 then begin
        ; read one day of data
        resul = str_sep(found(0),'.')
        r_noaa_flux_onefile, resul(0), sat, /VERBOSE

        ;check if required mag coords are in data structure.
        ;make if needed.
        m_noaa_eph, PlotInfo 

        ;if a file was succesfully read, concatonate!
        if get_err_no eq 0 then begin 
            if no_files_read eq 0 then begin   ;start the array buffer
              input_data_buf=temporary(input_data)
              input_header_buf = input_header
            endif else begin                   ;otherwise concatonate arrays
              input_data_buf=[temporary(input_data_buf),temporary(input_data)]
            endelse        
            input_data=0                     ;clear array for reuse
            no_files_read=no_files_read+1
        ENDIF 
    ENDIF
    time.mjd = time.mjd+1
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then goto, done
endwhile         
done:
   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
     get_err_no=0
     get_err_msg='% r_noaa_flux: '+string(no_files_read,format="(i2)")+ $
       ' files read succesfully'
     ;clear arrays and load buffer into output array
     input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
     input_header = input_header_buf

endif else begin
     get_err_no=1
     get_err_msg='% r_noaa_flux: no files found for timerange'
     return
endelse      


END 

;******************************************************************************
;*  NAME:        r_noaa_flux_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        September 2000
;*
;*  AUTHOR:       S. Bourdarie
;******************************************************************************
pro r_noaa_flux_onefile, file, sat, SAVE = SAVE, VERBOSE=VERBOSE
  
COMMON noaa_flux_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON noaa_all

IF keyword_set(SAVE) THEN BEGIN
    save, file = file+'.idl', input_data,  /COMPRESS,  /verb
    message, 'idl saveset written', /cont
    print, '  ', file+'.idl'
    return
ENDIF 

fln = file  
get_err_no=0
get_err_msg='noaa_flux onefile read succesfully'
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
on_ioerror, read_original
   
restore,fln+'.idl',/verbose
dealloc_lun,/quiet    ;frees logical units used by save/restore
message, 'idl saveset read succesfully', /cont
print, '  ', fln+'.idl'
return

read_original:
; if the .idl save file could not be found, then try to read the original
r_noaa_flux_oneday_original, fln, sat
      
end   

;****************************************************************************
;*  NAME:        getmonth_onefile
;*
;*  DESCRIPTION: Routine to read noaa_flux files. Adheres to
;*               PAPCO conventions: Error common block
;*                                  Uses MJDT time
;*                                  PAPCO read philosophy
;*
;*  INPUT:       none (uses common block time from PAPCO)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        September 2000
;*
;*  AUTHOR:       S. BOURDARIE
;****************************************************************************
PRO r_noaa_flux_oneday_original, fln, sat

COMMON noaa_flux_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg

; clear exisiting data array
input_data=0

message, 'Reading '+fln+'.dat.gz (slow)', /cont

CASE sat OF
    'NOAA_N12': BEGIN
        data = fltarr(19)
    END 
    'NOAA_N14': BEGIN
        data = fltarr(19)
    END 
    'NOAA_N15': BEGIN
        data = fltarr(21)
    END 
    'NOAA_N16': BEGIN
        data = fltarr(21)
    END 
ENDCASE

time={MJDT, MJD:0l, T:0.0}
dummystr='' & dummytime=' 0 0 0'
dummy_struct=create_struct(NAME=sat, {alt:0.0}, {long:0.0}, {lat:0.0}, $
              {data:data}, {mjdt_time:time}, {time:0.0d0}, {tai:0.0d})
input_data=replicate(dummy_struct,11000)

openr,f_Unit,fln+'.dat.gz', /GET_LUN, ERROR=err,/compress

IF err NE 0 THEN BEGIN
    message, fln+'.dat.gz'+'  :  not available.', /cont
    get_err_no=2 & get_err_msg=!ERR_STRING
    message, !ERR_STRING, /cont
    RETURN
ENDIF 

; read the header
dec_y = 0.d0 & altitude = 0.0 & longitude = 0.0 & latitude = 0.0

i=-1
while not eof(f_Unit) do begin 
    i=i+1

    readf,f_Unit,dec_y, altitude, longitude, latitude, data

    aux  = dec_y & year = fix(aux) & Njours = 365.D0
    ;if (year eq 4*(year/4) and year ne 100*(year/100)) then Njours = 366.D0
    if (year eq 4*(year/4)) then Njours = 366.D0
    aux = (aux-double(year))*Njours & doy = 1+fix(aux)
    aux = (aux-double(doy-1))*24.D0 & hour = fix(aux)
    aux = (aux-double(hour))*60.D0  & min = fix(aux)
    aux = (aux-double(min))*60.D0   & s = fix(aux)

    time_mjdt=convert_date_to_t90(year = year, doy = doy, hour = hour,$
                                  minute = min, second = s,  /MJDT)

    input_data(i).time = double(time_mjdt.mjd) + double(time_mjdt.t) / 86400.d0
    input_data(i).tai = utc2tai({mjd:time_mjdt.mjd, time:time_mjdt.t*1000}) 
    input_data(i).mjdt_time = time_mjdt
    input_data(i).alt = altitude
    input_data(i).long = longitude
    input_data(i).lat = latitude
    input_data(i).data  = data

ENDWHILE 

close,f_Unit & free_lun,f_Unit
input_data=temporary(input_data(0:i))

; save read data with correct time. Rename data to *.idl
save,filename=fln+'.idl',input_data, /COMPRESS, /verbose

END


;****************************************************************************
; check / make magg coords / model mag field as needed.
; data needs to already be read into input_dat
;****************************************************************************
PRO m_noaa_eph, plotinfo

COMMON noaa_flux_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON mjdt, mjdt_start, mjdt_end
COMMON noaa_all
    
IF PlotInfo.PANELKIND NE "noaa_eph" THEN BEGIN
    intmod = 0 &  extmod = 0    ;chose IGRF, no external
ENDIF ELSE BEGIN
    intmod = PlotInfo.ioptions(0) & extmod = PlotInfo.ioptions(1)
ENDELSE

new_struct_name='MODEL_'+varprt(intmod)+'_'+varprt(extmod)
new_struct=create_struct(name='UNILIBR', 'L',0.0,'MLT',0.0,'MLAT',0.0, $
                           'B_GSM', fltarr(4)) 
tags=tag_names(input_data(0))
    
;check if required mag coords exist in file.
index=where(tags eq new_struct_name,c)
if c eq 0 then BEGIN
    ;add new structure to input_data
    dummy2=create_struct(input_data(0),new_struct_name,new_struct)
    dummy3=replicate(dummy2,n_elements(input_data)) 
    ;fill new array with copies of existing array
    struct_assign, input_data, dummy3
    input_data=temporary(dummy3)     

    message,' calculating L, MLT, MLAT, B for '+new_struct_name, /cont
    
    ;get the index of the new mag coords tag
    tags=tag_names(input_data(0)) 
    index=where(tags eq new_struct_name)
    new_idx=index(0)
    ndat =  n_elements(input_data)

    ;get mag coords (from UNILIB) 
    tai = input_data.tai
    pos = dblarr(3, ndat)
    pos(0, *) =  input_data.ALT+6371.20d
    pos(1, *) = 90.0d - input_data.LAT
    pos(2, *) = input_data.LONG
    cor_type='SPH' & cor_sys='GED'

    get_mag_unilib, tai, pos, cor_type, cor_sys, $
      bvec, Lval, mlt, ilat, error, INTMOD=INTMOD, EXTMOD = EXTMOD
 
    input_data.(new_idx).L = Lval
    input_data.(new_idx).mlt = mlt 
    input_data.(new_idx).mlat = ilat
    input_data.(new_idx).b_gsm = bvec

    r_noaa_flux_onefile, fln, /SAVE, /VERBOSE

ENDIF ELSE message,'mag coords exist in file', /cont   

END 
