;******************************************************************************
;*  NAME:        r_noaa_lspec
;*
;*  DESCRIPTION: Routine to read lspec files. 
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        December 2002
;*
;*  AUTOR:       R. Friedel
;*
;******************************************************************************
pro r_noaa_lspec, plotinfo, VERBOSE = VERBOSE

common noaa_lspec_data, input_header, input_data
common mjdt, mjdt_start, mjdt_end
common get_error, get_err_no, get_err_msg
COMMON noaa_all ;commons for module - see m_noaa_commons.pro for details

get_err_no=0 & get_err_msg='map read succesfully'

; check env. variable pointing to data directory
if not (papco_check_data_env('NOAA_DATA', PATH=path)) then return
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     

;make list of month dates for time period.
convert_t90_to_date, mjdt_start, year_start, day_oy, hour, minute, second,$
                          month_start, cmonth, day_om, /mjdt   
convert_t90_to_date, mjdt_end, year_end, day_oy, hour, minute, second,$
                          month_end, cmonth, day_om, /mjdt   

month = month_start & year = year_start
date = string(year, month, format = "(i4.4,i2.2,'01')")
WHILE (year NE year_end) OR (month NE month_end) DO BEGIN
    month = month+1
    IF month GT 12 THEN BEGIN
        month = 1 & year = year+1
    ENDIF
    date = [date, string(year, month, format = "(i4.4,i2.2,'01')")]
ENDWHILE

sat = PlotInfo.filename
CASE sat OF
    'NOAA_N12': BEGIN
        channel = PlotInfo.ioptions(0)
        typeNames=canal_NOAA12+ ' ' +gammep_NOAA12
    END 
    'NOAA_N15': BEGIN
        channel = PlotInfo.ioptions(1)
        typeNames=canal_NOAA15+ ' ' +gammep_NOAA15
    END 
    'NOAA_N16': BEGIN
        channel = PlotInfo.ioptions(2)
        typeNames=canal_NOAA16+ ' ' +gammep_NOAA16    
    END 
ENDCASE

; use first set channel only for Lspec plots
n_ch = n_elements(typeNames) & to_plot=bytarr(n_ch)
FOR i=0l,n_ch-1 DO IF (channel AND 2l^i) EQ 2l^i THEN to_plot(i)=1
plot_index=where(to_plot EQ 1, nplot)
IF nplot EQ 0 THEN BEGIN 
    message, 'No channels selected', /cont  &  return
ENDIF
var_name = typeNames(plot_index(0))
r = strsplit(var_name, ' ', /extract) & var = r(0)

no_files_read = 0
for i=0, n_elements(date)-1 DO BEGIN 
    ; clear data arrays for reuse
    input_header=0  &  input_data=0

    year = strmid(date(i), 0, 4)
    fln =  path+sat+'/sem/'+year+'/map/'+ date(i)+'_'+sat+'_'+var+'_lspec.idl'

    r_noaa_lspec_onefile, fln,/verbose

    if get_err_no eq 0 then begin 
        if no_files_read eq 0 then begin   ;start the array buffer
            input_data_buf=temporary(input_data)
            input_header_buf = input_header
        endif else begin                   ;otherwise concatonate arrays
            input_data_buf=[temporary(input_data_buf),temporary(input_data)]
        endelse        
        input_data=0                       ;clear array for reuse
        no_files_read=no_files_read+1
    endif
ENDFOR  

; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then BEGIN 
    get_err_no=0
    get_err_msg='% r_noaa_lspec: '+string(no_files_read,format="(i2)")+ $
       ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
    input_header = input_header_buf
endif else BEGIN 
    get_err_no=1 & get_err_msg='% r_noaa_lspec: no files found for timerange'
     return
ENDELSE

END 


;******************************************************************************
;*  NAME:        r_noaa_lspec_onefile, fln, SAVE=SAVE
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  KEYWORDS     SAVE - save file to saveset fln instead
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        December 2002
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_noaa_lspec_onefile, fln, SAVE = SAVE, VERBOSE = VERBOSE
  
COMMON noaa_lspec_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
   
get_err_no=0 & get_err_msg='noaa_lspec onefile read succesfully'
   
IF keyword_set(SAVE) THEN BEGIN 
    save, file = fln, input_header, input_data, /COMPRESS, VERBOSE = VERBOSE
    message, 'saved '+fln, /cont
    return
ENDIF

; clear data arrays for reuse
input_header=0  &  input_data=0   
   
on_ioerror, read_original
   
restore,fln, VERBOSE = VERBOSE
dealloc_lun, /quiet    ;frees logical units used by save/restore
message,'read '+fln, /cont
return

read_original:
; if the .idl save file could not be found, then try to read the original
r_noaa_lspec_onefile_original, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_map_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1997
;*
;*  AUTOR:       S. BOURDARIE
;*
;*  MODIFICATION HYSTORY : September 1999, by S. BOURDARIE
;******************************************************************************
pro r_noaa_lspec_onefile_original, fln, VERBOSE=VERBOSE
  
common noaa_lspec_data, input_header, input_data
common get_error, get_err_no, get_err_msg  
   
get_err_no=0 & get_err_msg='map_onefile_orignal read succesfully'
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file 

on_ioerror,out
openr, iunit, fln, /get_lun, /compress
;
;  read the header of the original file
;
   titre  = ''
   xtitre = ''
   ytitre = ''
   com1   = ''
   com2   = ''
   com3   = ''
   com4   = ''
   com5   = ''
   Nval   = LONARR(2)

   READF,iunit,titre
   READF,iunit,com1
   READF,iunit,com2
   READF,iunit,com3
   READF,iunit,com4
   READF,iunit,com5
   READF,iunit,xtitre
   READF,iunit,ytitre
   READF,iunit,Nval
   Etiquette = ''
   readf,iunit,Etiquette
   En = FLTARR(Nval(1))
   READF,iunit,En


; define the header and data structures. Edit to suit your data!
   max_no_rec=Nval(0)              ;the maximum number time records in your data
;
   time_mjdt={MJDT, MJD:0l, T:0.0}
   input_header={yarr:fltarr(Nval(1)), $
                            comment1:'', $
                            comment2:'', $
                            ytitle:'',$
                            unit:''}
   dummy={time:0.d0, ymat:fltarr(Nval(1)-1)}
   input_data=replicate(dummy, max_no_rec)
   
; fill header and data structures
   input_header.yarr=En
   input_header.comment1 = titre
   input_header.comment2 = com1
   input_header.ytitle = ytitre

   res=str_sep(com5,'-')
   result = ''
   for i=0,n_elements(res)-2 do result = result + res(i) + '!e-'
   result = result + res(n_elements(res)-1)
   res=str_sep(result,' ') 
   result = ''
   for i=0,n_elements(res)-1 do result = result + res(i) + '!n' 
   input_header.unit='['+result+']'
    
   val = dblarr(Nval(1)) 

for i=0L,max_no_rec-1L do begin
     READF,iunit,val
     ;tranform val(0) - time dec day - y,doy,hour,min,s
     aux  = val(0)
     year = fix(aux)
     Njours = 365.D0
     if (year eq 4*(year/4) and year ne 100*(year/100)) then Njours = 366.D0
     aux = (aux-double(year))*Njours
     doy = 1+fix(aux)
     aux = (aux-double(doy-1))*24.D0
     hour = fix(aux)
     aux = (aux-double(hour))*60.D0
     min = fix(aux)
     aux = (aux-double(min))*60.D0
     s = fix(aux)
     time_mjdt=convert_date_to_t90(year=year, doy=doy ,$
                             hour=hour , minute = min , second = s,  /MJDT)
     input_data(i).time=double(time_mjdt.mjd) + double(time_mjdt.t) / 86400.d0
;
;
;
     input_data(i).ymat=float(Val(1:Nval(1)-1))
endfor
close,iunit & free_lun,iunit

; end of user written section
;------------------------------------------------------------------------------

; save read data with correct time. Rename data to *.idl
;  save,filename=fln+'.idl',input_header,input_data,/verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='r_map_onefile:!C '+!ERR_STRING
   
end
