;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  omni_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the panel  editor window.
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      swittch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, swittch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supploed widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $          	 	
;*	                xut2:0.0}             	 	 
;*
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION omni_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON omni_panel_data, widgetData
COMMON omni, var_info, expert_users

;check if user is "expert user" by checking PAPCO_PATH for that user name. 
current = getenv('PAPCO_PATH') & expert = 0
FOR i = 0, n_elements(expert_users)-1 DO BEGIN
    pos = strpos(current, expert_users(i))
    IF pos NE -1 THEN expert = 1
ENDFOR

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
panel_base=papco_widget_base(title=aPlotInfo.panelKind,$
	                     column=1, GROUP_LEADER=group, /MODAL)

; All papco_panel_Add* routines are papco panel widgets that can be included
; in the panel editor or not. They provide functionalty that papco supports.
; They dynamically extend the panel editors widget syructure, and all the 
; events they generate are supoported by papco_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)

;
; <<<<< insert your widgets ! (replace the labels below) >>>>>>>>>
;
  
var_names = var_info(3:48, 0)
var_base = CW_BGROUP(panel_base, var_names, COLUMN = 3, /EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Select Data:', $
                     UVALUE='spb_var', $
                     SET_VALUE=aplotinfo.typevector(0)) 

IF expert THEN BEGIN 
    toggle_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=3, /NONEXCLUSIVE)
    spb_update=WIDGET_BUTTON(toggle_base,UVALUE='spb_update', $
                             VALUE='Update DST / KP')
    WIDGET_CONTROL, spb_update, SET_BUTTON= aPlotInfo.typevector(1)
ENDIF 

; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $papco_MODULES/panels&slice/more_panel_widgets
; 
; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInforStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the  widget is always reinitialized to its last state.

; The next four papco panel widgets will be added only if you want to
; support that functionalty, and if they make sence for your omni.

;--- add the  papco panel widgetfor choosing the plot-style for line plots ----
widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
  
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exisit in the omni example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'omni_panel.help', $
                 typevector:aPlotInfo.typevector}
widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
papco_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
papco_xmanager, 'omni_panel', panel_base

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - the typeVector-tag and swittch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
returnData={typevector:widgetData.typevector}
additionalRData=papco_panel_ReturnData(widgetData)
return, create_struct(ReturnData, additionalRData)
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO omni_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;******************************************************************************
PRO omni_panel_event, event

COMMON omni_panel_DATA, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF 
    'spb_update': widgetData.typevector(1)=event.select
    'spb_var': widgetData.typevector(0)=event.value
    else: widgetData=papco_panel_Event(event, widgetData)
ENDCASE

END
