;******************************************************************************
;* sw_data2pres,dens,veloc
;* 
;* Calculates solar wind pressure (dynamic) from 
;* density(cm^-3) and solar wind velocity (m*s^-1)
;******************************************************************************

FUNCTION sw_data2pres, dens, veloc

  p_m = 1.6726e-27

  pressure = p_m*dens*veloc^2*10e20
  return, pressure
END

;*******************************************************************************
;* function to calculate the magnetospheric subsolar point
;* No function has been implemented yet!!!
;********************************************************************************
 
FUNCTION sw_data2subsol, den, veloc, imf_z
  ;; should get algorithm to estimate subsolar point
  SUBSOL = make_array(n_elements(den), value = 10)    ;---cy, change to -999 then unilib will calculate dis defaultly
 
  return, subsol
END 

;*********************************************************************************
;* omni_params,t1,t2,TAI=tai
;* 
;* DESCRIPTION function to calculate the 10 parameters to be used as
;*             input for the unilib magnetic field models. 
;* INPUT
;*               T1       -  start date/time in any PAPCO format, string
;*               T2       -  end date/time in any PAPCO format, string
;* OUTPUT
;*               model_params - (10*t_el) array of [kp],[dst], [swpres], [swdens],
;*                              [swvel], [imf_x], [imf_y],
;*                              [imf_z],[subsol], [lastpara]
;* KEYWORDS 
;*               TAI - array of tai input times. Force routine to
;*                      return model_params for these times, using
;*                      interpolation. A solar wind lag of 30 min is
;*                      used (from observation point)
;*********************************************************************************
FUNCTION omni_params,t1, t2, TAI = tai 

;---cy, set Tai to get the parameters@Tai, output array(10)
;---cy, defualtly get the parameters@T1,output array(24,10), hourly value

  COMMON omni_data ;input_header, input_data
  COMMON mjdt, mjdt_start, mjdt_end

  model_input_txt = ['kp', 'dst', 'swpres', 'swdens', 'swvel', 'IMF_x', 'IMF_y' $
  ,'IMF_z','subsol', 'n/a']

  a_omni, T1 = t1,  T2 = t2, DATA = 0 ;returns the entire year of omni data

  result = convert_timeToSeconds(T1, /T90, /MJDT)
  IF result.err EQ 0 THEN mjdt_start =result.value ELSE BEGIN
     message, 'Error in start time (T1)', /cont
     print,  result.errtext
     return, 0
  ENDELSE
  
  result = convert_timeToSeconds(T2, /T90, /MJDT)
  IF result.err EQ 0 THEN mjdt_end =result.value ELSE BEGIN
     message, 'Error in end time (T2)', /cont
     print,  result.errtext
     return, 0
  ENDELSE 

  tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
  tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

  IF keyword_set(TAI) THEN BEGIN 
     tai = tai-1800e ; trying to make up for lag of solar wind (half hour delay)
     ndat = n_elements(tai)
     

     omni_tai = input_data.tai 
     
     kp = interpol(input_data.kp, omni_tai, tai)/10                ;---cy, put on the 10
     dst = interpol(input_data.dst, omni_tai, tai)
     swdens = interpol(input_data.proton_density, omni_tai, tai)   ;---cy, change the tag name to proton_density
     FOR jj = 0, ndat-1 DO IF swdens(jj) GT 100 THEN swdens(jj) = 25   ;---cy, make sure we got right SW density       
     swvel = interpol(input_data.plasma_bulk_speed, omni_tai, tai)
     FOR jj = 0, ndat-1 DO IF swvel(jj) GT 2000 THEN swvel(jj) = 300   ;---cy, make sure we got right SW speed
     IMF_x = interpol(input_data.bx_gse_gsm,  omni_tai, tai)
     FOR jj = 0, ndat-1 DO IF IMF_x(jj) GT 100 THEN IMF_x(jj) = 6   ;---cy, make sure we got right SW Bx
     IMF_y = interpol(input_data.by_gsm, omni_tai, tai)
     FOR jj = 0, ndat-1 DO IF IMF_y(jj) GT 100 THEN IMF_y(jj) = 0   ;---cy, make sure we got right SW By
     IMF_z = interpol(input_data.bz_gsm, omni_tai, tai)
     FOR jj = 0, ndat-1 DO IF IMF_z(jj) GT 100 THEN IMF_z(jj) = 0   ;---cy, make sure we got right SW Bz
     swpres = sw_data2pres(swdens, swvel)
     subsol = sw_data2subsol(swdens, swvel, imf_z)
     model_params = [[kp],[dst], [swpres], [swdens], [swvel], [imf_x], $
                     [imf_y], [imf_z],[subsol], [tai]]              ;---cy, put in the tai as the last param
  ENDIF ELSE BEGIN

      
        
    

  year = input_data(0).year
  doy = input_data.decimal_day
  
  utc = doy2utc(doy, year) 
  index = where(utc.mjd EQ mjdt_start.mjd)
  ;---cy, above line make possible to get 1 day parametes (T1) so far 
  kp = input_data(index).kp/10.
  dst = input_data(index).dst
  swdens = input_data(index).proton_density       ;---cy, change tag name to proton_density from ion_density
  FOR jj = 0, ndat-1 DO IF swdens(jj) GT 100 THEN swdens(jj) = 25   ;---cy, make sure we got right SW density
  swvel =  input_data(index).plasma_bulk_speed
  FOR jj = 0, ndat-1 DO IF swvel(jj) GT 2000 THEN swvel(jj) = 300   ;---cy, make sure we got right SW speed
  IMF_x = input_data(index).bx_gse_gsm
  FOR jj = 0, ndat-1 DO IF IMF_x(jj) GT 100 THEN IMF_x(jj) = 6   ;---cy, make sure we got right SW Bx
  IMF_y = input_data(index).by_gsm
  FOR jj = 0, ndat-1 DO IF IMF_y(jj) GT 100 THEN IMF_y(jj) = 0   ;---cy, make sure we got right SW By
  IMF_z = input_data(index).bz_gsm
  FOR jj = 0, ndat-1 DO IF IMF_z(jj) GT 100 THEN IMF_z(jj) = 0   ;---cy, make sure we got right SW Bz
  swpres = sw_data2pres(swdens, swvel)
  subsol = sw_data2subsol(swdens, swvel, imf_z)
  lastpara = input_data(index).tai              ;---cy, tai for the last parameter
  
  model_params = [[kp],[dst], [swpres], [swdens], [swvel], [imf_x], [imf_y], [imf_z],[subsol], [lastpara]]

ENDELSE

;stop, 'make check'
  return, model_params
END 



;******************************************************************************
;*  NAME:        a_omni
;*
;*  DESCRIPTION:r access to omni module's data
;*               For use inside the PAPCO environment 
;*               (papco is compiled, required modules are loaded)
;*               
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:   
;*               T1       -  start date/time in any PAPCO format, string
;*               T2       -  end date/time in any PAPCO format, string
;*               VERBOSE  -  enables print comments, save/restore
;                            comments
;*               DATA     -  
;*         
;*  Enter a time in one of these formats: date time, date
;*  where time is in one of those formats: h:m:s, h:m
;*  and date is in one of those formats: m/d/y, doy/y, d.m.y, m/d/y, yyyymmdd
;*  where d is day and doy is day of year, h:m:s is the time
;*  (year > 1990 if you use T90 format)
;*  e.g: 1/90 14:11, 1/1990 1:1:1, 350/1991 01:00:03
;*       15.1.90 14:11, 1/15/1990 15:3:1                  
;*
;*  DATE:        March 2003
;*
;*  AUTHOR:       R. Friedel
;*
;*  CHANGES:     
;******************************************************************************
PRO a_omni, T1 = T1, T2 = T2, DATA = data

;first, we need to call papco once so that it initializes itself. This
;sets up a bunch of common blocks, structures and environmental
;variables that are needed by various modules.

papco, /set_only ;only initializes papco then exists.

IF NOT keyword_set(T1) THEN BEGIN
    message, 'No start time (T1) defined', /cont & return
ENDIF 

IF NOT keyword_set(T2) THEN BEGIN
    message, 'No end time (T2) defined', /cont & return
ENDIF 

IF NOT keyword_set(DATA) THEN data = 0

;PAPCO sets the start and end time in a common block, which controls
;the time range for all read routines and all plot routines in all
;papco modules that are loaded.
COMMON mjdt, mjdt_start, mjdt_end

;The time format in mjdt_start, mjdt_end is modified Julian date and
;seconds of day, in a structure:
; mjdt_start = {MJD:0l, T:0.0}

;PAPCO provides a set of time conversion routines that can convert
;between string times to papco time and a variety of other formats:
;look in ../papco/papco_lib/convert_time.pro

;for the example here, if the input string is in any of the PAPCO time
;formats, use the function convert_timeToSeconds to convert to
;mjdt_start, mjdt_end. This function returns a structure containing
;the tags err, value and errtext. If err is zero, the time string was
;recognized, and the converted time is in value. If err eq -1 there
;was an error, the error message is in errtext. This errtext the also
;contains a list of the "allowed" formats...

result = convert_timeToSeconds(T1, /T90, /MJDT)
IF result.err EQ 0 THEN mjdt_start =result.value ELSE BEGIN
    message, 'Error in start time (T1)', /cont
    print,  result.errtext
    return
ENDELSE

result = convert_timeToSeconds(T2, /T90, /MJDT)
IF result.err EQ 0 THEN mjdt_end =result.value ELSE BEGIN
    message, 'Error in end time (T2)', /cont
    print,  result.errtext
    return
ENDELSE 

;if you don't like the papco time formats you can also use the time
;routines provided under
;../papco/papco_lib/cdssw/time/
;these will come in handy in any case, since a lot of the papco
;modules use the TAI format internally - international atomic time.
;There's an informative readme file in that directory too.
;Here's a quick way of converting mjdt_start, mjdt_end to TAI:
tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

;Once the time common block is set, the module routines can be called.

;In general, the module's read routine will return data for the time
;set in common mjdt. You'll need to look at that read routine to see
;what parameters it needs to be called with. This can be as little as
;nothing, to something like a Satellite ID, or a more complete
;plotinfo structure( see explanation below...). In the read routine
;you'll also find the common block that the data is returned in, in
;general it looks something like:
common omni_data, input_header, input_data

;where  input_header is a structure containing info about the data
;       input_data is an array of structures containing the data
;investigate by using  help,input_header,/st

;This data might not be in the format you want, as it generally will
;need some calibration/conversion etc applied to it. Often this is
;done by a module's plot routine. 
;A lot (but possibly not all) modules now support "getting" data from
;the plot routine, if the keyword OUTPUT is set to 2. basically this
;calls the plot routine, and returns from it just before the actual
;plot is made, at which point all the plot's data is left in a common
;block. 
;Again, you need to look at the plot routine to see what the call is,
;and what the common block is. many plot routines now use the same
;plotinfo structure as the read routine as input. The data is returned
;in a common block that looks something like this:

;common slice_lanl_geo2, time, ymat, utitle, uytitle, ych

;The same common block is also used by the slice routines...


;In case this isn't confusing enough, one can combine both of the
;above steps. PAPCO internally keeps a list of all the plotinfo
;structures for each plot it needs to make, and calls one interface
;routine when it needs to make a plot. That routine first calls the
;read routine, and then the plot routine. You can call it yourself,
;with the keyword OUTPUT set to 2. This routine is always called
;draw_module_name, and always has the same set of input parameters and
;keywords. It resides in the modules
;../papco_interface/papcoadd_module_name.pro file:
;
;pro draw_lanl_geo, panel, PlotInfo, orbitNo, $
;                   get_data_call, slice_type, IsRead, $
;                   OUTPUT=OUTPUT

;when using this routine "manually" many of the above parameters are
;not important. panel is a vector specifying the panel position of the
;plot, set it to a default (panel=[0,1,1]). PlotInfo is the
;panel's structure containing all it's plot information, that needs to
;be set (see below). orbitNo is used if the module you're using allows
;time specification by orbit number, generally that is not the case,
;just set it to zero. get_data_call, slice_type, are return
;values which are not needed further but need to be in the call. 
;IsRead needs to be set to zero, meaning the data is not read. if
;successful, it is returned set to 1, and if you call the routine
;again with isread set to 1 it won't read the data again as it is
;already in the read common block. Useful if you're re-reading the
;same data often but getting different plot data made... use with
;care, or always set to zero to force a read.
;OUTPUT is the keyword that needs to be set to 2!


;now, the explanation of the PlotInfo structure. This is a general
;structure that represents all the data needed to specify the plot
;the module needs to make. This structure contains tags used by PAPCO,
;and tags used by the user who wrote the module. It is this structure
;that is modified by the module's panel editor (the widget you get with
;all the choices for a given module). So by setting this structure
;correctly, you can control exactly what the data is that is going to
;be plotted, in exactly the same manner as you would by clicking the
;options in the widget. The only problem of course is to know how each
;option is represented in the plotinfo structure, and this is where
;things may get difficult: Because PAPCO is pretty general, THE NAME
;OF THE TAGS USED WILL BE THE SAME, BUT THE WAY THEY ARE SET WILL MEAN
;DIFFERENT THINGS IN DIFFERENT MODULES.
;So where do you get this information? PAPCO convention requires this
;info to be in the module's help file, exactly for this
;purpose. However, not all module authors write such a help file or
;keep it up-to-date... then your only backup is to actually look at the
;module's panel_editor routine (the one that makes the widget) and to
;work out manually what options are set in what way in the plotinfo
;structure. This can be painful...

;The module's help file in in its papco_interface directory:
;../module_name/papco_interface/module_name_panel.help
;this help file is displayed when you click "help" on the module's
;panel editor. 
;In this help file the PlotInfo structure information is in the
;section "DESCRIPTION OF USE of PLOTINFO STRUCTURE"


;So in general, you need to "make" a plotinfo structure, set it
;according to what data you want, and then call the draw_module_name
;routine.

;To make a plotinfo structure there's a papco routine you can use:

plotinfo = papco_getplotinfostruct()

;So for the lanl_geo module you might do something like this:

plotinfo.ioptions(0) =  data    ;t.d 10(?) = Bx GSE/GSM

;Finally you're set to read / make the data!

panel = [0, 1, 1]
orbitNo = 0
;plotinfo we set above
IsRead = 0
output = 2


draw_omni, panel, PlotInfo, orbitNo, get_data_call, slice_type, IsRead, $
               OUTPUT = output

;you can check if the above call was successful. All module read
;routines set an error common block that can be interrogated:

common get_error, get_err_no, get_err_msg

;get_err_no is zero if successful, non-zero if not, then get_err_msg
;contains the error message.

IF get_err_no NE 0 THEN BEGIN
    message, get_err_msg, /cont
    return
ENDIF

;if you get this far then the data was read and resides in the common
;block from the plot routine.

;The data structure from the omni module is returned for a  whole
;year!




;Now you're done, have the data you need in common blocks, and can do
;what you want with it...


END 
