;******************************************************************************
;* PROCEDURE:     
;*      p_omni_line, panel, plotinfo, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_OMNI_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written July 1997, Haje Korth
;******************************************************************************
pro p_omni_line, panel, plotinfo, OUTPUT=OUTPUT, OVERPLOT=OVERPLOT, $
                        PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                        SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed:
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl		       ;man/auto yscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
COMMON omni_data, input_header, input_data
COMMON omni, var_info

if keyword_set(OUTPUT) then output=OUTPUT else output=0

panelset,panel			;sets the panel position viewport

 ;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1 
   
var_id = plotinfo.typevector(0)
yray=data.(var_id+4)
utitle = var_info(var_id+3, 0)
uytitle = var_info(var_id+3, 1)
nodata = 1e31

idx = where(yray NE nodata, c)
IF c EQ 0 THEN GOTO, noplot

; if this routine is called with OUTPUT=2, data is in common block, just return
if output eq 2 then return

; if this routine is called with OUTPUT=3, then need to output data to MEMORY,
; for use in scratch pad.
if output eq 2 then begin
    format='MEM'
    goto, output
endif else format=0

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded
; in.

if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
endif else BEGIN
    papco_ylim, time, yray, yst, yen, BAD_DATA = nodata
    yscl(panel(0),1)=yst  &   yscl(panel(0),2)=yen
endelse

; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)
  
; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override papco defaults.
; See $papco_BASE/papco/papco_draw for the definition of the papco default
; graphic sytem variable settings.
extra_plotPar={yrange:[yst,yen], xrange:[xut1,xut2], $
               ytitle:uytitle, ylog:yscl(panel(0),3), xtickformat:'noticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
  
; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than
; 'noticks'. It's always drawn in black
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar  
     
; now do the actual data plot, data only, no axis   
gap_plotPar=create_struct({color:extra_plotPar.color, $
                           thick:extra_plotPar.thick, $
                           psym:extra_plotPar.psym, $
                           symsize:extra_plotPar.symsize,  $
                           linestyle:extra_plotPar.linestyle})  
 
plot, [0, 1], [0, 1], _extra=extra_plotPar, ystyle = 5, xstyle = 5, /nodata

;case 1 of 
    ;(var_id eq 34) or (var_id eq 35): begin
    ;   id_plot,time,yray
    ;end
;    (var_id eq 24): begin
;      if gap_plotPar.psym eq 10 then time=time+tres/2.0
;      oplot,time,yray,_extra=gap_plotPar       
;    end  
;    (var_id eq 27): begin
;      if gap_plotPar.psym eq 10 then time=time+tres/2.0
;      oplot,time,yray,_extra=gap_plotPar       
;    end  
;    else: BEGIN
;      plot, time, yray,_extra=extra_plotPar 
;      ;omni_gap_oplot,time,yray,_extra=gap_plotPar
;    end
;endcase

papco_gap_plot, time, yray, nodata, $
  _extra = extra_plotPar, ystyle = 5, xstyle = 5

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y
  
; overplot a dotted line for zero level 
oplot, [xut1,xut2],[0,0], color=1, linestyle=2

if not keyword_set(overplot) then begin
     right_side_label,panel,utitle,_extra=extra_Par
endif else begin
     for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
     right_side_label,panel,utitle,_extra=extra_Par
endelse

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      TRANGE        array(2) of user time start and end of data

output:  

if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data to file',/cont
    description='omni data output'
;    time_label='Seconds since start of day'
    time_label='TAI'
    xut1=tai_1  &  xut2=tai_2 & time = data.tai   ; this line only if output with TAI 
    trange=[xut1,xut2]
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, TRANGE=trange, $
                      RS_LABEL=RS_LABEL, FORMAT=format
ENDIF

return

;jump to here if no data to plot 
noplot: 
message,'No data to plot', /cont
plotted_x = !x  &  plotted_y = !y

END

;******************************************************************************
;* PROCEDURE:     
;*      id_plot, time, id_arr, panel
;* 
;* DESCRIPTION:  
;*      This procedure plots the spacecraft id's. the spacecraft represented
;*      by the each color is plotted on the right side of the plot.
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*
;*      id_arr  array of spacecraft id's.
;*
;*      time    time array
;*              
;* OUTPUTS: 
;*      none
;* 
;* CALLING SEQUENCE:
;*       id_plot, time, id_arr, panel
;*
;* MODIFICATION HISTORY:       
;*     written July 1997, Haje Korth
;******************************************************************************
pro id_plot,time,id_arr,panel
  
COMMON papco_color_names

  finito=0
  used_sc=0
  sc_arr=id_arr
  size=100*(!p.position(3)-!p.position(1))
  usersym,[0,0],[-size/2,size/2]
  color_arr=[ black, red, green, yellow, blue, magenta, cyan, $
              white, burgundy, olive, dark_green, teal, $
              royal_blue, violet, dark_grey, grey]

  repeat begin
    used_sc=[used_sc,sc_arr(0)]
    index=where(sc_arr ne sc_arr(0),count)
    if count ne 0 then sc_arr=sc_arr(index) else finito=1
  endrep until finito
  used_sc=used_sc(1:n_elements(used_sc)-1)
  
  for i=0,n_elements(used_sc)-1 do begin
    case used_sc(i) of
      0: sc_name=''
      1: sc_name='HEOS 1 / HEOS 2'
      3: sc_name='VELA 3'
      5: sc_name='OGO 5'
     10: sc_name='PROGNOZ 10'
     11: sc_name='ISEE 1'
     13: sc_name='ISEE 3'
     18: sc_name='IMP 1'
     28: sc_name='IMP 3'
     33: sc_name='AIMP 1'
     34: sc_name='IMP 4'
     35: sc_name='AIMP 2'
     41: sc_name='IMP 5'
     43: sc_name='IMP 6'
     47: sc_name='IMP 7'
     50: sc_name='IMP 8'
     51: sc_name='WIND'
     98: sc_name='LANL IMP-6,-7,-8'
     99: sc_name='LANL VELA Speeds'
    endcase
    index=where(id_arr eq used_sc(i),count)
    for j=0l,n_elements(index)-1 do $
       plots,time(index(j)),0.5,color=color_arr(i),thick=15,psym=8
    for k=0, i do sc_name='!C'+sc_name
    right_side_label,panel,sc_name,color=color_arr(i)
  endfor
  
end

;******************************************************************************
;* PROCEDURE:     
;*      omni_gap_oplot, x, y, _EXTRA=e
;* 
;* DESCRIPTION:  
;*      This procedure plots data without connecting data point if they are
;*      further than 3 hours apart.
;*
;* INPUTS:       
;*      x,y     arrays to be plotted
;*              
;* OUTPUTS: 
;*      none
;* 
;* CALLING SEQUENCE:
;*      omin_gap_oplot, x, y, _EXTRA=e
;*
;* MODIFICATION HISTORY:       
;*     written July 1997, Haje Korth
;******************************************************************************
pro omni_gap_oplot,x,y,_EXTRA=e

; check for psym=10 and disallow, doesn't work for plots 
  if e.psym eq 10 then begin
    e.psym=0
    message, 'histogram mode does not work here',  /cont
  end
    
; first remove all data invalid points
   ndat=n_elements(x) 
   index=where(y ne 0,count)
   if count ne 0 then begin
      xray=x(index)
      yray=y(index)
   endif

; work out average spacing along time axis. If actual spacing encountered is
; more than three times the average, don't plot.

   ndat=n_elements(xray)
   diff_time=abs(xray(1:ndat-2) - xray(0:ndat-1))
   av_int=total(diff_time)/(n_elements(diff_time))

; work out jumps in y. If a jump is larger than 5 times the average, don't plot

   for i=0l,ndat-2 do begin
      int=abs(xray(i+1)-xray(i))
      if int lt av_int*3 then begin
        plots,xray(i),yray(i) 
        plots,xray(i+1),yray(i+1),/continue,noclip=0,_EXTRA=e
      endif
    endfor

END 
