;******************************************************************************
;*  NAME:        r_omni
;*
;*  DESCRIPTION: Routine to read omni files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       Haje Korth
;*
;*  Written:     July 1997, Haje Korth
;*
;******************************************************************************
PRO r_omni, plotinfo, VERBOSE=VERBOSE

COMMON omni_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read-
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.
get_err_no=0 & get_err_msg='omni read succesfully'

; check for env. variable pointing to data directory
if not (papco_check_data_env('OMNI_DATA', PATH=path)) THEN return

convert_t90_to_date, mjdt_start, ayear, aday_oy, ahour, aminute, asecond, $
                    amonth, acmonth, aday_om, /mjdt
convert_t90_to_date, mjdt_end,   byear, bday_oy, bhour, bminute, bsecond, $
                    bmonth, bcmonth, bday_om, /mjdt

files_read=0 & update = plotinfo.typevector(1)

FOR curyear=ayear,byear DO BEGIN

    IF update THEN m_prov_omni_dst_kp, curyear

    curyearstr=strtrim(string(curyear),2)
    fln='omni2_'+curyearstr
    ext = '.idl'

    ;see if idl saveset exists. If not, try to get remotely
    res = findfile(path+fln+ext, count = c)

    IF c EQ 0 THEN BEGIN 
        ;try to get remotely
        download_dir = path
        url = getenv('OMNI_FTP_ARCHIVE')
        papco_get_remote_data, url, fln+ext, download_dir
        ;check if file "arrived"
        res = findfile(path+fln+ext, count = c)
        IF c EQ 0 THEN BEGIN
            get_err_no = 1 & get_err_msg = 'No file downloaded'
        ENDIF
        ;try read original ascii file (local)
        r_omni_ascii, fln, VERBOSE=VERBOSE
        res = findfile(path+fln+ext, count = c)
        IF c EQ 0 THEN BEGIN
            get_err_no = 2 & get_err_msg = 'No ascii read'
            CONTINUE 
        ENDIF 
    ENDIF

    restore,path+fln+ext, VERBOSE=verbose

    files_read=files_read+1

    IF curyear EQ ayear THEN tmp_input_data=temporary(input_data) $
      ELSE tmp_input_data=[temporary(tmp_input_data),temporary(input_data)]

    message,fln+' read successfully', /cont
ENDFOR

if files_read eq 0 then begin
    get_err_no=2
    get_err_msg='No files found in timerange'
    return
endif else begin
    get_err_no=0
    get_err_msg='All files found read succesfully'
endelse

input_data=temporary(tmp_input_data)

END

;****************************************************************************
;*  NAME:        r_omni_ascii
;*
;*  DESCRIPTION: Routine to read omni files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*                              running example module on it's own
;****************************************************************************
PRO r_omni_ascii, fln, VERBOSE=VERBOSE

COMMON omni_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON omni

; clear exisiting data array
input_data=0

; check for env. variable pointing to data directory
if not (papco_check_data_env('OMNI_DATA', PATH=path)) THEN return

;see if ascii file exists, if not, call routine to make prov file
;which is stored as .idl
r = findfile(path+fln+'.dat', count =  c)
IF c EQ 0 THEN BEGIN
    year = fix(strmid(fln, 6, 4))
    m_prov_omni_dst_kp, year, /VERBOSE
    return
ENDIF

message,'reading '+path+fln+' (slow)', /cont

;make structure for data
var_names = var_info(*, 0)
n_var = n_elements(var_names)
FOR i = 0, n_var-1 DO BEGIN
    var_names(i) = strsubst(var_names(i), ' ', '_')
    var_names(i) = strsubst(var_names(i), '>', 'g')
    r = execute('type = '+var_info(i, 3))
    IF i EQ 0 THEN dummy = create_struct(var_names(i), float(type)) $
      ELSE dummy = create_struct(dummy, var_names(i), float(type))
ENDFOR

dummy2 = create_struct('tai', 0.0d, dummy)
input_data=replicate(dummy2,9000)
i=-1

;now read the ascii file, and compute tai
openr, u, path+fln+'.dat', /get_lun, ERROR=err

while not eof(u) do begin
    i = i+1
    readf, u, dummy
    struct_assign, dummy, dummy2
    utc = DOY2UTC(dummy.Decimal_Day, dummy.Year)
    utc.time = dummy.Hour*60l * 60l * 1000l
    dummy2.tai = utc2tai(utc)
    input_data(i) = dummy2
endwhile

close, u & free_lun, u

input_data=temporary(input_data(0:i))

;now assign the SAME bad data flag to each variable - 1e31
FOR i = 0, 48 DO BEGIN ;loop through all variables
    var = input_data.(i+1)
    nodata = var_info(i, 3)
    idx = where(var EQ nodata,c)
    IF c NE 0 THEN BEGIN
        var(idx) = 1e31
        input_data.(i+1) = var
    ENDIF
ENDFOR 

; save read data with correct time. Rename data to *.idl
save, filename=path+fln+'.idl', input_data, /compress, VERBOSE=VERBOSE

END

;******************************************************************************
; makes a provisional omni file containing only dst reading from the
; provisional dst outputfile created from the web site
; http://swdcdb.kugi.kyoto-u.ac.jp/dstdir/dst1/prov.html
;
; Simply make files from the website by saving the page for a month as text.
; Naming the files
; 'dstprovyyyymm.txt'    where yyyy is the year, mm is the month number.
; Then call this routine. Files must reside in omni directory.
;******************************************************************************
pro m_prov_omni_dst, year, VERBOSE=VERBOSE, FORCE=FORCE

; check for env. variable pointing to data directory
  data_dir='OMNI_DATA'
  path=papco_getenv(data_dir)
  if strlen(path) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif

; make an omni-like data structure
  time={MJDT, MJD:0l, T:0.0}
  dummystr=''
  dummytime=' 0 0 0'
  dummy_struct={omni0,bartels_rot_num:0, id_imf:0, $
                      id_sw:0, fin_tim_imf:0, fin_tim_plasma:0, $
                      field_mag_avg:0.0, mag_avg_field_vec:0.0, $
                      lat_ang:0.0, long_ang:0.0, bx_gse:0.0, by_gse:0.0, $
                      bz_gse:0.0, by_gsm:0.0, bz_gsm:0.0, sigma_totb:0.0, $
                      sigma_b:0.0, sigma_bx:0.0, sigma_by:0.0, sigma_bz:0.0, $
                      plasma_temp:0.0, ion_density:0.0, flow_speed:0.0, $
                      flow_long:0.0, flow_lat:0.0, sigma_t:0.0, sigma_n:0.0, $
                      sigma_v:0.0, sigma_phi_v:0.0, sigma_theta_v:0.0, $
                      kp_index:0, c9_index:0, sunspot_num:0, dst_index:0l, $
                      pf1:0.0, pf2:0.0, pf4:0.0, pf10:0.0, pf30:0.0, $
                      pf60:0.0, msph_flux_flag:0, mjdt_time:time, $
                      time:0.0}

  input_data=replicate(dummy_struct,9000)


;read in the dst data from provisional files

  dst_dummy_str={dst1, year:0, month:'', day:0, hour:0, dst_av:0}

  fln_templ=path+string(year,format="('dstprov',i4.4,'*.txt')")
  files=findfile(fln_templ,count=c)
  if c eq 0 then begin
    message,'Files '+fln_templ+' not found',/cont
    return
  endif

  for i=0,c-1 do message,files(i),/cont
  print,''

  junk='' &  inarr=intarr(25) & n=0
  for j=0, n_elements(files)-1 do BEGIN
    openr,unit,files(j),/get_lun
    WHILE strmid(junk, 0, 3) NE 'WDC' DO readf,unit,junk
    readf,unit,junk ;& print,junk
    readf,unit,junk ;& print,junk
    result=str_sep(strtrim(junk,2),'  ')
    year=fix(result(1))
    month=strlowcase(strmid(result(0),0,3))
    message,month+'   '+varprt(year), /cont
    readf,unit,junk ;& print,junk
    readf,unit,junk ;& print,junk
    readf,unit,junk ;& print,junk

    while not eof(unit) do begin
      readf,unit,junk
      if junk eq '' then goto, next
      if strmid(junk,0,7) eq 'contact' then goto, next
      if strmid(junk,0,1) eq '<' then goto, next
      reads,junk,inarr,format="(i2,i5,7(i4),i5,7(i4),i5,7(i4))"
      for i=1,24 do begin
        dummy_struct.mjdt_time=convert_date_to_t90(day=inarr(0), month=month,$
                                                   year=year, hour=i, /MJDT)
        dummy_struct.dst_index=inarr(i)
        index=n*24+(i-1)
        if keyword_set(VERBOSE) then $
          print,index,dummy_struct.mjdt_time, year,month, inarr(0),i,inarr(i)

        input_data(index)=dummy_struct

      endfor
      n=n+1
      next:
    endwhile
    close,unit & free_lun,unit
  endfor

  input_data=temporary(input_data(0:index))

; save read data with correct time. Rename data to *.idl
; if there is an exisiting .idl for that year, read it and only add in
; the next DST info!
  fln='omni_'+string(year,format="(i4.4)")+'.idl'
  files=findfile(path+fln, count=c)
  if (c ne 0) and NOT (keyword_set(FORCE)) then begin
    message,'Reading existing '+path+fln+ ' and adding in new DST info',/cont
    dst = input_data.dst_index
    restore, file = files(0), /verb
    in_dst=input_data.dst_index
    idx = where(in_dst EQ 99999, c)
    IF c NE 0 THEN input_data(idx).dst_index = dst(idx) $
    ELSE BEGIN
        message,' omni file up to date', /cont
        return
    ENDELSE
    message, 'Updating exisiting omni file '+ path+fln, /cont
    save,filename=path+fln,input_data,/verbose
  endif else begin
    message, 'Saving to new omni file '+ path+fln, /cont
    save,filename=path+fln,input_data,/verbose
  endelse

end

;******************************************************************************
; makes a provisional omni file containing only dst and kp read from
; the data files downloaded daily by Mike Henderson.
;
; data are in $DST_KP_DATA/Dst    /yyyy/Dst_yyyymmdd.dat
;             $DST_KP_DATA/Kp     /yyyy/Kp_yyyymmdd.dat
; orginised in subdirs by year, ASCI files by day
;******************************************************************************
pro m_prov_omni_dst_kp, year, VERBOSE=VERBOSE, FORCE=FORCE

COMMON omni_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON omni, var_info

; check for env. variable pointing to data directory
if not (papco_check_data_env('OMNI_DATA', PATH=omni_path)) THEN return
if not (papco_check_data_env('DST_KP_DATA', PATH=dstkp_path)) THEN return

year_dir = string(year, format = "(i4.4, '/')")
this_year = year

message, 'Updating Dst/Kp - slow..', /cont

IF NOT (papco_check_data_env('PAPCO_DOWNLOAD', PATH=temp_path)) THEN return

var_names = var_info(*, 0)

;read in existing omni idl data set if exists, if not, create one,
;filled with nodata values and time tags
fln='omni2_'+string(this_year,format="(i4.4)")+'.idl'
result = findfile(omni_path+fln, count = c)
IF c EQ 1 THEN BEGIN
    message, 'Updating existing omni file ', /cont
ENDIF ELSE BEGIN
    message, 'Making new structure for data ', /cont
    var_names = var_info(*, 0)
    n_var = n_elements(var_names)
    FOR i = 0, n_var-1 DO BEGIN
        var_names(i) = strsubst(var_names(i), ' ', '_')
        var_names(i) = strsubst(var_names(i), '>', 'g')
        r = execute('type = '+var_info(i, 3))
        IF i EQ 0 THEN dummy = create_struct(var_names(i), type) $
          ELSE dummy = create_struct(dummy, var_names(i), type)
    ENDFOR
    dummy2 = create_struct('tai', 0.0d, dummy)
    input_data=replicate(dummy2,9000)
    ;now fill the time tags of the data structure with times for this year.
    start_tai = UTC2TAI({YEAR:this_year, MONTH:1, DAY:1, $
                         HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
    end_tai   = UTC2TAI({YEAR:this_year+1, MONTH:1, DAY:1, $
                         HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
    tai = start_tai & i = 0
    WHILE tai LT end_tai DO BEGIN
        print,  i, '   ', TAI2UTC(tai, /ECS)
        t = TAI2UTC(tai, /EXTERNAL)
        input_data(i).YEAR = t.year
        input_data(i).DECIMAL_DAY = UTC2DOY(t)
        input_data(i).HOUR = t.hour
        tai = tai+3600.0d & i = i+1
    ENDWHILE
    input_data = input_data(0:i-1)
ENDELSE

;get nodata value for Dst
idx = where(var_info(*, 0) EQ 'DST')
nodata = 1e31

;read in the dst data from daily downloaded files
dst_path = dstkp_path+'Dst/'+year_dir

;The files here have 5 columns
;
;	Column 1: UT in hours
;	Column 2: DOY (plus fraction)
;	Column 3: YEAR (in decimal years)
;	Column 4: Julian Date
;	Column 5: Dst in nT

; files can be concatonated to get a full year period. Concatonate to
; $PAPCO_DOWNLOAD
cmd = 'cat '+dst_path+'Dst_*.dat >! ' + temp_path+'dst_temp.dat'
print, cmd & spawn, cmd

;now read the ascii file, update dst
dummy = {ut:0.0, doy:0.0d, year:0.0d, mjd:0.0d, dst:0l}
ON_IOERROR, done_dst
openr, u, temp_path+'dst_temp.dat', /get_lun
WHILE NOT eof(u) DO BEGIN
    readf, u, dummy
    hr = fix(dummy.ut) & doy = fix(dummy.doy)
    idx = where((input_data.HOUR EQ hr) AND $
                (input_data.DECIMAL_DAY EQ doy) AND $
                (input_data.YEAR EQ this_year), c)
    IF c EQ 0 THEN BEGIN        ;add data sutructure to input_data, compute tai
        idx = [n_elements(input_data)-1]
        CALDAT, dummy.mjd, Month, Day, Year, Hour, Minute, Second
        input_data(idx(0)).HOUR = hr
        input_data(idx(0)).DECIMAL_DAY = doy
        input_data(idx(0)).YEAR = Year
        input_data(idx(0)).TAI = UTC2TAI({YEAR:Year, MONTH:Month, DAY:Day, $
                                 HOUR:hr, MINUTE:0, SECOND:0, MILLISECOND:0})
    ENDIF

    IF input_data(idx(0)).dst EQ nodata THEN BEGIN
        IF dummy.dst NE 9999 THEN  BEGIN
            IF abs(dummy.dst) GT 1e5 THEN dummy.dst = nodata
            input_data(idx(0)).dst = dummy.dst
        ENDIF
    ENDIF

ENDWHILE
done_dst:
close, u & free_lun, u
spawn, 'rm '+temp_path+'dst_temp.dat'

ndat = n_elements(input_data)

;read in the kp data from daily downloaded files
kp_path = dstkp_path+'Kp/'+year_dir

;The Kp files contain 8 lines with the following columns:

;Column 1: UT in hours
;Column 2: DOY (plus frac of day)
;Column 3: YEAR (plus frac of year)
;Column 4: Julian Date
;Column 5: Bartel Rotation Number
;Column 6: Day in Bartel Rotation
;Column 7: Kp index
;Column 8: ap index

; files can be concatonated to get a full year period. Concatonate to
; $PAPCO_DOWNLOAD
cmd = 'cat '+kp_path+'Kp*.dat >! ' + temp_path+'kp_temp.dat'
print, cmd & spawn, cmd

;now read the ascii file, update kp
dummy = {ut:0.0, doy:0.0d, year:0.0d, mjd:0.0d, bart:0, dbart:0, kp:0.0, ap:0}
ON_IOERROR, done_kp
openr, u, temp_path+'kp_temp.dat', /get_lun
WHILE NOT eof(u) do BEGIN
    readf, u, dummy
    hr = fix(dummy.ut) & doy = fix(dummy.doy)
    idx = where((input_data.HOUR EQ hr) AND $
                (input_data.DECIMAL_DAY EQ doy) AND $
                (input_data.YEAR EQ this_year), c)
    IF c EQ 1 THEN BEGIN
        IF input_data(idx(0)).BARTELS_ROTATION_NUMBER EQ nodata THEN $
          input_data(idx(0):(idx(0)+2) < (ndat-1)).BARTELS_ROTATION_NUMBER = dummy.bart
        IF input_data(idx(0)).KP EQ nodata THEN $
          input_data(idx(0):(idx(0)+2) < (ndat-1)).KP = fix(dummy.kp*10)
    ENDIF

ENDWHILE
done_kp:
close, u & free_lun, u
spawn, 'rm '+temp_path+'kp_temp.dat'

print, '  ', omni_path+fln
save, filename = omni_path+fln, input_data, /compress, /verbose

end
