;******************************************************************************
;* FUNCTION:
;*      FUNCTION  papco_cdf_panel, plotinfo, GROUP=group, ACTION=ACTION
;*
;* DESCRIPTION:
;*	Open an editor widget for a panel, using the methods defined in the
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:
;*      plotinfo        a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type.
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from plotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(plotinfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  papco_cdf_panel, plotinfo, GROUP=group, ACTION=ACTION

COMMON papco_cdf_panel_data, widgetData
COMMON papco_cdf_info, svnames, file, dirs
COMMON papco_cdf ;see papco_cdf_commons.pro for contents

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize
;    the window, you have to add your own window-specific data to that
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5***
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in papco_cdf_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(papco_cdf_control)
control = *plotinfo.USR_PTR1
;control now contains the user defiend control structure for this module.

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
panel_base=papco_WIDGET_BASE(TITLE=plotinfo.panelKind, ROW=1, $
                             GROUP_LEADER=group, /MODAL)
panel_left  = widget_base(panel_base, COLUMN=1)
panel_right = widget_base(panel_base, COLUMN=1)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event,
;    transparent to the user.

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_left, plotinfo)
additionalWData=create_struct(widgets, additionalWData)

;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

;--- get list of data directories available -----------------------------------

;make a list of cdf directories. These dirs are all under the $PAPCO_CDF_DATA
;data tree where links are maintained to all the cdf dirs.

; check env. variable pointing to data directory
if not (papco_check_data_env('PAPCO_CDF_DATA', PATH=path)) then $
  return, plotinfo

dirs = papco_get_dirlist(path)


cdfdir_base = CW_BGROUP(panel_left, dirs, COLUMN=5, /EXCLUSIVE, /FRAME, $
                        LABEL_TOP='Select CDF directory:', $
                        UVALUE='spb_cdfdir',SET_VALUE=control.cdf_dir)

;--- report on the sample file found for cdf_dir selection --------------------
cdf_file_lbl_base = widget_base(panel_left, /FRAME, /ROW)
dummy = WIDGET_LABEL(cdf_file_lbl_base, VALUE = 'Template: ')
IF control.cdf_file NE '' THEN $
  file_template = papco_subst_file(control.cdf_file, /TEMPLATE, /VERSION) $
ELSE file_template = ''
cdf_file_lbl = WIDGET_LABEL(cdf_file_lbl_base, $
  VALUE = strFormat(file_template, 75, /LEFT))

;--- depending on what is cdf directory is chosen, display the variables.
cdfvar_lbl = WIDGET_LABEL(panel_left, $
                          VALUE = 'Variable name - variable type - decription')

cdfvar_list = WIDGET_LIST(panel_left, SCR_YSIZE = 100, SCR_XSIZE = 70, $
                          UVALUE = 'll_cdfvar', VALUE = strarr(30))
widget_control, cdfvar_list, SET_LIST_SELECT = control.var_list

;--- if the cdf chosen is for cluster, allow choices for spacecraft
sc_names=['CLUSTER 1','CLUSTER 2', 'CLUSTER 3','CLUSTER 4']
sc_id_base=CW_BGROUP(panel_left, sc_names, COLUMN=4, /EXCLUSIVE, /FRAME, $
                     UVALUE='spb_sc_id', LABEL_LEFT = 'Spacecraft :', $
                     SET_VALUE=control.clust_no)
widget_control, sc_id_base, SENSITIVE = 0

dummy1 = widget_base(panel_left, ROW = 1)

;--- provide choices for coordinates ------------------------------------------
coord_names = ['X', 'Y', 'Z', 'Magnitude']
n_coord = n_elements(coord_names)
set = bytarr(n_coord)
FOR i=0,n_coord-1 DO $
  IF (control.coord AND 2^i) eq 2^i THEN set(i)=1 ELSE set(i)=0
coord_base=CW_BGROUP(dummy1, coord_names, COLUMN=4, /NONEXCLUSIVE, /FRAME, $
                     UVALUE='spb_coord', LABEL_LEFT = 'Coordinates:', $
                     SET_VALUE=set)

;--- provide smoothing options ------------------------------------------------
ef_smooth=CW_FIELD(dummy1, ROW=1, /INTEGER, $
                   TITLE = 'Smooth in time:', XSIZE = 3, $
                   UVALUE='ef_smooth', VALUE=control.smooth)

;MATT ADDITIONS start
;--------------------------------------time shift input
tshift_base=CW_FIELD(dummy1, ROW=1, /FLOAT, /RETURN_EVENTS, $
                     TITLE = 'time_shift (mins) :', XSIZE = 3, $
                     UVALUE= 'time_shift', VALUE=control.time_shift)

;--- control on line lable position -------------------------------------------
title = 'none'
names = papco_make_choice_names(control.label_info, TITLE = title)
label_base = papco_CW_BGROUP(panel_left, names, ROW=1, /EXCLUSIVE, /FRAME, $
                             UVALUE='spb_label', LABEL_LEFT = title, $
                             SET_VALUE=control.label)

;-- provide extra autoscaling options -----------------------------------------
scaling_names = ['Use cdf Y-scaling', 'Use PAPCO autoscaling']
scaling_base = CW_BGROUP(panel_left, scaling_names, ROW=1, $
                         /EXCLUSIVE, /FRAME, UVALUE='spb_scaling', $
                         LABEL_LEFT = 'Y autoscaling :', $
                         SET_VALUE=control.scaling)



;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this.
;widgets=papco_panel_AddFile_Widgets(panel_left, plotinfo)
;additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the color bar for color plots ----
;widgets=papco_panel_Addcolorbar_Widgets(panel_left, plotinfo)
;additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(panel_left, plotinfo, 0)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(panel_right, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_panel_AddZScale_Widgets(panel_right, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_right, Action, plotinfo, $
                                     /CONFIG, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the papco_cdf example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'papco_cdf_panel.help', $
                 sc_id_base:sc_id_base, $
                 base:panel_base, $
                 scaling_base:scaling_base, $
                 dirs:dirs, $
                 ef_smooth:ef_smooth, $
                 cdf_file_lbl:cdf_file_lbl, $
                 cdfvar_list:cdfvar_list,$
                 tshift_base:tshift_base}

; add control struct, papco_panel structs
widgetData=papco_PANEL_CreateWidgetData(control, thisWidgetsData)
widgetData=papco_PANEL_CreateWidgetData(widgetData, additionalWData)

;--- realize the window -------------------------------------------------------
papco_cdf_refresh, /ALL
papco_Panel_Realize, panel_base, widgetData
papco_xmanager, 'papco_cdf_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified }
additionalRData=papco_panel_ReturnData(widgetData)
return, create_struct(ReturnData, additionalRData)

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_cdf_refresh
;*
;* DESCRIPTION:
;*      refresh rotuine for papco_cdf_panel, called by event handler
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*
;*
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO papco_cdf_refresh, ALL = ALL, VARLIST = VARLIST, AUTOSCALING = AUTOSCALING

COMMON papco_cdf_panel_data, widgetData
COMMON papco_cdf_info, svnames, file, dirs

IF keyword_set(ALL) THEN BEGIN
    VARLIST = 1 & AUTOSCALING = 1
ENDIF

psp = papco_get_pathsep()

IF keyword_set(VARLIST) THEN BEGIN
    r = papco_check_data_env('PAPCO_CDF_DATA', PATH=path)
    ;check if cdf_dir points at a cluster dir, then select
    ;according to spacecraft #
    cdf_type = widgetData.dirs(widgetData.cdf_dir)
    widgetData.cdf_type = cdf_type
    cdf_dir = papco_checkdirname(path+cdf_type)
    pos = strpos(cdf_dir, 'cluster')
    pos2= strpos(cdf_dir, 'cluster_aux')
    if pos2 ne -1 then  pos = -1

    IF pos NE -1 THEN BEGIN
        widget_control, widgetData.sc_id_base, sensitive = 1
        sc_id = widgetData.clust_no+1
        mask = '*c'+varprt(sc_id)+'*' & mask2 = '*C'+varprt(sc_id)+'*'
    ENDIF ELSE BEGIN
        widget_control, widgetData.sc_id_base, sensitive = 0
        mask = '*' & mask2 = '*'
    ENDELSE

    ;pick a file to enquire. **Files could be in subdirs by year!**
    res = findfile(cdf_dir+mask+'.cdf', count = n)
    IF n EQ 0 THEN res = findfile(cdf_dir+psp+mask2+'.cdf', count = n)
    IF n EQ 0 THEN BEGIN
        years = papco_get_dirlist(cdf_dir)
        IF years(0) NE '' THEN BEGIN
            FOR i = 0, n_elements(years)-1 DO BEGIN
                cdf_mask = cdf_dir+psp+years(i)+psp+mask+'.cdf'
                res = findfile(cdf_mask, count = n)
                IF n EQ 0 THEN BEGIN
                    cdf_mask = cdf_dir+psp+years(i)+psp+mask2+'.cdf'
                    res = findfile(cdf_mask, count = n)
                ENDIF
                IF n NE 0 THEN break
            ENDFOR
            file = res(0)
        ENDIF ELSE BEGIN
            file = ''
            svnames = ['', 'Cannot find any '+cdf_type+' cdf files']
        ENDELSE
    ENDIF ELSE BEGIN
        file = res(0)
    ENDELSE

    widgetData.cdf_file = file
    file_template = papco_subst_file(widgetData.cdf_file, /TEMPLATE, /VERSION)
    widget_control, widgetData.cdf_file_lbl, $
       SET_VALUE = strFormat(file_template, 75, /CENTER)

    IF file NE '' THEN pick_myVARIABLES, file, svnames,  /NOWIDGET
    widget_control, widgetData.cdfvar_list, set_value = svnames
    widget_control, widgetData.cdfvar_list, $
      set_list_select = widgetData.var_list
    cdfvar = svnames(widgetData.var_list)
    pos = strpos(cdfvar, ' data ')
    IF pos EQ -1 THEN pos = strpos(cdfvar, 'SC1_CODH_0')

    IF pos NE -1 THEN cdfvar_name = strtrim(strmid(cdfvar, 0, pos), 2) $
      ELSE cdfvar_name = 'not valid data variable'
    widgetData.cdfvar_name = cdfvar_name

ENDIF

IF keyword_set(AUTOSCALING) THEN BEGIN
    IF widgetData.manualYScaling EQ 0 THEN sensitive = 1 ELSE sensitive = 0
    widget_control, widgetData.scaling_base, SENSITIVE = sensitive
ENDIF

END


;******************************************************************************
;* PROCEDURE:
;*      PRO papco_cdf_panel_event, event
;*
;* DESCRIPTION:
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:
;*      event            an XManager event
;*
;* KEYWORDS:
;*      none
;*
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO papco_cdf_panel_event, event

COMMON papco_cdf_panel_data, widgetData
COMMON papco_cdf_info, svnames, file, dirs

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF

    'spb_label':widgetData.label = event.value

    'time_shift':BEGIN
        widgetdata.time_shift=event.value
     END
    'spb_scaling': BEGIN
        widgetData.scaling = event.value
    END

    'spb_cdfdir':BEGIN
        widget_control, /HOURGLASS
        widgetData.cdf_dir = event.value
        papco_cdf_refresh, /VARLIST
    END

    'll_cdfvar':BEGIN
        widget_control, /hourglass
        cdfvar = svnames(event.index)
        widgetData.var_list = event.index
        pos = strpos(cdfvar, ' data ')
        sc_str = 'SC'+varprt(widgetData.clust_no+1)
        IF pos EQ -1 THEN pos = strpos(cdfvar, sc_str+'_CODH_0')
        IF pos EQ -1 THEN pos = strpos(cdfvar, sc_str+'_HIAIons0')
        IF pos NE -1 THEN BEGIN
           cdfvar_name = strtrim(strmid(cdfvar, 0, pos), 2)
           widgetData.cdfvar_name = cdfvar_name
           IF event.clicks EQ 2 THEN BEGIN ;double clicked
                  cdf_var_struct = read_mycdf(cdfvar_name, file)
                  help, cdf_var_struct.(0), /st, output = str_list
                  result = messagebox(str_list(1:n_elements(str_list)-1), $
                               ['Cancel'], SCROL = 30, $
                               GROUP_LEADER=widgetData.base, $
                               TITLE = 'Contents of '+cdfvar_name(0))
           ENDIF
       ENDIF
    END

    'spb_sc_id' : BEGIN
        widget_control, /HOURGLASS
        widgetData.clust_no=event.value
        papco_cdf_refresh, /VARLIST
    END

    'spb_coord':BEGIN
        widget_control, event.id, get_value = set
        widgetData.coord = 0
        for i=0,n_elements(set)-1 DO IF set(i) THEN $
          widgetData.coord = widgetData.coord+2^i
    END
    ELSE: BEGIN
        WIDGET_CONTROL, widgetData.ef_smooth, GET_VALUE=x
        widgetData.smooth = x
        widgetData=papco_panel_Event(event, widgetData)
        IF ( (uval NE 'pb_done_and_cont') AND $
             (uval NE 'pb_cancel') AND $
             (uval NE 'pb_done') ) THEN papco_cdf_refresh, /AUTOSCALING
    END

ENDCASE


END
