;******************************************************************************
;*  NAME:        r_papco_cdf
;*
;*  DESCRIPTION: Routine to read papco_cdf files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_papco_cdf, plotinfo, VERBOSE=VERBOSE

COMMON papco_cdf
COMMON papco_cdf_data, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

FORWARD_FUNCTION papco_subst_file

get_err_no=0  &  get_err_msg='papco_cdf read succesfully'

; check env. variable pointing to data directory
IF NOT (papco_check_data_env('PAPCO_CDF_DATA', PATH=cdf_path)) THEN return

; check VERBOSE keyword and set
IF keyword_set(VERBOSE) THEN verbose = VERBOSE ELSE verbose = 0

;make list of files to be read for time period
mjdt_now = mjdt_start  &  no_files_read = 0

;retrieve the control structure for this module.
control = *plotinfo.USR_PTR1

IF keyword_set(VERBOSE) THEN print, '1 ', control.cdf_file
IF keyword_set(VERBOSE) THEN print, ''

;set number of files needed (# of days)
n_files = mjdt_end.mjd-mjdt_start.mjd+1

;check if there is any time to be read on the end day.
IF mjdt_end.t EQ 0 THEN n_files = n_files-1

files = strarr(n_files) & req_files = files

FOR i = 0, n_files-1 DO BEGIN

    mjdt_now.mjd = mjdt_start.mjd + i  & mjdt_now.t = 0

    this_file = papco_subst_file(control.cdf_file, mjdt_now,  /VERSION)
    req_files(i) = this_file
    IF keyword_set(VERBOSE) THEN print, '2 ', this_file
    ;look for files matching date and version wildcards - use latest version
    res = findfile(this_file, count = n)
    IF n NE 0 THEN this_file = res(n-1) ELSE this_file = ''
    files(i) = this_file
    IF keyword_set(VERBOSE) THEN print, '3 ', files(i)
    IF keyword_set(VERBOSE) THEN print, ''

ENDFOR

;req_files contains all files needed, files contains what's on site
;try get the missing ones remotely
FOR i = 0, n_elements(files)-1 DO BEGIN
    IF files(i) EQ '' THEN BEGIN

        papco_sepdir, req_files(i), download_dir, fln, ext
        rem_file_path = strmid(download_dir, strlen(cdf_path), $
                               strlen(download_dir)-strlen(cdf_path))

        pos = strpos(control.cdf_type, 'cluster')
        IF pos NE -1 THEN BEGIN ;get cluster cdf's from mrfrench
            url = papco_getenv('PAPCO_CDF_DATA_FTP_ARCHIVE')
        ENDIF ELSE BEGIN
            url = papco_getenv('PAPCO_CDF_CDAWEB_FTP_ARCHIVE')
            rem_file_path = strsubst(rem_file_path, '_', '/')

            ;reset remote path for Cluster Aux files
            IF strmid(rem_file_path, 0, 11) EQ 'Cluster/aux' THEN BEGIN
                rem_file_path =  'cluster/cl/sp/aux'+$
                                  strmid(rem_file_path, 11, 5)
stop
            ENDIF 
            pos = strpos(fln, 'v')
            strput, fln, 'v**', pos
        ENDELSE

        help, url, rem_file_path, fln, ext, download_dir

        papco_get_remote_data, url+rem_file_path, fln+ext, download_dir

        IF get_err_no EQ 0 THEN BEGIN     ;check if file "arrived"
            res = findfile(req_files(i), count = n)
            IF n NE 0 THEN files(i) = res(n-1)
        ENDIF ELSE get_err_no = 0
        print, files(i)

    ENDIF
ENDFOR

idx = where(files NE '', n)
IF n EQ 0 THEN BEGIN
    msg = 'No '+control.cdf_type+' cdf files found for timeperiod'
    message, msg, /cont & get_err_no = 1 & get_err_msg = msg
    return
ENDIF

files = files(idx)

cdfvar_name = strcompress(control.cdfvar_name , /REMOVE_ALL)

;use CDAWLIB routine to read all the data for this time!
cdf_var_struct = read_mycdf(cdfvar_name, files)

input_data = temporary(cdf_var_struct)

;see if an error Tag was returned
tags = tag_names(input_data)
pos = strpos(tags, 'ERROR')
idx = where(pos NE -1, c)
IF c NE 0 THEN BEGIN
    print, input_data.ERROR
    IF control.cdfvar_name EQ 'not valid data variable' THEN BEGIN
        out_str = control.cdf_type + ', '+control.cdfvar_name
        msg = ['Variable choice not a valid cdf data variable:', out_str]
    ENDIF ELSE $
        msg = 'No '+control.cdfvar_name+' variable found for timeperiod'
    message, msg(0), /cont
    get_err_no = 1 & get_err_msg = msg
    return
ENDIF

END

;******************************************************************************
;Function to substitute into a path/file the year/date given by
;the input papco time
;
;Keywords: START_MJDT - use current PAPCO start time
;          TEMPLATE - substitute current template string
;          VERSION - check for version numbers in file "_vXX" or
;                    "_VXX" replace XX by wildcards
;******************************************************************************
FUNCTION papco_subst_file, fln, mjdt, START_MJDT = START_MJDT, $
                           TEMPLATE = TEMPLATE,  VERBOSE = VERBOSE,  $
                           VERSION = VERSION

COMMON mjdt, mjdt_start, mjdt_end

IF keyword_set(VERBOSE) THEN print, ''

CASE 1 OF
    (keyword_set(TEMPLATE) EQ 1): BEGIN
        year_str = 'YYYY' & date_str = 'yyyymmdd'
    END
    (keyword_set(START_MJDT) EQ 1): BEGIN
        convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second, $
                             month, cmonth, day_om, /mjdt
        date_str = string(year,month,day_om,format='(i4.4,i2.2,i2.2)')
        year_str = string(year, format='(i4.4)')
    END
    ELSE: BEGIN  ;set by input mjdt
        convert_t90_to_date, mjdt, year, day_oy, hour, minute, second, $
                             month, cmonth, day_om, /mjdt
        date_str = string(year,month,day_om,format='(i4.4,i2.2,i2.2)')
        year_str = string(year, format='(i4.4)')
    END
ENDCASE

out_fln = fln

;get position of version number in string, replace by wildcards
IF keyword_set(VERSION) THEN BEGIN
    IF keyword_set(VERBOSE) THEN message, 'substitute version wildcard', /cont
    ver_pos = stregex(fln, '_v[0-9]{2}')
    IF ver_pos NE -1 THEN strput, out_fln, '**', ver_pos+2
    ver_pos = stregex(fln, '_V[0-9]{2}')
    IF ver_pos NE -1 THEN strput, out_fln, '**', ver_pos+2

    IF keyword_set(VERBOSE) THEN print, fln
    IF keyword_set(VERBOSE) THEN print, out_fln
    IF keyword_set(VERBOSE) THEN print, ''
ENDIF

;get position of date string in file
IF keyword_set(VERBOSE) THEN message, 'substitute date string', /cont
date_pos = stregex(fln, '[0-9]{8}')
strput, out_fln, date_str, date_pos

IF keyword_set(VERBOSE) THEN print, fln
IF keyword_set(VERBOSE) THEN print, out_fln
IF keyword_set(VERBOSE) THEN print, ''

;get position of year subdir in path
IF keyword_set(VERBOSE) THEN message, 'substitute year string', /cont
year_pos = stregex( fln, '\'+papco_get_pathsep()+'[0-9]{4}'+'\'+papco_get_pathsep())


IF year_pos NE -1 then  strput, out_fln, year_str, year_pos+1

IF keyword_set(VERBOSE) THEN print, fln
IF keyword_set(VERBOSE) THEN print, out_fln
IF keyword_set(VERBOSE) THEN print, ''

return, out_fln

END


;******************************************************************************
PRO m_lanl_mpa_cdf_fortran, STARTDATE = STARTDATE, ENDDATE = ENDDATE, $
                            CDF_PATH = CDF_PATH, BIN_PATH = BIN_PATH, $
                            SAT_ID = SAT_ID
papco, /set_only
;check inputs
print, ''
IF keyword_set(STARTDATE) THEN date1 = long(STARTDATE) ELSE date1 = 19910605l
IF keyword_set(ENDDATE) THEN date2 = long(ENDDATE) ELSE date2 = 19910605l
message, 'Doing from '+string(date1)+' to '+string(date2), /cont
print, ''
IF keyword_set(CDF_PATH) THEN in_path = CDF_PATH ELSE $
  in_path = '/n/radbelt/dlnk/cdf/lanl_mpa/'
message, 'Reading from '+in_path, /cont
print, ''
IF keyword_set(BIN_PATH) THEN out_path = BIN_PATH ELSE $
  out_path = '/n/radbelt/data4/tmp/lanl/geo/mpa_from_cdf/'
message, 'Reading from '+out_path, /cont

IF keyword_set(SAT_ID) THEN sat_id = SAT_ID ELSE sat_id = '1989-046'

;get list of files to process
print, ''
files = findfile(in_path+sat_id+'*.cdf', count = n)
IF n EQ 0 THEN BEGIN
    message, 'Found no files under:', /cont
    print, ' ', in_path+sat_id+'*.cdf'
    return
ENDIF

message, 'Found following files:', /cont
dates = strmid(files, strlen(in_path)+9, 11)
intdates = lonarr(n)
FOR i = 0, n-1 DO BEGIN
    t = STR2UTC(dates(i), /external)
    intdates(i) = string(t.year, t.month, t.day, format = "(i4.4,i2.2,i2.2)")
    print, files(i), ' -> ', intdates(i)
endfor

idx = where((intdates GE date1) AND (intdates LE date2), c)
IF c EQ 0 THEN BEGIN
    message, 'No files found in date range', /cont
    return
ENDIF

intdates = intdates(idx)
files = files(idx)

;now loop and read cdf files, make binary fortran copy
cdfvar_names = ['ecp', 'Uthours', 'backgrd', 'Azanglp', 'Pcounts']
VARS = strupcase(cdfvar_names)
N_VAR = n_elements(VARS)
print, ''
FOR i = 0, c-1 DO BEGIN
    message, 'Reading '+files(i), /cont

    ;pick_myVARIABLES, files(i), svnames,  /NOWIDGET
    ;stop

    cdf_var_struct = read_mycdf(cdfvar_names, files(i))
    test = size(cdf_var_struct,/struct)
    IF test.TYPE_NAME EQ 'INT' THEN BEGIN
        message, 'Error reading cdf file', /cont
        continue
    ENDIF

    FOR j = 0, N_VAR-1 DO BEGIN
        r = execute(VARS(j)+' = cdf_var_struct.'+VARS(j)+'.dat')
        r = execute('size_'+VARS(j)+'= lonarr(11)')
        r = execute('res = size('+VARS(j)+')')
        r = execute('FOR k = 0, n_elements(res)-1 DO size_'+VARS(j)+'(k) = res(k)')

    ENDFOR

    spawn,  'rm /u/friedel/mypapco/TMP*'

    help,  N_VAR, size_ECP, size_UTHOURS, size_BACKGRD, size_AZANGLP, size_PCOUNTS, ECP, UTHOURS, BACKGRD, AZANGLP,PCOUNTS

stop

    ;make new file name
    papco_sepdir, files(i), path, fln, ext
    out_file = out_path+fln+'.bin'
    message, 'Writing '+out_file, /cont

    ;write fortran style binary.
    openw, u, out_file, /F77_UNFORMATTED, /GET_LUN

    writeu, u, N_VAR
    writeu, u, size_ECP
    writeu, u, size_UTHOURS
    writeu, u, size_BACKGRD
    writeu, u, size_AZANGLP
    writeu, u, size_PCOUNTS
    writeu, u, ECP, UTHOURS, BACKGRD, AZANGLP, PCOUNTS

    close, u & free_lun, u

ENDFOR


END
;******************************************************************************
