;******************************************************************************
;* PROCEDURE:     
;*      p_papco_conj_line, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_papco_conj_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_papco_conj_line, panel, channel, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_papco_conj_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, swittch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
  common yscale, yscl                             ;man/auto yscaling
  common shift_label, down                        ;for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  ;info on coords
  
  common get_error, get_err_no, get_err_msg
  common papco_color_names
  my_col=[black,red,blue,dark_green, yellow]
  common ampte_eph_data, ampte_eph_header, ampte_eph_data
  common geo_eph_data, geo_eph_header, geo_eph_data
     
  panelset,panel			;sets the panel position viewport
  ;resize panel viewport
  shrink=0.03
  width=!P.POSITION(2)-!P.POSITION(0)
  height= !P.POSITION(3)-!P.POSITION(1)
  if width ge height then begin ;shrink height
    !P.POSITION(1)=!P.POSITION(1)+shrink
    !P.POSITION(3)=!P.POSITION(3)-shrink
    !P.POSITION(0)=!P.POSITION(0)+shrink
    !P.POSITION(2)=!P.POSITION(0)+height
  endif else begin  
    
  endelse  

; set the begin end end times. papco convention is to use running seconds
; starting with the no of seconds since the beginning of the start day.
  ;time=input_data.time
  xut1=mjdt_start.t                       
  xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
  
  utitle='papco_conj'

  ;draw geomag equator plot
  papco_draw_eq_plane, 10, /NOERASE
  
; plot info right of the panel. No allowance for overplots.
  panelset,panel			;sets the panel position viewport
  right_side_label,panel,utitle,_extra=extra_Par
  
  ;do ampte plot
  r_ampte_eph
  if get_err_no ne 0 then begin
    message,'Error reading AMPTE Ephemeris',/cont  &  goto,do_lanl
  endif  
  ndat=n_elements(ampte_eph_data)  &  xarr=fltarr(ndat)  &  yarr=fltarr(ndat) 
  message,'Drawing mlt / L  for AMPTE...',/cont
  color=my_col(0)
  for i=0, ndat-1 do begin
    theta=ampte_eph_data(i).mlt *0.261799388
    xarr(i)=ampte_eph_data(i).l * cos(theta)
    yarr(i)=ampte_eph_data(i).l * sin(theta)
  endfor
  oplot,xarr,yarr,color=color
  ;plot satellite position at midpoint in time
  plots,xarr(ndat/2),yarr(ndat/2),psym=4, symsize=2, color=color, /data
  
; plot info right of the panel.
  extra_Par.color=color
  right_side_label,panel,'!CAMPTE',_extra=extra_Par  
  
  do_lanl:
  ;do lanl plot
  print,''
  ;result=lanl_geo_satlist(/REQUEST)
  ;lanl_geo_dates_needed, date
  
  newl='!C'
  for j=1,n_elements(result)-1 do begin      
    r_geo_eph, result(j)
    if get_err_no ne 0 then begin
      message,'Error reading LANL GEO Ephemeris',/cont  &  goto,skip
    endif        
    ndat=n_elements(geo_eph_data)  &  xarr=fltarr(ndat)  &  yarr=fltarr(ndat) 
    message,'Drawing mlt / L for '+result(j),/cont
            
    for k=0,ndat-1 do begin
      if geo_eph_data(k).l eq -1 then begin
        xarr(k)=-1  & yarr(k)=-1
      endif else begin  
        theta=geo_eph_data(k).mlt *0.261799388
        xarr(k)=geo_eph_data(k).l * cos(theta)
        yarr(k)=geo_eph_data(k).l * sin(theta)      
      endelse
    endfor
    print,''
    
    index=where(xarr ne -1,c)
    if c ne 0 then begin
      xarr=xarr(index)  &  yarr=yarr(index)
    endif
    color=my_col(j)
    oplot,xarr,yarr,color=color
    ;plot satellite position at midpoint in time
    ndat=n_elements(xarr)
    plots,xarr(ndat/2),yarr(ndat/2),psym=4, symsize=2, color=color, /data    
    extra_Par.color=color
    newl=newl+'!C' &  outstr='LANL '+result(j)
    right_side_label,panel,newl+outstr,_extra=extra_Par          
    skip:
  endfor 
  
; store the coordinate information into common block for slices
  plotted_x = !x  &  plotted_y = !y
  
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_papco_conj_line: Writing plot data out to file'
    description='papco_conj template sample data - single energy channel line'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
  endif  

end 
