;******************************************************************************
;* FUNCTION:     
;*      FUNCTION papco_event_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=papco_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     written February 1996, Haje Korth
;*     modified February 1997, Reiner Friedel
;******************************************************************************
FUNCTION  papco_event_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON papco_event_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

dirpath=papco_getenv('PAPCO_EVENT_DATA')
filter='*.event'
   
; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
panel_base=papco_widget_base(title=aPlotInfo.panelKind, COLUMN=1, $
                             GROUP_LEADER=group, /BASE_ALIGN_CENTER)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
  
label=widget_label(panel_base, VALUE='Select file of event times:')
  
;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this.
widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo,$
                                      DIRPATH=dirpath, FILTER=filter)
additionalWData=create_struct(widgets, additionalWData)      
  
;--- Select some overall plot options -----------------------------------------
yval_names=['Draw', "Don't Draw"]
yval_base=CW_BGROUP(panel_base, yval_names,/FRAME, ROW=1,/EXCLUSIVE, $
                        LABEL_LEFT='Y-Value Lines:', UVALUE='mpb_yval', $
                        SET_VALUE=aPlotInfo.typevector(1))

;--- Select timerange boxes  times or lines------------------------------------
names=['Event only','Event range box','Event range line',$
       'Event range box & line', 'Y-Value only'] 
mpb_select=CW_BGROUP(panel_base, names, COLUMN=3,/EXCLUSIVE, $
                     LABEL_TOP='Select type of event display', $
                     UVALUE='mpb_select', SET_VALUE=aPlotInfo.typevector(0))
  
;--- Enter yvalues for drawin of horizontal lines -----------------------------
yval_entry=cw_field(panel_base, $
              TITLE='Enter Y-value(s) for horizontal lines (y1,y2,y3...):', $
              /RETURN_EVENTS, XSIZE=75, /COLUMN, $
              VALUE=aPlotInfo.soptions(0), $
              UVALUE='ef_yval_entry')
  
;--- add the widgets for choosing the plot-style ------------------------------
widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for specifying own plot description label -----
widgets=papco_panel_AddDescr_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'papco_event_panel.help', $
                 typevector:aPlotInfo.typevector, $
                 yval_entry:yval_entry,$
                 soptions:aPlotInfo.soptions}
widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData
      
;--- refresh the papco panels
papco_panel_Refresh, widgetData, /ALL

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
papco_xmanager, 'papco_event_panel', panel_base

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - e.g. the typeVector-tag.
returnData={typevector:widgetData.typevector,$
            soptions:widgetData.soptions}
additionalRData=papco_panel_ReturnData(widgetData)
return, create_struct(ReturnData, additionalRData)
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_event_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written February 1996, Haje Korth
;******************************************************************************
PRO papco_event_panel_event, event

COMMON papco_event_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we don't need a special event-processing, so we just
; pass the event on to 'papco_panel_Event'. If you need to process
; events, you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your new widgets
   
case uval of
    'mpb_select': widgetData.typevector(0)=event.value
    'mpb_yval':   widgetData.typevector(1)=event.value
    'ef_yval_entry': begin
      widget_control,widgetData.yval_entry, get_value=yval_str
      widgetData.soptions(0)=yval_str
    end
    else: BEGIN
        widget_control,widgetData.yval_entry, get_value=yval_str
        widgetData.soptions(0)=yval_str
        widgetData=papco_panel_Event(event, widgetData)
    END 
ENDCASE 

END





