;******************************************************************************
;* PROCEDURE:     
;*      p_papco_event, panel, type, userstyle, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type papco_event for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale     
;*
;* INPUTS:       
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.   
;* 
;* CALLING SEQUENCE:
;*       p_papco_event, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written February 1996, Haje Korth
;*
;***************************************************************************
pro p_papco_event, panel, PlotInfo, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           SUBTABLE=SUBTABLE, _EXTRA=extra_par

common times, tev, events             ;plot data
COMMON papco_event_data,  input_header,  input_data
common mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
common yscale, yscl		        ;man/auto yscaling
common shift_label, down              ;common for x-axis label shifting
common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
panelset,panel			;sets the panel position viewport

filename=PlotInfo.filename
yvals=PlotInfo.soptions(0)
select=PlotInfo.typevector(0)

;use TAI as the time standard. WHen plotting, subtract startime so
;that times always start at zero.
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})
xut1=0  &  xut2=tai_2-tai_1 

IF select EQ 4 THEN BEGIN       ;draw y-values over full range only
    
ENDIF ELSE BEGIN

; restrict data in time  
index=where((input_data.t1_tai ge tai_1) AND (input_data.t2_tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
t1=data.t1_tai-tai_1 & t2=data.t2_tai-tai_1

ENDELSE 

;see if y values are set
boxy=[1.25,1.75] & yst = 1 & yen = 2 
if yvals ne '' then BEGIN
    draw_yvals=1
    yvals=strtrim(strcompress(yvals),2)
    y=float(str_sep(yvals,','))
    papco_ylim,t1,y,yst,yen, /EXTEND
    ;if we have at least two y's set, use them for box y limits.
    if n_elements(y) ge 2 then boxy=[y(0),y(1)]
endif else draw_yvals=0 

if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
ENDELSE 

; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.

IF yst EQ yen THEN BEGIN
    yst = yst-abs(yst*0.2) & yen = yen+abs(yen*0.2)
ENDIF

IF yst EQ yen THEN BEGIN
    yst = -1 & yen = 1
ENDIF 

papco_y_label, yst, yen, log=yscl(panel(0),3)

; set up plot keywords
extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
               ylog:yscl(panel(0),3), xtickformat:'noticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's xtickformat if it is something other than
; 'noticks'.
; If you use this rotuine, make sure your data plot routine uses
; xstyle=5 and ystyle=5 (inhibit axis) so that you don't clobber the
; axes drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar, $
  ystyle = 5   

; now do the actual data plot, data only, no axis
plot, [0, 1], [0, 1], _extra=extra_plotPar, ystyle = 5, xstyle = 5, /nodata
CASE select OF  
    0: BEGIN
        for i=0,n_elements(t1)-1 do $
          oplot,[t1(i),t1(i)], [yst,yen], _extra=extra_par
    END
    2: BEGIN
        for i=0,n_elements(t1)-1 do BEGIN
            oplot,[t1(i),t1(i)], [yst,yen], _extra=extra_par
            oplot,[t2(i),t2(i)], [yst,yen], _extra=extra_par, linestyle = 1
        ENDFOR     
    END
    1: BEGIN 
        for i=0,n_elements(t1)-1 do $
          polyfill,[t1(i),t1(i),t2(i),t2(i)], $
            [boxy(0),boxy(1),boxy(1),boxy(0)],/data, _extra=extra_par
    END  
    3: BEGIN 
        for i=0,n_elements(t1)-1 do begin
          oplot,[t(i),t(i)], [yst,yen], _extra=extra_par
          oplot,[t2(i),t2(i)], [yst,yen], _extra=extra_par
        end  
    END   
    4: BEGIN

    END 
    ELSE: BEGIN
        message, 'Not a valid event type', /cont
        GOTO, noplot
    END
ENDCASE 
  
if draw_yvals then $
    for j=0,n_elements(y)-1 do oplot,[xut1,xut2], [y(j),y(j)], _extra=extra_par

; store the coordinate information into common block
plotted_x = !x  &   plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin

endif   

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
IF NOT keyword_set(OVERPLOT) THEN BEGIN 
    papco_sepdir, filename, path, file, ext
    lbl = 'papco_event!C'+file
    IF keyword_set(PLOTS_ATTHISPOSITION) THEN $
        FOR j = 0, PLOTS_ATTHISPOSITION-1 DO lbl = '!C!C'+lbl
    right_side_label,panel, lbl, _extra=extra_Par
ENDIF

return

;jump to here if no data to plot 
noplot: 
message,'No data to plot',/cont
plotted_x = !x  &  plotted_y = !y  

END
