;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  papco_file_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  papco_file_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON papco_file_panel_data, widgetData, channel_set

;--- if GROUP is not specified, the group-leader will be the desktop
  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
  panel_base=papco_widget_base(title=aPlotInfo.panelKind, column=1, /modal, $
                               GROUP_LEADER=group)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
  widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
  
;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
  path=papco_getenv('PAPCO_FILE_DATA')
  widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo, $
                                     DIRPATH=path)
  additionalWData=create_struct(widgets, additionalWData)     
  
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
  
;--- add descroiption and format strings --------------------------------------
  descr_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1)
  title='Date/time file column description string:'
  title=strformat(title,75,/left)
  column_label=CW_FIELD(descr_base, /COLUMN, title=title, $
    value=aPlotInfo.soptions(1), uvalue='column_label', /string)
  
  str_formats='(YYYY DOY DD MM hh mm ss ddf dhf dmf dms)'
  title='Format for the date/time '+str_formats+':'
  title=strformat(title,75,/left)
  time_format=CW_FIELD(descr_base, /COLUMN, title=title, $
    value=aPlotInfo.soptions(2), uvalue='time_format', /string)
    
;--- add channel select buttons -----------------------------------------------
  channel_names=strarr(120)
  for i=0,119 do channel_names(i)=string(i,format="('empty ',i3.3)")

;--- use typevector(0), and typevector(1) as an 120 bit integer, as a
;    logical flag to set channels - a maximum of 120 channels can be set.
  channel_set=bytarr(120) 
  for i=0,59 do begin
    if (aPlotInfo.typeVector(0) and 2ll^i) eq 2ll^i then channel_set(i)=1 $
    else channel_set(i)=0
  endfor
  for i=60,119 do begin
    if (aPlotInfo.typeVector(1) and 2ll^(i-60)) eq 2ll^(i-60) then $
      channel_set(i)=1 $
    else channel_set(i)=0
  endfor
  
  channel_base=WIDGET_BASE(panel_base, /FRAME, ROW=40,/nonexclusive, $
                           /scroll, Y_SCROLL_SIZE=120) 
  
  mpb_type=lonarr(N_ELEMENTS(channel_names))

  FOR i=0, N_ELEMENTS(channel_names)-1 DO $
    mpb_type(i)=WIDGET_BUTTON(channel_base, $
                              VALUE=strcenter(channel_names(i),18), $
                              UVALUE='mpb_type'+strtrim(string(i), 2))
  
  for i=0,119 do begin
    if channel_set(i) eq 1 then $
      WIDGET_CONTROL, mpb_type(i), SET_BUTTON=1 $
    else $
      WIDGET_CONTROL, mpb_type(i), SET_BUTTON=0
  endfor  
  
;--- add some choices for the plot --------------------------------------------
  choices_base=WIDGET_BASE(panel_base, ROW=1,/FRAME)
  smooth_base=WIDGET_BASE(choices_base, ROW=1)
  smooth_entry=CW_FIELD(smooth_base, $
                        title='Smooth Interval:', $
                        value=aPlotInfo.ioptions(0), $
                        uvalue='smooth_entry', /integer, xsize=2)
  
  bad_base=WIDGET_BASE(choices_base, ROW=1)
  bad_entry=CW_FIELD(bad_base, $
                        title='Bad Data Value:', $
                        value=aPlotInfo.soptions(0), $
                        uvalue='bad_entry', /string, xsize=12)  
  
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   

;--- add the papco panel widget for choosing the color bar for color plots ----
;  widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_panel_AddYScale_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
;  widgets=papco_panel_AddZScale_Widgets(panel_base, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the papco_file example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
  thisWidgetsData={helpfile:'papco_file_panel.help', $
                   column_label:column_label, $
                   time_format:time_format, $
                   mpb_type:mpb_type, $
                   channel_names:'', $
                   smooth_entry:smooth_entry, $
                   bad_entry:bad_entry, $
                   ioptions:aPlotInfo.ioptions, $
                   soptions:aPlotInfo.soptions, $
                   typevector:aPlotInfo.typevector}
  widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)
  
  
;--- realize the window -------------------------------------------------------
  papco_Panel_Realize, panel_base, widgetData
  
;--- refresh if needed
  
  if widgetData.filename ne '' then papco_file_panel_refresh, /CHANNEL
  
  papco_xmanager, 'papco_file_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
  returnData={typevector:widgetData.typevector, $
             soptions:widgetData.soptions, $
             ioptions:widgetData.ioptions}
  additionalRData=papco_panel_ReturnData(widgetData)
  
  return, create_struct(ReturnData, additionalRData)
   
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_file_panel_refresh, ALL=ALL, SATID=SATID, SATlist=SATlist,
;*                              CHANNEL=CHANNEL
;*
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      if set (e.g. /CHANNEL), only the widgets influenced by the information
;*      that the keyword's name describes, are updated.
;*      If the keyword /ALL is set, all of the window is updated.
;*      
;*      ALL     - all
;*      CHANNEL - energy channel
;*
;*
;* CALLING SEQUENCE:
;*      have a look the the event-routine papco_file_Panel_event
;******************************************************************************
PRO papco_file_panel_refresh, ALL=ALL, CHANNEL=CHANNEL
  
  COMMON papco_file_panel_data, widgetData, channel_set
  common get_error, get_err_no, get_err_msg
 ; case of

;  endcase
 

 
  IF KEYWORD_SET(ALL) THEN BEGIN
    CHANNEL=1
  ENDIF  
  
  IF KEYWORD_SET(CHANNEL) THEN BEGIN
  

    r_papco_file,widgetData.filename,widgetData.soptions, $
      input_header, input_data, /HEADER
    if get_err_no ne 0 then return
        stop
    header_list=str_sep(input_header.COLUMN_ID,' ')
    n=n_elements(header_list)
    n_time_tags=input_header.N_TIME_TAGS
        
    for i=n_time_tags,(n-1) < (119+n_time_tags) do begin
      widget_control,WidgetData.mpb_type(i-n_time_tags),$
        SET_VALUE=strcenter(header_list(i),16)
      widget_control,WidgetData.mpb_type(i-n_time_tags),sensitive=1
    endfor  
    for i=n-n_time_tags,119 do begin
      widget_control,widgetData.mpb_type(i),$
        SET_VALUE=strcenter(string(i,format="('empty ',i3.3)"),16)
      widget_control,widgetData.mpb_type(i),sensitive=0
    endfor     
    stop
  ENDIF  
    
end  

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_file_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO papco_file_panel_event, event
  
  COMMON papco_file_panel_data, widgetData, channel_set

  WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.
  stop
  old_filename=widgetData.filename
  
  if strleft(uval, 8) EQ 'mpb_type' then begin
    pressed=fix(strfrom(uval,8))
    if channel_set(pressed) eq 0 then channel_set(pressed)=1 $
    else channel_set(pressed)=0
    widgetData.typevector(0)=0
    WidgetData.typevector(1)=0
    for i=0,59 do $
      if channel_set(i) then $
        WidgetData.typevector(0)=WidgetData.typevector(0)+2ll^i 
    for i=60,119 do $
      if channel_set(i) then $
        WidgetData.typevector(1)=WidgetData.typevector(1)+2ll^(i-60)
    return
  endif  
    
  WIDGET_CONTROL, widgetData.smooth_entry, GET_VALUE=x
  widgetData.ioptions(0)=x
    
  WIDGET_CONTROL, widgetData.bad_entry, GET_VALUE=x_str
  widgetData.soptions(0)=x_str
  
  WIDGET_CONTROL, widgetData.column_label, GET_VALUE=x_str
  widgetData.soptions(1)=x_str
  
  WIDGET_CONTROL, widgetData.time_format, GET_VALUE=x_str
  widgetData.soptions(2)=x_str
  
  widgetData=papco_panel_Event(event, widgetData)
    
  if widgetData.filename ne '' then begin
    if widgetData.filename ne old_filename then begin
      papco_file_panel_refresh, /CHANNEL
    endif  
  endif   
    
END
