;******************************************************************************
;* file p_papco_file_line.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_papco_file_line, panel, type, ioptions, soptions, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_papco_file_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_papco_file_line, panel, type, ioptions, soptions, $
                      input_header, input_data, $ 
                      OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_papco_file_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, swittch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
  common yscale, yscl                             ;man/auto yscaling
  common shift_label, down                        ;for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  ;info on coords
  
  panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.
  ytickno=50*(!p.position(3)-!p.position(1))
  if ytickno gt 10 then ytickno=10
  ytickno=fix(ytickno) > 1
   
; if yaxis is logarithmic, allow automatic yticks.
  if yscl(panel(0),3) eq 1 then ytickno=0

; since there is a common time axis plots look better if they all use the same
; number of xticks. In papco, this number is 6 by default. This is set by papco
; using the IDL graphics system variable !X.TICKS.
; You can override this for your own plot by setting the variable
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by papco to default values that guarantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options, different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.
; In this example the data array consist of flux values in ten channels 
; at different energies, typical of particle spectrometer data. Here I use the
; channel information passed in the plotinfo structure from the panel editor
; to select the channel to be drawn.

; set the begin end end times. papco convention is to use running seconds
; starting with the no of seconds since the beginning of the start day.
  time=double(input_data.(0))+double(input_data.(1))/86400.0
  
  xut1=mjdt_start.mjd+mjdt_start.t/86400.0
  xut2=mjdt_end.mjd+mjdt_end.t/86400.0
  
  channel_list=str_sep(input_header.COLUMN_ID,' ')
  channel_list=channel_list(input_header.N_TIME_TAGS:n_elements(channel_list)-1)
  
  ytit_list=str_sep(input_header.COLUMN_UNITS,' ')
  
  ;set mask for channels to plot
  ch_set=bytarr(120) 
  for i=0,59 do $
    if (type(0) and 2ll^i) eq 2ll^i then ch_set(i)=1 else ch_set(i)=0
  for i=60,119 do $
    if (type(1) and 2ll^(i-60)) eq 2ll^(i-60) then ch_set(i)=1 else ch_set(i)=0
  
  index=where(ch_set ne 0,c)
  if c eq 0 then begin
    message,'No variables chosen', /cont
    plotted_x = !x  &  plotted_y = !y
    return
  endif  
    
  uytitle=ytit_list(index(0))
  yray=input_data.(index(0)+2+input_header.N_TIME_TAGS)
  
  fln=strmid(input_header.FILE_NAME,0,strlen(input_header.FILE_NAME)-4)
  
  utitle='papco_file!C'+fln+'!C'+channel_list(index(0))

; restrict data to actual time range requested - this makes for faster zooming
  index=where((time ge xut1) and (time le xut2),count)
  if count ne 0 then begin
    yray=yray(index)
    time=time(index)
  endif
  
;smooth if needed, take out bad values is needed
  if soptions(0) ne '' then begin
    message,'Removing bad data',/cont
    bad_value=float(soptions(0))
    index=where(yray ne bad_value,c) 
    if c ne 0 then begin
      yray=yray(index)
      time=time(index)
    endif  
  endif  
  
  if ioptions(0) gt 2 then begin
    message,'Smoothing data',/cont
    yray=smooth(yray,ioptions(0))
    utitle=utitle+'!Csmooth: '+varprt(ioptions(0))
  endif  
  
; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; the procedure YLIM3 is optional and sets the ylimit automatically for the
; data leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; (in $papco/papco_Lib/ylim.pro)

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.  
  if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)
    yen=yscl(panel(0),2)
  endif else begin
    if yscl(panel(0),3) eq 1 then ylim3,time,yray,yst,yen $ ;for log plots
      else ylim4,time,yray,yst,yen                   ;for lin plots
    yscl(panel(0),1)=yst
    yscl(panel(0),2)=yen
  endelse
  
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
  extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
                 ylog:yscl(panel(0),3), $
                 yticks:ytickno, ytitle:uytitle, xtickformat:'noticks'}
   
; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 
  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse     
   
; add keyword structure set here with the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; change the xtickformat if the panel is the bottom one and needs to
; plot a time axis   
  if panel(0) eq 0 then extra_plotPar.xtickformat='mjd_dec_2'
  
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.
   
; "down" sets the axis labeling level. 
  down=0 

; To keep things general, plotting of the data is split up into plotting the 
; axis with no data, and then plotting the data with no axis. This is done to 
; enable the feature of overplots, which just plot data into an existing axis.
; plot the axis in color=1, and only plot axis if if you're NOT over-plotting!
; down sets the axis labeling level (adds carriage returns [!C] to labels)
  down=0
  color1_plotpar=extra_plotPar
  color1_plotpar.color=1
  if not keyword_set(overplot) then plot,time, _extra=color1_plotPar, /NODATA
  
; now do the actual data plot, data only, no axis
  plot,time,yray,  _extra=extra_plotPar
 
; store the coordinate information into common block for slices
   plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; The routine x_side_label puts x-axis label to the right of the axis.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; Or use any other function you define yourself.
; down sets the axis labeling level
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    extra_plotPar.xtickformat='mjd_dec_1'
    extra_plotPar.color=1     
    down=1
    plot,time,yray, _extra=extra_plotPar, /NODATA
    x_side_label,panel,'time (UT)!CDate'
  endif

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION

  if not keyword_set(overplot) then begin
    right_side_label,panel,utitle,_extra=extra_Par
  endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
  endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_papco_file_line: Writing plot data out to file'
    description='papco_file template sample data - single energy channel line'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
  endif  

end 
