;******************************************************************************
;*  NAME:        r_papco_file
;*
;*  DESCRIPTION: Routine to read papco_file files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*               FILES ARE EXPECTED TO BE IN PUM-FILE FORMAT!!!!!!
;*
;*  INPUT:       fln - filename to read
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               HEADER   -  only reads header of file
;*
;*  DATE:        July 1999
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_papco_file, fln, soptions, input_header, input_data, $
                  VERBOSE=VERBOSE, HEADER=HEADER

  common r_papco_file,  column_label, time_format
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

  get_err_no=0
  get_err_msg='% r_papco_file: '+fln+' read succesfully'
  
  column_label=strtrim(soptions(1),2)
  time_format=strtrim(soptions(2),2)
   
; check for env. variable pointing to data directory
  data_dir='PAPCO_FILE_DATA'
  path=papco_getenv(data_dir)
  if strlen(path) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif
   
; check for existence of data directory
  cc=papco_finddir(path)
  if cc eq 0 then begin
    get_err_no=2
    get_err_msg='Data directory '+path+' not found'
    print,get_err_msg
    return
  endif
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
  
; here we only read one file by definition. reading over file
; boundaires is not possible!!!
   
  ; read one file of data
  r_papco_file_onefile, path, fln, input_header, input_data, $
    VERBOSE=VERBOSE, HEADER=HEADER
  
; check if any files were read   
  if (get_err_no ne 0) then begin
    get_err_no=1
    get_err_msg='% r_papco_file: no file found'
    return
  endif     
  
 if keyword_set(HEADER) then return
 
; restrict data to actual time range requested
  ut1=mjdt_start.mjd
  ut2=mjdt_end.mjd
  index=where((input_data.(0) ge ut1) and (input_data.(0) le ut2),count)
  if count ne 0 then begin
    input_data=input_data(index)
  endif   

end

;******************************************************************************
;*  NAME:        r_papco_file_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_papco_file_onefile, path, fln, input_header, input_data, $
                          VERBOSE=VERBOSE, HEADER=HEADER
  
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0
  get_err_msg='papco_file onefile read succesfully'
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
  
  if keyword_set(HEADER) then goto,read_original
  
  on_ioerror, read_original
   
  restore,path+'savesets/'+fln+'.idl',/verbose
  dealloc_lun,/quiet    ;frees logical units used by save/restore
  message,'idl saveset read succesfully', /cont
  return

  read_original:
  ; if the .idl save file could not be found, then try to read the original
  r_papco_file_onefile_original, path, fln, input_header, input_data, $
    VERBOSE=VERBOSE, HEADER=HEADER
      
end   
  
;******************************************************************************
;*  NAME:        r_papco_file_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_papco_file_onefile_original, path, fln, input_header, input_data, $
                                   VERBOSE=VERBOSE, HEADER=HEADER
  
COMMON r_papco_file, column_label, time_format
COMMON get_error, get_err_no, get_err_msg  
   
get_err_no=0
get_err_msg='papco_file_onefile_orignal read succesfully'
   
; clear data structures for reuse
input_header=0  &  input_data=0  
instr=''
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!

CASE source OF

    'CIS_HIA_pad' : BEGIN       ;for obj6_Pab1_CIS2_P09...
;        ;open file
;        ;search for '!                                  Data';

;        ;read a line, convert time, start time
        ;array, readin in to pa array until
        ;next time.
  
;       data = fltarr(360)
;       dummy = {tai:0.0d, data:data}

;       instr = ''

;       c = 0
        ;loop thropugh file
;       WHILE NOT eof(u) DO BEGIN 

;           readf,unit,instr
            ;seperate instr into time / data
;            r = strsplit(instr, ',', /EXTRACT}
           ;r(0) has time string -> convert to tai, out in input data(c).tai
           ;r(2) has PA
           ;r(5) has flux;
;          data_idx = fix(r(2))+179
;          data(data_idx) = float(r(5))

;           IF fix(r(2) EQ 179 THEN c = c+1

 ;       IF c EQ 0 THEN input_data = dummy ELSE input_data = [input_data, dummy]
        
  ;           
;       endwHILE
   ;     close, u & free_lun, u

 END
 
  ELSE: BEGIN
  
  ;make the "create_struct" function call needed to make input_header 
  call_prm="name='"+fln+'_header1'+ $
    "','LABEL','PAPCO PUM FILE','COLUMN_ID','','COLUMN_UNITS',''"+ $
    ",'FILE_NAME','','N_TIME_TAGS',0,'TIME_FORMAT',''"
  result=EXECUTE('input_header=create_struct('+call_prm+')')
  input_header.FILE_NAME=fln
  
  ;make the "create_struct" function call needed to make input_header 
  ;for that read the column header form the data file 
  on_ioerror, out  
  openr,unit,path+fln,/get_lun
  ;get to header line. Assume beginning of line contains column_label
  while not eof(unit) do begin  
    readf,unit,instr  &  instr=strtrim(instr,2)
    test=strmid(instr,0,strlen(column_label))
    if test eq column_label then goto, found_column_id
  endwhile
  
  get_err_no=1
  get_err_msg='No column_id line found in file'
  message,get_err_msg,/cont
  return
  
  found_column_id:
  message,'Found column_id line: '+test,/cont
  n_time_tags=n_elements(str_sep(strcompress(test),' '))
  column_id=strcompress(instr)
  column_list=str_sep(column_id,' ')
  n_col=n_elements(column_list)
  input_header.COLUMN_ID=column_id
  input_header.N_TIME_TAGS=n_time_tags
  input_header.TIME_FORMAT=time_format

  n=n_elements(column_list) & initval=fltarr(n)
  call_prm="'TIME_MJD',0l,'TIME_DAYSEC',0.0"
  result=EXECUTE('dummy=create_struct('+call_prm+')')
  for i=0,n-1 do begin
    call_prm="dummy,'"+column_list(i)+"',"+string(initval(i),format="(f3.1)")
    result=EXECUTE('dummy=create_struct('+call_prm+')')
  endfor  
  call_prm="name='"+fln+'_data'+"',dummy,'"+column_list(n-1)+"',"+ $
    string(initval(n-1),format="(f3.1)")   
  result=EXECUTE('dummy=create_struct('+call_prm+')')  
  
  if keyword_set(HEADER) then begin
    close,unit  &  free_lun,unit   
    return
  endif
  
  ;read units info line if exists.
  readf,unit,instr
  print,instr  
  input_header.COLUMN_UNITS=strtrim(strcompress(instr),2)
  
  message,'Reading original format .... (slow): '+fln,/cont
  
  ;make a data array 
  ndat=2000
  input_data=replicate(dummy, ndat)
  
  c=0l
  while not eof(unit) do begin  
    readf,unit,instr
    instr=strtrim(strcompress(instr),2)
    instr_list= str_sep(instr,' ')
    if n_elements(instr_list) ne n_col then goto, next

    for i=0,n-1 do dummy.(i+2)=float(instr_list(i))
    
    if c eq n_elements(input_data) then begin
      message,'Increasing data array to '+ $
        varprt(n_elements(input_data)+ndat) ,/cont
      input_data=[input_data,replicate(dummy, ndat)]
    endif
    
    input_data(c)=dummy
    
    c=c+1
    next:
  endwhile
  
  input_data=temporary(input_data(0:c-1))
    
  close,unit  &  free_lun,unit
  
  ;now use the time format info to convert time tags to mjdt
  print,''
  message,'Converting time to mjdt.... (slow)',/cont
  
  for i=0,c-1 do begin
    result=papco_file_format(input_data(i),n_time_tags, time_format)
    input_data(i).(0)=result.(0)
    input_data(i).(1)=result.(1)    
  endfor  

END 

ENDCASE 

; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
  save,filename=path+'savesets/'+fln+'.idl',input_header,input_data,/verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='% r_papco_file_onefile:!C '+!ERR_STRING
  
  
end

