;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  papco_text_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.papco_text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=papco_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*      written February 1997, Reiner Friedel
;***************************************************************************
FUNCTION  papco_text_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON papco_text_panel_DATA, widgetData
  
; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'
   
   dirpath=papco_getenv('PAPCO_TEXT_DATA')
   filter='.panel_text'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=papco_widget_base(title=aPlotInfo.panelKind,$
	                        column=1, /modal, $
	                        GROUP_LEADER=group)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
 
;--- some panels might need to read/input to papco_text files  ----------------
;    So a generic papco widget is supplied for this.
   widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo,$
                                       DIRPATH=dirpath, FILTER=filter)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- add some widgets for TEXT height, thickness ------------------------------
   format_base=widget_base(panel_base, ROW=1,/frame)
   fld_text_thick=cw_field(format_base, /FLOATING, /ALL_EVENTS, $
                           TITLE='Text thick:',$
                           VALUE=aPlotInfo.options(0), $
                           uvalue='fld_text_thick')
   fld_text_size=cw_field (format_base, /FLOATING, /ALL_EVENTS, $
                           TITLE='Text size:',$
                           VALUE=aPlotInfo.options(1), $
                           uvalue='fld_text_size')   
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'papco_text_panel.help', $
                    options:aPlotInfo.options}
   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
   papco_xmanager, 'papco_text_panel', panel_base

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={options:widgetData.options}
      
   additionalRData=papco_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_text_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written February 1996, Haje Korth
;******************************************************************************
PRO papco_text_panel_event, event

  COMMON papco_text_panel_data, widgetData

  WIDGET_CONTROL,GET_UVALUE=uval, event.id
  
  case uval of 
    
    'fld_text_thick' : begin
      widget_control,event.id, get_value=thick
      widgetData.options(0)=thick
    end  
    
    'fld_text_size' : begin
      widget_control,event.id, get_value=size
      widgetData.options(1)=size
    end  
    
    else: widgetData=papco_panel_Event(event, widgetData)
    
  endcase  
   
; use the options(19) tag to indicate that the panel has been updated.
  if widgetData.canceled eq 0 then $
    widgetData.options(19)=widgetData.options(19)+1

END

















