;******************************************************************************
;* file papcoadd_papco_text
;*      papcoadd_papco_text
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a papco_text. 
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_papco_text,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_papco_text, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO canget_papco_text, canget
;*        -> Sets a variable to indicate output to file is supported.
;*
;*    PRO descriptor_papco_text, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_papco_text
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_papco_text, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           papco_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_papco_text, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_papco_text,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this papco_text to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      papco_texts.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_papco_text,plottypes_base,buttonGeometry,pb_plottype
;******************************************************************************
pro addplottype_papco_text,plottypes_base,buttonGeometry,pdm_papco_text

; VALUE assigns a label for the button
; The papco_text portion of UVALUE is used to construct the calls to all
; other procedures in this add-in needed by papco.
  
  MenuDesc = [ $
        { CW_PDMENU_S, 3, strcenter('PUM Text',10) }, $ 
        { CW_PDMENU_S, 0, ' configure papco_text ' }, $
        { CW_PDMENU_S, 2, ' plot papco_text      ' } ]  
  
  pdm_papco_text = CW_PDMENU(plottypes_base, MenuDesc, $
                            UVALUE='pdm_papco_text')  

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_papco_text
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the papco_text.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_papco_text
;******************************************************************************
pro addpanel_papco_text,event

  common get_error, get_err_no, get_err_msg
  
; The FIRST time you use this module you will have to configure it,
; i.e. set the data paths etc. Once you have done this PAPCO will
; write a papco_text.config file into your papco_user_XX/papco_modules_defaults
; directory. 
; So here we check if this papco_text.config file exists, and if not,
; we disallow the execution of any other option apart from "configure
; papco_text". In this way the user is forced to configure the module
; before using it the first time!
  
  r_module_config, strlowcase('papco_text'), config_data, msg
  if (get_err_no ne 0) and (event.value ne 1) then begin
    print,'%addpanel_papco_text: You need to configure the module first'
    papco_module_configure,'papco_text'
    return
  endif  
  
  panelKind='papco_text'
 
  CASE event.value of
    1: BEGIN
        papco_module_configure, panelKind
    end
    2: BEGIN
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.panelheight=1
        defaults.options(0:3)=[1.,1.,0.,0.]
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    else: begin
      print,'%addpanel_papco_text: Nothing defined for this submenu entry'
      return
    end  
  endcase  
  
  ;panelKind='papco_text'
  ;panelEditor='papco_text_panel'
  ;defaults=papco_getPlotInfoStruct()
  ;defaults.typeVector=[0,0,0,0]
  ;defaults.panelheight=1
  ;papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_papco_text, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by CRRES orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_papco_text, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_papco_text, plottable

   plottable=1

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_papco_text, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that papco_text support the OUTPUT
;*      functionality. 
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_papco_text.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                               1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_papco_text, canget
;******************************************************************************
pro canget_papco_text, canget

   canget=  1      ; 1 -> plot routine can return just data to file
                   ; 0 -> feature disabled
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_papco_text, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.papco_text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_papco_text, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_papco_text, plotInfo, descriptStr

   descriptStr=descriptStr+'papco_text Panel: File '+plotInfo.filename

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_papco_text, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      papco_EditPlot (papco_paneledit.pro).
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                  the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_papco_text, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_papco_text, plotNumber, newPlotData
  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=papco_text_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_papco_text, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype.
;*      
;*
;* INPUTS:       
;*      panelVector      position of the plot (see papco_conventions.papco_text, 6)
;*      currPlotInfo      structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.papco_text) 
;*
;* OUTPUTS: 
;*      get_data_call   a string containing the data call. Used by other
;*                      routines that need to know how to get the data
;*                      in a panel, e.g. to draw spectra, distrib. functions
;*      IsRead          a boolean which is set when the the data is read the
;*                      first time. Useful if a given data-read puts data
;*                      into commons which can be plotted in many ways.
;*                      When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 swittches. Currently
;*                      These are supported (this may be extended later)
;*                       1 = mouse zooms 
;*                       2 = mouse slices
;*                       4 = mouse calls level zero
;*                       8 = mouse calls polar slice
;*                       16 = mouse views ephemers
;*                       32, 64, 128 = not supported yet
;*                       0 = disable all
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      draw_papco_text, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_papco_text, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead

  common extra, extra_PlotInfo  ; passes in papco-set plot keywords structure

  get_data_call=''
  
; set up which mouse functions to allow
  slice_type=0
  
  IF NOT IsRead THEN BEGIN ; check if data has already been read
    r_papco_text, currPlotInfo.filename
    ; check the read for success  
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1 $
    ELSE IsRead=0
  ENDIF
  
  IF IsRead THEN BEGIN
    IsRead=0 ; always re-read
    p_papco_text, panelVector, currPlotInfo, _EXTRA=extra_PlotInfo
  ENDIF

END
