;******************************************************************************
;* file plot_type_slice.pro
;*
;* this file contains the routines needed to enable the extra mouse features
;* provided for in papco. 
;*
;* Currently possible:
;*
;*    mouse zooms                        no routine here, in papco main
;*    mouse vertical slice               routine here
;*    mouse views slice (polar)          specialty slice - no routine here
;*    mouse views ephemeris              under development (POLAR only)
;*    mouse accesses level zero          routine here
;*    mouse writes paneldata             routine here
;*    mouse horizontal slice             routine here
;* 
;* Here the procedure names must again adhere to a strict naming convention so 
;* that they can be automatically called when needed.
;*
;* The slice types supported by these routines must be ENABLED by the user, so 
;* that papco can call them. This is done by setting the value of 
;* slice_type in draw_plot_type.pro (contained in papcoadd_plot_type.pro).
;* slice_type is defined in the following way:
;*
;*   slice_type:   Selects which slices are supported and for which slice
;*                 routines have been written.
;*                 This is an integer logical with 16 swittches. Currently
;*                 there are 5 papco slices. You may extend this by
;*                 further USER slices up to a total of 16 slices.
;*                     0 = disable all slices
;*                     1 = papco Zoom                 (2^0)
;*                     2 = papco View Slice           (2^1)
;*                     4 = papco Access level Zero    (2^2)
;*                     8 = papco Write Paneldata      (2^3)
;*                    16 = papco Hair Cursor          (2^4)
;*                    32 = papco horizontal Slice     (2^5)    
;*                    64 = USER slice A or not used   (2^6)
;*                   128 = USER slice B or not used   (2^7)
;*                    "     "     "   "  "  "    "      "    
;*                 32768 = USER slice Z or not used  (2^15)
;*                        
;*   You can support more than one type of slice by arithmetically adding the
;*   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* The "papco Zoom" function uses the mouse to define an area of a panel. 
;* Dragging the mouse with the left button pressed marks out a shaded rectangle
;* in the panel. The time range and y-range of the rectangle are read out in a
;* window and the user has the option of adopting these new ranges. The papco 
;* main common block for time (common mjdt, mjdt_start, mjdt_end) and y-range
;* (common yscale, yscl) are updated accordingly. Provided within PAPCO.
;*
;* The "papco Views Slice" function uses the time returned by papco and the
;* get_data_call specified for the plot_type to read the data and to draw a 
;* slice at the clicked position. This is commonly used in spectra color plots 
;* to draw a spectrum at that time, or a pitch angle distribution etc, or to
;* call up any other plot at that time as specified by the user. 
;* -> user defined interface routine needed, template included here
;*
;* The "papco Accesses level Zero" uses the time passed and calls up another
;* widget application to read and work with level zero data. This application
;* must be entirely independent of papco and is only called from papco via
;* a mouse click in the panel required for the time required.
;* -> user defined interface routine needed, template included here
;*
;* The "papco Write Paneldata" function allows the user to read off data from 
;* panels using the mouse and to write that data to a file. This is a papco
;* supplied utility slice which by default only writes the time and y-value of
;* the mouse-click to file. The user can however interface his own routine to
;* add auxiliary data to the string written to file.
;* The output file is written in ASCII and has the same format as the .event
;* files used by the event times utility module. The files can thus be used
;* in the events time module to mark events on panels.
;* -> user defined interface routine possible, template included here
;*
;* The "papco Hair Cursor" function draws a large hair cursor and outputs
;* x,y,and z corrdinates onto the plot.  Provided within PAPCO.
;*
;* If your new data contains several sub-data types, one slice and/or levelzero
;* routine has to be provided for each type, since in general they would do
;* different things. See 
;* $papco_modules\crres_meb\papco_interface\crres_slices.pro  
;* for such an example.
;*
;*      PRO DrawSlice_plot_type, panelNr, seconds, yValue
;*        -> Draw a slice for plot_type
;*
;*      PRO levelzero_plot_type, seconds, yValue
;*        -> This procedure calls the widget for levelzero data access 
;*           for this plottype.
;*
;*      PRO WriteSlice_plot_type, panelNr, seconds, yValue
;*        -> This procedure calls the widget for writing panelinfo data to file
;*    
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;*      modified may 1997, Reiner Friedel
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO DrawSlice_plot_type, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for plot_type
;*      calling a separate window for the plot.
;*      slice_type in draw_plot_type must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;******************************************************************************
PRO DrawSlice_plot_type, panelNr, seconds, yValue

   COMMON PAPCO_DRAWSLICE, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; papco uses the the getdatacall you set in papco_add_plot_type to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 

   common plot_type_data, input_header, input_data
   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt

   time=input_data.time           ; dummy, remove once your own is here

   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector

; set the panel window for the slice to full window

   !p.position=[0.1,0.1,0.9,0.9]

; select the index for the time chosen out of the time array

   DiffTimes=ABS(time - seconds)        ; this sets timeIndex to the index with
   result=MIN(DiffTimes, timeIndex)     ; t90(index) is closest to 'seconds'

; these two info strings are user-definable output strings to label the slice 
; window. To calculate the time use the start time in mjdt and the seconds 
; passed in. If your plot uses the papco philosophy then the time axis is in 
; running seconds since the start of the start day.
;
; However, you of course can use your own time format, then the information
; in seconds is whatever your time format is. papco provides a function
; to return a verbose time for ANY time format.    
; papco always knows the start end time of a plot from the time common
; block, and the start/end value of the x_axis shouldalways be stored in 
; papco by the user's plot routine:;
;
;   common coordinateSystems, plotted_x, plotted_y  
;
;   plotted_x = !x
;   plotted_y = !y     -   after the user has set up his axis.
;
; If this is done properly then the call
;   
;   time_str=papco_cursor_time(seconds,panelNr)
;   
; should return the correct time.
   
   time_str=papco_cursor_time(seconds,panelNr)
   time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)
   
   info1='You got to the default slice window'
   info2=string(time_str, yValue, $
                format="('time: ',A20,'   Y-value: ',f9.2)")

; Replace these labels and include your plot! 

; user defined routine for plotting the slice included here. An example is in 
; $papco_MODULES/crres_MEB/papco_interface/crres_slices.pro
; You typically need to include the commons from the read routine, a way
; of finding the index of the data you want to plot using the time "seconds"
; plus a simple plot statement to plot the data.
; Two strings, info1 and info2, can be used to annotate the slice window
; by providing additional information, ie ephemeris data, etc.

; these calls put your info strings in the provided labels of the slice widget

   WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
   WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO levelzero_plot_type, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*      This procedure calls the widget for level zero data access 
;*      for this plottype. 
;*      slice_type in draw_plot_type must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*      seconds           T90 time at which mouse is clicked
;*      yvalue            y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      levelzero_plot_type, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel at Goddard
;******************************************************************************
pro levelzero_plot_type, seconds, yvalue

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $papco_MODULES/addpapco_swe_plots.pro.
; Provided as parameters to be passed to the new application, are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be swittched to (e.g. to get a new
; time from the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('papco_main') 
; to swittch back to the papco main calling widget. 

; call the levelzero widget.  
;   plot_type_levelzero, seconds, yvalue

; the following line is printed on your xterm as a check to see if papco
; correctly called this routine. If there is a problem make sure your naming 
; convention is O.K. : This routine MUST be called levelzero_plot_type, 
; where plot_type is the name of the panel you have set in papcoadd_plot_type.

    print, 'got to the levelzero interface  ',seconds, yvalue

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO WriteSlice_plot_type, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a Write Paneldata slice for plot_type
;*      calling a seperate window for the widget
;*      slice_type in draw_ceppad must be set to +32 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      READDATA    if set the data needs to be re-read from file
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      may 1997, written by R.Friedel
;******************************************************************************
PRO WriteSlice_plot_type, panelNr, seconds, yValue,READDATA=READDATA

   COMMON PAPCO_WRITE_PANELDATA, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData

; THIS TEMPLATE ACTUALLY DOES NOTHING, ONLY THE DEFAULT WRITE SLICE 
; FUNCTIONALITY WILL WORK (WRITE TIME AND Y-VALUE OF MOUSECLICK TO FILE). THE 
; EXAMPLE HERE IS TO SHOW HOW ONE COULD ADD ADDITIONAL INFORMATION.

; add in here any other common blocks you would need to make your additional
; data available (ie ephemeris).

; papco uses the the getdatacall you set in papco_add_plot_type to read the
; plotdata For your panel. As long as you stay in one panel with the 
; mouse, the data does not need to be read again (READDATA=0). When you swittch
; panels, new data needs to be read (READDATA=0). It's up to the user to read 
; in the data he needs here. 

;   common plot_type_data, input_header, input_data
;   common plot_type_ephem, plot_type_ephem
;   common mjdt, mjdt_start, mjdt_end    ;common time limits in mjdt

   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector
   getdatacall=thisPlot.get_data_call

;read the data if needed

;   if READDATA then begin 
;      callgood=execute(getdatacall)
;      r_plot_type_ephem
;   endif

; select the index for the time chosen out of the time array

;   DiffTimes=ABSplot_type(.time - seconds)
                                        ; this sets timeIndex to the index 
;   result=MIN(DiffTimes, tidx)         ; closest to 'seconds'

; these strings are user-definable output strings to label the slice window.
; The "write Panelinfo" window papco supplies has 6 lines of user-definable
; text. 

   info1='User label 1'
   info2='User data  1'
   info3='User label 2'
   info4='User data 2'
   info5='User label 3'
   info6='User data 3'

; Time and y-value of mouse click are automatically put into labels.
; The calls below put your info strings in the provided user labels (max 6).
;
; NB NB NB:
; However, only info2, info4, info6 are written to file in the form
; info2+info4+info6.
;
; The idea is that info1 will have the labels for the data in info2, info3 the
; labels for the data in info4, and info5 the labels for the data in info6.
; Time and y-value of mouse click are written first, then info2+info4+info6
; followed by a user defined comment from the Write Paneldata window.

   WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
   WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
   WIDGET_CONTROL, sliceWidgetData.lbl_slice3, SET_VALUE=info3
   WIDGET_CONTROL, sliceWidgetData.lbl_slice4, SET_VALUE=info4
   WIDGET_CONTROL, sliceWidgetData.lbl_slice5, SET_VALUE=info5
   WIDGET_CONTROL, sliceWidgetData.lbl_slice6, SET_VALUE=info6

END
