;------------------------------------------------------------------------------                          
; File: BUFF_CAM.PRO    CAMMICE Level Zero Data Buffer Routines                                
; Revision: 01/19/97    J L Roeder                                
;-----------------------------------------------------------------------------                          
; Routines:                          
;       def_camrec_strc Define the structure to hold a CAMMICE data record                            
;       init_camLZ      Initializes CAMMICE buffer common block for new file                                     
;       cam_record      Extracts a particular record from CAMMICE LZ buffer                          
;       buff_cam        Buffers CAMMICE Level Zero Record into DRH and data                          
;       tspose1         Transposes first two indices of a third rank array                          
;       tspose2         Transposes first two indices of a third rank array                          
;------------------------------------------------------------------------------                          
pro def_camrec_strc, strc, nbytes                        
                        
;       Define the structure to hold a CAMMICE data record                            
                        
;      Define the data record header structure using the routine in HEAD_LZ.PRO                        
def_drh_strc, drh_nbytes, drh                         
                                        
strc = { Cammice_Record,             $                                   
        data_rec_hdr: drh,           $ ; Data Record Header Stucture                          
        hk: bytarr( 250, 2),         $ ; Housekeeping data array                        
        subcom: bytarr( 250, 2),     $ ; Subcomm data array                        
        mics_fixed: bytarr( 50, 15), $ ; Fixed column mics data array                        
        hit_fixed: bytarr( 50, 15)   $ ; Fixed column mics data array                        
        }                        
                        
nbytes = n_tags( strc, /length)                        
                        
end                        
;------------------------------------------------------------------------------                          
pro init_camLZ, lunit, nbytes, nfill, reclast                   
                      
;       Initializes CAMMICE buffer common block to start new file                                     
                                     
; Input:                    
;      lunit          Logical unit number of open data file                   
;      nbytes         Number of bytes per record                   
;      nfill          Number of fill bytes per record                   
;      reclast        Number of last record in file                   
                    
; Output: None                      
                                                    
common camLZ_buff,    $ ; Block holds many records of data                     
        unit,         $ ; Logical unit number of data file                     
        numbytes,     $ ; Number of bytes per record                     
        numbfill,     $ ; Number of fill bytes per record                      
        lastrec,      $ ; Number of last record in data file                   
        recrng,       $ ; Record number range, first record is one                       
        recstruc        ; Record structure array                         
                 
;   Initialize to values for current data file                  
unit = lunit                     
numbytes = nbytes                     
numbfill = nfill                                     
lastrec = reclast                   
                 
;   Record number range is set to empty buffer                 
recrng = [-1L, -1L]                     
                   
;   Initialize buffer on first execution only                   
if n_elements( recstruc) eq 0L then begin                   
                   
;   Define structure to hold one cammice record                   
    def_camrec_strc, strc, nlength
                   
;   Buffer sized to hold optimum number of records for each machine                    
;   The user may adjust nrecs to achieve optimum efficiency, but note that                   
;   program must be recompiled completely from scratch if nrecs is changed.                   
    nrecs = 400L                   
    recstruc = replicate( strc, nrecs)                       
                   
endif                   
                                                                                           
end                                     
;------------------------------------------------------------------------------                        
pro cam_record, rec_numb, drh1, hk1, subcom1, mfixed1, hfixed1, err                          
                          
;   Extracts a particular record from the CAMMICE LZ buffer                          
                          
; Inputs:                          
;   rec_numb        Record number, 1-9391                          
                          
; Outputs:                          
;   drh1            Data record header for this record                          
;   hk1             Housekeeping bytes for this record                          
;   subcom1         Subcom bytes for this record                          
;   mfixed1         MICS fixed-colummn bytes for this record                          
;   hfixed1         HIT fixed-colummn bytes for this record                          
                          
common camLZ_buff,    $ ; Block holds many records of data                     
        unit,         $ ; Logical unit number of data file                     
        numbytes,     $ ; Number of bytes per record                     
        numbfill,     $ ; Number of fill bytes per record                      
        lastrec,      $ ; Number of last record in data file                   
        recrng,       $ ; Record number range, first record is one                       
        recstruc        ; Record structure array                         
        
if n_elements( recrng) eq 0 then recrng = [-1, -1]        

if (rec_numb ge recrng(0)) and (rec_numb le recrng(1)) then begin                          
                          
    err = ''                   
    index = rec_numb - recrng(0)                       
    rec_struc = recstruc( index)                        
    drh1 = rec_struc.data_rec_hdr                          
    hk1 = rec_struc.hk                          
    subcom1 = rec_struc.subcom                          
    mfixed1 = rec_struc.mics_fixed                          
    hfixed1 = rec_struc.hit_fixed                          
                     
endif else begin                      
                      
    err = 'cam_record: record not in buffer'                               
    rec_numb = -1L                            
    drh1 = -1B                             
    hk1 = -1B                            
    subcom1 = -1B                            
    mics_fixed1 = -1B                            
    hit_fixed1 = -1B                            
                       
endelse                      
            
end                          
;------------------------------------------------------------------------------                          
pro buff_cam, record_number, err                   
                          
;   Unpacks several CAMMICE Level Zero Records into a Data Record Header                           
;   and the various types of instrument data                                
                          
; Inputs:                          
;   record_number    Min record number to input, 1-9392                   
                   
; Outputs:                   
;   err              Error message string is null if no error                    
                          
common camLZ_buff,    $ ; Block holds many records of data                     
        unit,         $ ; Logical unit number of data file                     
        numbytes,     $ ; Number of bytes per record                     
        numbfill,     $ ; Number of fill bytes per record                      
        lastrec,      $ ; Number of last record in data file                   
        recrng,       $ ; Record number range, first record is one                       
        recstruc        ; Record structure array                         
                   
if n_elements( unit) eq 0L then unit = -1       
       
;   If file has been opened then check instrument name in file header record                   
if unit ne -1 then check_instr_lz, 'CAMMICE', err else $                  
    err = 'buff_cam: Input file is not open.'                   
                  
;   Read and process data if no errors                  
if err eq '' then begin                  
                  
;   Number of Level Zero records                          
    nrecs = n_elements( recstruc)                   
    recrng = [record_number, (record_number + nrecs - 1L)< lastrec]                          
    mrecs = recrng(1) - recrng(0) + 1L                   
                          
;   Read the data from the file                                    
    aa = assoc( unit, bytarr(numbytes, mrecs), numbytes*long(record_number))                                          
    data_bytes = aa( 0)                       
                                                           
;   Separate data into Data Record Header                               
    recstruc(0:mrecs-1).data_rec_hdr = Lz_drh( data_bytes(0:299, *))                               
                          
;   Check first data record header for validity                       
    cam_inst = 9L                   
    drh0 = recstruc(0).data_rec_hdr                     
    if (drh0.inst_num ne cam_inst) or (drh0.rec_num ne record_number+1L) then $                       
       err = 'buff_cam: invalid data record header'                       
                    
;   Separate out instrument data                          
    data = reform( data_bytes( 300:*, *), 10, 250, mrecs)                          
                          
;       Reformat the MICS Fixed Column data                          
    mics_fixed = data(4:6, *, *)                          
    mics_fixed = reform( mics_fixed, 15, 50, mrecs)                          
    tspose1, mics_fixed                          
    recstruc(0:mrecs-1).mics_fixed = mics_fixed                       
                             
;   Reformat the HIT Fixed Column data                               
    hit_fixed = data(7:9, *, *)                          
    hit_fixed = reform( hit_fixed, 15, 50, mrecs)                          
    tspose1, hit_fixed                          
    recstruc(0:mrecs-1).hit_fixed = hit_fixed                       
                          
;   Switch first two indices of data array so that dat( 250, 4, mrecs)                          
    dat = data( 0:3, *, *)                          
    tspose2, dat                          
                          
;   Separate housekeeping data                           
    recstruc(0:mrecs-1).hk = dat(*, 0:1, *)                       
                          
;   And subcom data                          
    recstruc(0:mrecs-1).subcom = dat(*, 2:3, *)                          
                          
endif                  
 
end                                
;------------------------------------------------------------------------------                          
pro tspose1, data                          
                          
;       Transposes first two indices of a third rank array                          
;       For example a( m1, m2, m3) --> b( m2, m1, m3)                          
;       Note that input array is changed in place                          
;       This routine is identical to the tspose2 procedure which can                   
;       be used to process arrays of different dimensions in parallel                   
                                 
;       If dimensions are unchanged then the index array is stored                           
;       in the common block and does not have to be computed again                          
common tp_index1,       $ ; BLOCK to store index                          
        nx1,            $ ; First dimension of input array                          
        ny1,            $ ; Second dimension of input array                          
        nz1,            $ ; Third dimension of input array                          
        index             ; Index for transposition                          
                          
;       Initialize common block if first execution                          
if n_elements( nx1) eq 0L then begin                          
                          
        nx1 = -1L                          
        ny1 = -1L                          
        nz1 = -1L                          
                          
endif                          
                          
;       Dimensions of input data arra                          
sz = size( data)                          
if sz(0) ne 3 then stop,' Error trans_pose: input array must be rank three.'                          
nx = sz(1)                          
ny = sz(2)                          
nz = sz(3)                          
                          
;       Have dimensions of input array changed?                          
if (nx ne nx1) or (ny ne ny1) or (nz ne nz1) then begin                          
                          
;       Compute new index array                          
        np = nx * ny                          
        c = transpose( reform( lindgen( np), nx, ny))                          
        d = lindgen( np * nz)                          
        index = ( d/np) * np + c(d mod np)                          
                          
;       Save dimensions in common for next execution                          
        nx1 = nx                          
        ny1 = ny                          
        nz1 = nz                          
                          
endif                          
                          
;       Switch data words to transpose array                          
data = reform( data( index), ny, nx, nz, /OVERWRITE)                          
                          
end                          
;------------------------------------------------------------------------------                          
pro tspose2, data                          
                          
;       Transposes first two indices of a third rank array                          
;       For example a( m1, m2, m3) --> b( m2, m1, m3)                          
;       Note that input array is changed in place                          
;       This routine is identical to the tspose1 procedure which can                   
;       be used to process arrays of different dimensions in parallel                   
                          
;       If dimensions are unchanged then the index array is stored                           
;       in the common block and does not have to be computed again                          
common tp_index2,       $ ; BLOCK to store index                          
        nx1,            $ ; First dimension of input array                          
        ny1,            $ ; Second dimension of input array                          
        nz1,            $ ; Third dimension of input array                          
        index             ; Index for transposition                          
                          
;       Initialize common block if first execution                          
if n_elements( nx1) eq 0L then begin                          
                          
        nx1 = -1L                          
        ny1 = -1L                          
        nz1 = -1L                          
                          
endif                          
                          
;       Dimensions of input data array                          
sz = size( data)                          
if sz(0) ne 3 then stop,' Error tspose2: input array must be rank three.'                          
nx = sz(1)                          
ny = sz(2)                          
nz = sz(3)                          
                          
;       Have dimensions of input array changed?                          
if (nx ne nx1) or (ny ne ny1) or (nz ne nz1) then begin                          
                          
;       Compute new index array                          
        np = nx * ny                          
        c = transpose( reform( lindgen( np), nx, ny))                          
        d = lindgen( np * nz)                          
        index = ( d/np) * np + c(d mod np)                          
                          
;       Save dimensions in common for next execution                          
        nx1 = nx                          
        ny1 = ny                          
        nz1 = nz                          
                          
endif                          
                          
;       Switch data words to transpose array                          
data = reform( data( index), ny, nx, nz, /OVERWRITE)                          
     
end                          
                         
