;------------------------------------------------------------------------------ 
; File: CHECK_LZ.PRO    Routines to Validate Level Zero Files                   
; Revision: 05/16/97    J L Roeder                   
;------------------------------------------------------------------------------ 
; Routines: 
;       length_lz       Length of records and fill in various LZ files  
;       sc_tm_mode      Spacecraft telemetry mode from LZ file header 
;       sc_name         Spacecraft name string from LZ file header             
;       instr_name      Instrument name from Level Zero file header            
;       check_lz        Validate Level Zero file by checking header bytes  
;       instr_lz        Get instrument name for calling program 
;       spacecraft_lz   Get spacecraft name for calling program 
;------------------------------------------------------------------------------ 
pro length_lz, spacecraft, instr, tm_mode, rec_length, fill_length 
             
;       Length of Level Zero file in bytes for a given spacecraft, 
;       instrument, and telemetry mode. Returns zero if unknown inputs.    
             
; Inputs:             
;       spacecraft      Must be the string 'POLAR'             
;       instr           Instrument name string             
;       tm_mode         Telemetery mode 3-char abreviation, if null then   
;                       return rec lengths for all three possible modes    
             
;       Number of words per minor frame for each mode and instrument 
nwords_mf = long([[21, 21, 42], $       ; PWI             
                [24, 2, 47], $          ; HYD             
                [6, 6, 10], $           ; MFE             
                [21, 21, 41], $         ; TIM             
                [21, 21, 42], $         ; TID             
                [58, 49, 1], $          ; UVI             
                [54, 55, 2], $          ; VIS             
                [24, 11, 2], $          ; PIX             
                [10, 10, 17], $         ; CAM             
                [18, 18, 33], $         ; CEP             
                [13, 13, 25], $         ; EFI             
                [9, 52, 9]])            ; SCR            
             
;       Compute record length as data length + data record header 
length_rec = nwords_mf *250L + 300L 
 
;       Minimum record size is 2792 bytes, fill out to this length 
length_fill = (2792L - length_rec) > 0L 
length_rec = length_rec + length_fill 
 
;       Fill all records out to 4-byte word boundary 
rem4 = length_rec mod 4L 
fill4 = (4L - rem4) * (rem4 ne 0L) 
length_rec = length_rec + fill4 
length_fill = length_fill + fill4 
 
;       Check if spacecraft is POLAR             
if spacecraft eq 'POLAR' then begin             
             
;       Instrument index by matching input against possiblities             
        instr_names = ['PWI','HYD','MFE','TIMAS','TIDE','UVI','VIS','PIXIE', $             
                'CAMMICE','CEPPAD','EFI','SCR','QAF']             
        jinstr = where(instr eq instr_names)             
        if jinstr(0) ne -1L then begin             
             
;           Check if TM mode was input             
            if tm_mode ne '' then begin             
             
;               Spacecraft TM mode index by matching input against possiblities             
                tm_modes = ['SCIENCE','MANEUVER','CONTINGENCY']             
                jmode = where(tm_mode eq tm_modes)             
             
;               Select record length             
                if jmode(0) ne -1L then begin             
 
                        rec_length = length_rec( jmode(0), jinstr(0))  
                        fill_length = length_fill( jmode(0), jinstr(0))  
 
                endif else begin             
 
                        rec_length = 0L 
                        fill_length = 0L 
 
                endelse 
             
;           No mode given so return all possible record lengths for instrument             
            endif else begin 
 
                rec_length = length_rec( *, jinstr(0)) 
                fill_length = length_fill( *, jinstr(0)) 
 
            endelse 
             
;       Unknown instrument             
        endif else begin 
 
                rec_length = 0L             
                fill_length = 0L 
 
        endelse 
             
endif else begin             
             
;       Unknown spacecraft             
        rec_length = 0L             
        fill_length = 0L             
             
endelse             
             
end             
;------------------------------------------------------------------------------              
function sc_tm_mode, data, spacecraft, instr             
             
;       Returns spacecraft telemetry mode from LZ file headers             
             
; Inputs:             
;       data    Bytes from beginning of file, must have 14848 bytes to             
;               process all possible POLAR file types             
;       instr   Instrument name             
;       tm_mode Spacecraft Telemetry Mode Name             
             
;       Get all possible record lengths for this instrument             
tmmode = ''             
length_lz, spacecraft, instr, tmmode, len_rec, len_fill 
             
;       Number of data bytes input             
ndata = n_elements( data)             
             
;       Possible instrument names and id numbers for POLAR             
instr_names = ['PWI','HYDRA','MFE','TIMAS','TIDE','UVI','VIS','PIXIE', $             
   'CAMMICE','CEPPAD','EFI','SCR','QAF']            
instr_ids = lindgen(13)+ 1L            
instr_ids(12) = 99L                ; Out of sequence id for quality flags             
            
;       Find instrument identification number from name string             
jinstr = where( instr eq instr_names)             
if jinstr(0) ne -1 then begin             
             
    instr_id = instr_ids( jinstr(0))             
             
;       Start with first record lenth             
    index = 0             
             
;       Loop until good length found or run out of possibilities             
    done = 0             
    while (done eq 0) and (index lt n_elements( len_rec)) do begin             
             
        len = len_rec( index)             
        if ndata lt (len+48) then $             
                stop,'Error sc_tm_mode: data array too short = ', ndata             
             
        drh1 = data( len:len+47)             
        drh1 = long( drh1, 0, 12)             
             
;       Swap bytes if necessary for this platform             
        swap_bytes, drh1             
             
;       Instrument identification number             
        id_instr = drh1(0)             
             
;       Physical record number should be two.             
        rec_numb = drh1(1)             
                     
;       Extract telemetry mode and end loop if id's check out             
        if (id_instr eq instr_id) and (rec_numb eq 2L) then begin             
                tm_mode_id = drh1(11)             
                done = 1             
        endif else index = index + 1 
             
    endwhile             
             
;       Validate Spacecraft TM mode id             
    if done eq 1 then begin             
        tm_mode_ids = long([1,3,4])             
        tm_modes = ['SCIENCE','MANEUVER','CONTINGENCY']             
        jmode = where(tm_mode_id eq tm_mode_ids)             
        if jmode(0) ne -1L then tm_mode = tm_modes( jmode(0)) else begin 
                print,'Error sc_tm_mode: unknown mode id = ',tm_mode_id 
                print,'Assume SCIENCE mode' 
                tm_mode = tm_modes( 0) 
        endelse 
             
    endif else stop,'Error sc_tm_mode: invalid file headers'             
             
endif else begin             
             
    tm_mode = 'UNKNOWN TM MODE'             
    print,'Warning sc_tm_mode: ', instr, '  ', tm_mode             
             
endelse             
   
return, tm_mode                                           
end             
;------------------------------------------------------------------------------              
function sc_name, data4            
             
;       Returns spacecraft name string extracted from LZ file header             
             
; Inputs:             
;       data4       At least the first four bytes of Level Zero file             
             
; Outputs:             
;       spacecraft  Name of spacecraft is either 'POLAR', 'GEOTAIL',             
;                   'WIND', or 'UNKNOWN SPACECRAFT'             
             
;       Possiblities             
sc_names = ['GEOTAIL','WIND','POLAR']             
sc_ids = lindgen(3)+24L            
              
;       Turn data into longwords             
long_data = long(data4, 0)            
             
;       Swap bytes if necessary for this platform             
swap_bytes, long_data             
             
;       Determine spacecraft by matching first longword in data            
;       to spacecraft identification numbers             
sc_id = long_data(0)             
jsc = where( sc_id eq sc_ids)             
if jsc(0) ne -1 then begin            
            
;     Fill string output for valid match            
      spacecraft = sc_names( jsc(0))            
            
endif else begin             
            
;     No valid match            
      spacecraft = 'UNKNOWN SPACECRAFT'             
      print,'Warning sc_name: unknown id number = ', sc_id            
            
endelse            
            
return, spacecraft                
end            
;------------------------------------------------------------------------------              
function instr_name, data8, spacecraft             
             
;       Returns instrument name from Level Zero file             
             
; Inputs:             
;       data8        At least the first eight bytes from Level Zero file             
             
; Outputs:             
;       spacecraft   Name of spacecraft must be POLAR             
;       instr        Instrument name from choices listed below             
             
;       Only POLAR data files have recognized instrument names             
if spacecraft eq 'POLAR' then begin             
             
;       Possible instrument names and id numbers for POLAR             
        instr_names = ['PWI','HYDRA','MFE','TIMAS','TIDE','UVI','VIS','PIXIE', $             
        'CAMMICE','CEPPAD','EFI','SCR','QAF']            
        instr_ids = lindgen(13)+ 1L            
        instr_ids(12) = 99L         ; Out of sequence id for quality flags             
            
;       Turn data into longwords             
        long_data = long(data8, 0, 2)             
             
;       Swap bytes if necessary for this platform             
        swap_bytes, long_data             
             
;       Determine instrument from second longword in data             
        instr_id = long_data(1)             
        jinstr = where( instr_id eq instr_ids)             
        if jinstr(0) ne -1 then begin            
            
;               Fill string output for valid match            
                instr = instr_names( jinstr(0))             
            
        endif else begin            
            
;               No valid match            
                instr = 'UNKNOWN INSTRUMENT'             
                print,'Warning instr_name: unknown id number = ', instr_id            
            
        endelse             
             
endif else begin             
             
        instr = 'UNKNOWN INSTRUMENT'             
        print,'Warning instr_name: Only POLAR instruments recognized'             
             
endelse             
             
return, instr            
end            
;------------------------------------------------------------------------------              
pro check_lz, err             
             
;       Validate POLAR Level Zero file by checking header bytes             
;       All info is put in common block LZfile             
           
; BLOCK: LZfile                   
;       pth         Path to LZ file including device and all directories 
;       fname       LZ filename including path info                   
;       unit        Unit number if file is open, else -1             
;       spacecraft  Name string should be one of 'POLAR'             
;       instrument  Name string should be one of ['CAMMICE','CEPPAD',           
;                   'MFE', or 'PIXIE']             
;       tm_mode     Must be one of ['SCIENCE','MANUEVER','CONTINGENCY']  
;       nbytes      Number of bytes per record including fill bytes 
;       nfill       Number of fill bytes per record             
             
common LZfile, pth, fname, unit, spacecraft, instrument, tm_mode, nbytes, nfill 
             
;       Check if file already open             
err = ''                   
if unit eq -1 then begin             
             
;       Has file been chosen?             
        if fname ne '' then begin             
             
;               Open Level Zero File             
                openr, unit, fname, /GET_LUN, /BLOCK             
             
        endif else err='File Not Selected'                   
             
endif             
             
if (unit ne -1) and (err eq '') then begin             
             
;       Allocate large record for header validation             
        aa = assoc( unit, bytarr( 14848L))             
             
;       Read first record             
        data = aa(0)             
             
;       Spacecraft name             
        spacecraft = sc_name( data)           
             
;       Instrument name for POLAR spacecraft            
        instrument = instr_name( data, spacecraft)           
           
;       Determine spacecraft telemetry mode             
        tm_mode = sc_tm_mode( data, spacecraft, instrument)             
             
;       Lookup Level Zero record length for this file type             
        length_lz, spacecraft, instrument, tm_mode, nbytes, nfill 
  
        print,'LZ file: '+fname+'  '+spacecraft+'  '+instrument+ '  '+tm_mode
           
endif           
           
end                                                                                   
;------------------------------------------------------------------------------ 
pro check_instr_lz, name, err 
 
;       Check instrument name in file header record 
 
; Inputs: 
;       name        Instrument name string, ie 'CAMMICE' 
 
; Outputs:
;       err         Error message string is null if header matches input
    
; BLOCK: LZfile                   
;       pth         Path to LZ file including device and all directories 
;       fname       LZ filename including path info                   
;       unit        Unit number if file is open, else -1             
;       spacecraft  Name string should be one of 'POLAR'             
;       instrument  Name string should be one of ['CAMMICE','CEPPAD',           
;                   'MFE', or 'PIXIE']             
;       tm_mode     Must be one of ['SCIENCE','MANUEVER','CONTINGENCY']  
;       nbytes      Number of bytes per record including fill bytes 
;       nfill       Number of fill bytes per record             
             
common LZfile, pth, fname, unit, spacecraft, instrument, tm_mode, nbytes, nfill 
             

if n_elements( instrument) eq 0L then instrument = ''
if name eq instrument then err = '' else $
   err = 'check_instr_lz: input data file is not '+name+' level zero'

end 
;------------------------------------------------------------------------------ 
pro spacecraft_lz, name 
 
;       Get spacecraft name for calling program 
 
; Inputs: None 
 
; Outputs:  
;       name        Spacecraft name string, ie 'POLAR' 
 
; BLOCK: LZfile                   
;       pth         Path to LZ file including device and all directories 
;       fname       LZ filename including path info                   
;       unit        Unit number if file is open, else -1             
;       spacecraft  Name string should be one of ['POLAR','WIND','GEOTAIL'] 
;       instrument  Name string should be one of ['CAMMICE','CEPPAD',           
;                   'MFE', or 'PIXIE']             
;       tm_mode     Must be one of ['SCIENCE','MANUEVER','CONTINGENCY']  
;       nbytes      Number of bytes per record including fill bytes 
;       nfill       Number of fill bytes per record             
             
common LZfile, pth, fname, unit, spacecraft, instrument, tm_mode, nbytes, nfill 
             
name = spacecraft 
 
end 
