;------------------------------------------------------------------------------
; File: GET_HIT.PRO    CAMMICE HIT Level Zero File Parsing Routines            
; Revision: 08/22/96   J L Roeder            
;------------------------------------------------------------------------------
; Routines:            
;      hcomb_de        Combine HIT direct events from normal and cross modes
;      get_hde         Get HIT direct event tof's and energies
;      get_hsrates     Get HIT sectored rate scaler counts   
;      get_hxrates     Get HIT X scaler counts 
;      get_hyrates     Get HIT Y scaler counts 
;      get_hmatrx      Get HIT matrix scalers from major frame     
;      get_htotrts     Get HIT non-sectored spin-averaged rates counts   
;      get_hmux	       Get all HIT multiplex data 
;      get_hsing       Get HIT selected-sector singles compressed counts 
;      get_hsun        Get HIT sun sector and other status info
;      get_hdpustat    Get HIT DPU status bits
;      get_hdpuhk      Get HIT DPU housekeeping analogs
;------------------------------------------------------------------------------
pro hcomb_de, tmstat, deng, eng, eprime, sample, m
   
;       Get HIT direct events combining Normal and Cross Telemetry into    
;       single arrays   
   
; Inputs:   
;       tmstat       	HIT DPU Telemetry Status, 0-15
   
; Outputs: All singly dimensioned 
;       deng		Energy Loss in channel number, 0-1023   
;       eng		Energy in channel number, 01023
;	eprime		Energy from B or C detector
;	sample		Time slice identifier, 0-15
;       m		Mass group number if in cross mode, 0-15   
   
;       Get Direct Events in normal telemetry   
get_hfde_LZ, deng1, eng1, eprime1, sample1   
   
;       Number of events returned   
nev = n_elements( deng1)   
   
;       Check for good data   
if (nev gt 1L) or (deng1(0) ne -1) then begin   
   
;       In cross-mode telemetry?   
        if (tmstat and '08'XB) ne 0B then begin   
   
;               Get Direct Events in cross-mode telemetry    
                get_hcde_LZ, deng2, eng2, eprime2, sample2, m2
   
;               Combine Normal and cross mode direct events into    
;               a single array   
;               First the energy loss
                deng = intarr( nev, 4)   
                deng(0, 0) = deng1   
                deng(0, 1) = deng2   
                deng = reform( transpose( deng), nev * 4)   
   
;               The energies    
                eng = intarr( nev, 4)   
                eng(0, 0) = eng1   
                eng(0, 1) = eng2   
                eng = reform( transpose( eng), nev * 4)   
   
;               The eprimes
                eprime = bytarr( nev, 4)   
                eprime(0, 0) = eprime1   
                eprime(0, 1) = eprime2   
                eprime = reform( transpose( eprime), nev * 4)   
   
;               The samples
                sample = bytarr( nev, 4)   
                sample(0, 0) = sample1   
                sample(0, 1) = sample2   
                sample = reform( transpose( sample), nev * 4)   
   
;               The masses   
                m = replicate( 255B, nev, 4)   
                m(0, 1) = m2   
                m = reform( transpose( m), nev * 4)   
   
        endif else begin   
   
;               Use only normal mode telemetry   
                deng = deng1   
                eng = eng1   
                eprime = eprime1   
                sample = sample1   
                m = replicate( 255B, n_elements( deng))
                   
        endelse   
   
endif else begin   
   
;               No events found   
                deng = -1   
                eng = -1   
                eprime = -1B   
                sample = -1B   
                m = -1   
   
endelse   
   
end   
;-------------------------------------------------------------------------            
pro get_hde, times, eng, deng, eprime, sample, m, err   
   
;       Get HIT direct events for listing or plot for specified number of   
;       LZ records        
   
; Inputs: None   
   
; Outputs: All dimensioned 250 by number of records   
;	times		Universal times in msec
;       deng		Energy Loss in channel number, 0-1023   
;       eng		Energy in channel number, 0-1023   
;	eprime		Energy from B or C detector, 0-255?
;	sample		Time slice identifier, 0-15
;       m		Mass group number, 0-15   
;       err		Error string, null for no error   
   
common interval, times_rng, rec_rng   
   
;       Number of possible direct events if in CROSS mode   
nrecs = rec_rng(1) - rec_rng(0) + 1   
ndevnts = 200L * nrecs   
   
;       Setup output arrays   
times = lonarr( ndevnts)
eng = intarr( ndevnts)   
deng = intarr( ndevnts)
eprime = bytarr( ndevnts)
sample = bytarr( ndevnts)
m = bytarr( ndevnts)   
   
;       Setup working arrays for sectors 
nsectors = nrecs * 50L   
sectors = bytarr( nsectors)   
   
;       Initialize number of events and sectors stored   
idevnt = 0L   
isector = 0L   
   
;       Process one major frame at a time   
for irec = rec_rng(0), rec_rng(1) do begin   
   
;       Unpack a single record   
        unpack_cam, irec, err   
   
;       Test for read error   
        if err ne '' then begin   
   
;		Output flagged data
		times = -1L
                deng = -1   
                eng = -1   
                eprime = 255B 
                sample = 255B
                m = 255B
                return   
   
        endif   
   
;       Get DPU status structure   
        get_htmstat_LZ, tmstat   
   
;	Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin

;          Get HIT combined direct events   
           hcomb_de, tmstat, deng1, eng1, eprime1, sample1, m1

;          Check if any events found   
           if (n_elements( deng1) gt 1L) or (deng1(0) ne -1) then begin   
   
;               Get sector numbers   
                get_hsect_LZ, times1, sector1   
   
;               Load HIT sector numbers retaining only 4 msb's
                sectors( isector) = sector1 / 2B
   
;               Increment total number of sectors
		nsector1 = n_elements( sector1)
                isector = isector + nsector1
   
;               Number of events should be 50 in normal mode 
;		or 200 in cross mode
                nde = n_elements( deng1)   
   
;               Number of events per sector
                mde = nde / nsector1

;               Replicate times to get one for each event
                timde1 = times1( lindgen(nde) / mde)

;               Load output array with events from this record   
                times( idevnt) = timde1
                eprime( idevnt) = eprime1   
                sample( idevnt) = sample1 
                deng( idevnt) = deng1   
                eng( idevnt) = eng1   
                m( idevnt) = m1   
   
;               Increment total number of events   
                idevnt = idevnt + nde   
   
           endif   	; valid event block

	endif		; tm status block
   
endfor   		; record block
   
;       Reform output arrays to reduce memory use   
if idevnt gt 0 then begin 

	deng = deng(0: idevnt-1)   
	eng = eng(0: idevnt-1)   
	eprime = eprime(0: idevnt-1)   
	sample = sample(0: idevnt-1)   
	m = m(0: idevnt-1)   
	times = times(0: idevnt-1)

endif else begin

       if err eq '' then err = 'get_hde: No valid data was read.'
	deng = -1
	eng = -1
	eprime = 255B
	sample = 255B
	m = 255B
	times = -1L

endelse
   
end   
;-------------------------------------------------------------------------
pro get_hsrates, times, mf, sectors, chans, cnts, err
   
;       Get HIT sectored rate scaler counts for listing or plot   
;       from a specified number of LZ records        
   
; Inputs: None   
   
; Outputs:   
;	times	Universal times in long msec
;	mf	Minor frame number, 0-249, dim 50 * nrec  
;	sectors	Sector number, 0-31, dim 50 * nrec  
;	chans	Channel numbers, 0-11 are HR0-11 and 12-17 are HID0,2, 5-8
;	cnts	Rate scaler compressed counts
;       err	Error string, null for no error   
   
; BLOCK: interval  
;      time_rng    Time min and max in seconds from start of day  
;      rec_rng     Record number min and max  
  
common interval, times_rng, rec_rng   
   
; BLOCK: hrates    
;       possible_chan   Strings identifying channels for menu    
;       index           Index of which channels selected for listing    
    
common hrates, possible_chan, index    
    
;      Initialize number of events and sectors stored   
isector = 0L   

;	Initialize channel index common block first time through
if n_elements(index) eq 0 then hrates_init
   
;      Check if any channnels were selected  
if index(0) ne -1 then begin  
  
;      Number of possible sectors is half that of MICS
       nrecs = rec_rng(1) - rec_rng(0) + 1   
       nsectors = 25L * nrecs   
  
;      Number of channels to extract  
       nchan = n_elements( index)  
      
;      Output channel numbers
       chans = index

;      Setup output arrays   
       mf = bytarr( nsectors)   
       times = lonarr( nsectors)   
       sectors = bytarr( nsectors)   
       cnts = intarr( nchan, nsectors)   
   
;      Process one major frame at a time   
       for irec = rec_rng(0), rec_rng(1) do begin   
   
;	   Unpack a single record   
           unpack_cam, irec, err   
   
;          Test for read error   
           if err ne '' then begin   
   
;		Output only flagged data
		times = -1L
		mf = -1B  
		cnts = -1  
		sectors = -1B  
		return   
   
           endif   
   
;	   Get DPU status structure
	   get_htmstat_LZ, tmstat   

;	   Check for DPU in sensor data mode
           if (tmstat gt 3B) and (tmstat le 15B) then begin

;              Get sector numbers   
               get_hsect_LZ, times1, sector1   

;	       Present number of sectors
	       nsect1 = n_elements( sector1)
	       if (nsect1 eq 1L) and (sector1(0) eq 255B) then nsect1 = 0L

;	       Any good sectors?
	       if nsect1 ne 0L then begin

;		    Get HIT sectored rate scaler counts for this record   
		    get_hsrts_LZ, index, cnts1   
   
;          	    Select only even sectors and times from full set
;		    divide by two so that sector numbers range 0-15
           	    nsector1 = n_elements( sector1) / 2L
		    sector_even = sector1( lindgen( nsector1) * 2L) / 2B
		    times_even = times1( lindgen( nsector1) * 2L)
   
;          	    Load data into full array         
           	    cnts(0, isector) = cnts1   
           	    times( isector) = times_even
           	    sectors( isector) = sector_even

;      	   	    Construct minor frame count array  
           	    mf1 = 10 * indgen( nsector1) 

;	   	    Load minor frame numbers into output array
	   	    mf( isector) = mf1
      
;          	    Increment total number of sectors   
           	    isector = isector + n_elements( sector_even)

		endif	; Valid sector block

	   endif	; TM Status block
   
       endfor   	; Record block
   
endif else err = 'get_hsrates: no channels selected.'

;	Check if any sectors read
nsectors = isector
if nsectors gt 0L then begin

	times = times( 0:nsectors-1)
	mf = mf( 0:nsectors-1)
	sectors = sectors( 0:nsectors-1)
	cnts = cnts( *, 0:nsectors-1)

endif else begin  
  
       if err eq '' then err = 'get_hsrates: No data was read.'
       mf = -1B  
       cnts = -1  
       sectors = -1B
       times = -1L
  
endelse  

end   
;-------------------------------------------------------------------------            
pro get_hxrates, times, sectors, stype, snum, cnts, err
   
;       Get HIT X scaler counts for listing or plot   
;       from a specified number of LZ records        
   
; Inputs: None   
   
; Outputs:   
;	times	Universal times in long msec
;	sectors	Sector number, 0-31, dim 50 * nrec
;	stype	Scaler type, HR=0, HS=1, HID=2
;	snum	Scaler number, HR=0-15, HS=0-7, HID=0-10,12
;	cnts	Rate scaler compressed counts
;       err	Error string, null for no error   
   
; BLOCK: interval  
;      time_rng    Time min and max in seconds from start of day  
;      rec_rng     Record number min and max  
  
common interval, times_rng, rec_rng   
   
;      Initialize number of events and sectors stored   
isector = 0L   
   
;      Number of possible sectors is half that of MICS
nrecs = rec_rng(1) - rec_rng(0) + 1   
nsectors = 25L * nrecs   
  
;      Setup output arrays   
times = lonarr( nsectors)   
sectors = bytarr( nsectors)   
cnts = bytarr( nsectors)   
stype = bytarr( nsectors)   
snum = bytarr( nsectors)   
   
;      Process one major frame at a time   
for irec = rec_rng(0), rec_rng(1) do begin   
   
;	Unpack a single record   
	unpack_cam, irec, err   
   
;       Test for read error   
        if err ne '' then begin   
   
;		Output only flagged data
		times = -1L
		stype = 255B  
		snum = 255B  
		cnts = 255B
		sectors = 255B  
		return   
   
	endif   
   
;	Get DPU status structure
	get_htmstat_LZ, tmstat   

;	Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin

;              Get sector numbers   
               get_hsect_LZ, times1, sector1   

;	       Present number of sectors
	       nsect1 = n_elements( sector1)
	       if (nsect1 eq 1L) and (sector1(0) eq 255B) then nsect1 = 0L

;	       Any good sectors?
	       if nsect1 ne 0L then begin

;		    Get HIT sectored rate scaler counts for this record   
		    get_hxrts_LZ, cnts1   
   
;		    Get HIT scaler status for this record   
		    get_hxystat_LZ, xstype1, xsnum1, ystype1, ysnum1
   
;          	    Select only even sectors and times from full set
;		    divide by two so that sector numbers range 0-15
           	    nsector1 = n_elements( sector1) / 2L
		    sector_even = sector1( lindgen( nsector1) * 2L) / 2B
		    times_even = times1( lindgen( nsector1) * 2L)
   
;		    Manufactor status for each sector
		    s_type1 = replicate( xstype1, nsector1)
		    s_num1 = replicate( xsnum1, nsector1)

;          	    Load data into full array         
           	    cnts( isector) = cnts1   
           	    times( isector) = times_even
           	    sectors( isector) = sector_even
           	    stype( isector) = s_type1
           	    snum( isector) = s_num1
   
;          	    Increment total number of sectors   
           	    isector = isector + nsector1

		endif	; Valid sector block

	endif	; TM Status block
   
endfor   	; Record block
   

;	Check if any sectors read
nsectors = isector
if nsectors gt 0L then begin

	times = times( 0:nsectors-1)
	stype = stype( 0:nsectors-1)
	snum = snum( 0:nsectors-1)
	sectors = sectors( 0:nsectors-1)
	cnts = cnts( 0:nsectors-1)

endif else begin  
  
       if err eq '' then err = 'get_hxrates: No valid data was read.'
       stype = 255B  
       snum = 255B  
       cnts = 255B
       sectors = 255B
       times = -1L
  
endelse  

end   
;-------------------------------------------------------------------------            
pro get_hyrates, times, sectors, stype, snum, cnts, err
   
;       Get HIT Y scaler counts for listing or plot   
;       from a specified number of LZ records        
   
; Inputs: None   
   
; Outputs:   
;	times	Universal times in long msec
;	sectors	Sector number, 0-31, dim 50 * nrec
;	stype	Scaler type, HR=0, HS=1, HID=2
;	snum	Scaler number, HR=0-15, HS=0-7, HID=0-10,12
;	cnts	Rate scaler compressed counts
;       err	Error string, null for no error   
   
; BLOCK: interval  
;      time_rng    Time min and max in seconds from start of day  
;      rec_rng     Record number min and max  
  
common interval, times_rng, rec_rng   
   
;      Initialize number of events and sectors stored   
isector = 0L   
   
;      Number of possible sectors is half that of MICS
nrecs = rec_rng(1) - rec_rng(0) + 1   
nsectors = 50L * nrecs   
  
;      Setup output arrays   
times = lonarr( nsectors)   
sectors = bytarr( nsectors)   
cnts = bytarr( nsectors)   
stype = bytarr( nsectors)   
snum = bytarr( nsectors)   
   
;      Process one major frame at a time   
for irec = rec_rng(0), rec_rng(1) do begin   
   
;	Unpack a single record   
	unpack_cam, irec, err   
   
;       Test for read error   
        if err ne '' then begin   
   
;		Output only flagged data
		times = -1L
		stype = 255B  
		snum = 255B  
		cnts = 255B
		sectors = 255B  
		return   
   
	endif   
   
;	Get DPU status structure
	get_htmstat_LZ, tmstat 

;	Is the HIT DPU in cross mode?
	hcross = tmstat and '08'XB

;	Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) and (hcross ne 0B) then begin

;              Get sector numbers   
               get_hsect_LZ, times1, sector1   

;	       Present number of sectors
	       nsect1 = n_elements( sector1)
	       if (nsect1 eq 1L) and (sector1(0) eq 255B) then nsect1 = 0L

;	       Any good sectors?
	       if nsect1 ne 0L then begin

;		    Get HIT sectored rate scaler counts for this record   
		    get_hyrts_LZ, cnts1   
   
;		    Get HIT scaler status for this record   
		    get_hxystat_LZ, xstype1, xsnum1, ystype1, ysnum1
   
;          	    Select full set of sectors and times 
           	    nsector1 = n_elements( sector1)
   
;		    Manufactor status for each sector
		    s_type1 = replicate( ystype1, nsector1)
		    s_num1 = replicate( ysnum1, nsector1)

;          	    Load data into full array         
           	    cnts( isector) = cnts1   
           	    times( isector) = times1
           	    sectors( isector) = sector1
           	    stype( isector) = s_type1
           	    snum( isector) = s_num1
   
;          	    Increment total number of sectors   
           	    isector = isector + nsector1

		endif	; Valid sector block

	endif	; TM Status block
   
endfor   	; Record block
   

;	Check if any sectors read
nsectors = isector
if nsectors gt 0L then begin

	times = times( 0:nsectors-1)
	stype = stype( 0:nsectors-1)
	snum = snum( 0:nsectors-1)
	sectors = sectors( 0:nsectors-1)
	cnts = cnts( 0:nsectors-1)

endif else begin  
  
       if err eq '' then err = 'get_hyrates: No valid data was read.'
       stype = 255B  
       snum = 255B  
       cnts = 255B
       sectors = 255B
       times = -1L
  
endelse  

end   
;-------------------------------------------------------------------------            
pro get_hmatrx, times, cnts, err
   
;       Get HIT compressed matrix scaler counts for listing   
;       or plot from a specified number of LZ records        
   
; Inputs: None   
   
; Outputs:   
;	times		Universal times in long msec
;	cnts		Matrix compressed counts
;       err		Error string, null for no error   
   
; BLOCK: interval  
;      time_rng    Time min and max in seconds from start of day  
;      rec_rng     Record number min and max  
  
common interval, times_rng, rec_rng   
  
;  Number of possible experiment master frames  
nrecs = rec_rng(1) - rec_rng(0) + 1   
nmaster =  nrecs / 11 + 2   
 
;  Always same number of matrix scalers in a master frame 
nmatrix = 512L 
     
;  Setup output arrays   
times = replicate( -1L, nmaster)
cnts = bytarr( nmatrix, nmaster)   
   
;  Initialize number of master frames stored   
imaster = 0L

;  Initialize good data found flag to false
data_found = 0   
   
;  Initialize value of mfc for previous record processed 
last_mfc = 255B 
 
;  Process one major frame at a time   
for irec = rec_rng(0), rec_rng(1) do begin   
   
;      Unpack a single record   
       unpack_cam, irec, err   
   
;      Test for read error   
       if err ne '' then begin   
   
;	   Output flagged data
           cnts = -1B
	   times = -1L
           return   
   
       endif   
   
;	Get DPU status structure
	get_htmstat_LZ, tmstat   

;	Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin

;	    Get HIT matrix scaler counts from one record   
            get_hmatrx_LZ, mfc, cnts1 

;	    Check if good data was returned
            if mfc ne 255B then begin

;               Index into matrix array is found from the Master Frame Counter 
	        indx = 48 * mfc

;               Increment master frames based on last mfc and mfc to improve 
;               noise immunity 
                if (last_mfc eq 10) and (mfc eq 0) then imaster = imaster + 1L 
 
;		Set good data found flag
		data_found = 1

;               Load data into full array         
                cnts( indx, imaster) = cnts1   
   
;               Obtain UT of start of master frame
                get_htimes, mfrm, tmaster, tfirst, tsector

;               Accumulation over 16 spins beginning at start of master frame
;	        at 16 sectors per spin
                times(imaster) = tmaster + 128L * tsector / 1000L

;               Keep of last mfc 
                last_mfc = mfc  
 
	     endif else last_mfc = 255B		; Master frame counter block

	endif else last_mfc = 255B		; tm status block

endfor   					; record block

;	Check if any data accumulated
if data_found ne 0 then begin

;	Delete any unused master frame storage
	nmaster = imaster+1L
	times = times(0:nmaster-1L)
	cnts = cnts( *, 0:nmaster-1L)

;	Format output array for mass and mass/charge groups   
	nm = 16
	nmoq = 32
	cnts = reform( cnts, nm, nmoq, nmaster)

endif else begin

;	Output flagged data with error
	err = 'get_hmatrx: No valid data read.'
	times = -1L
	cnts = 255B

endelse

end   
;-------------------------------------------------------------------------            
pro get_htotrts, times, chans, counts, err
  
;  Get HIT spin-averaged rate scaler counts from a range of records           
  
; Inputs: None    
    
; Outputs:    
;       times           Universal times in long msec 
;       chans           Channel numbers, 0-15 are spin-averaged rates only,
;			0-3 = HR12-15, 
;			4-9 = HID1,3,4,9,10,12, 
;			and 10-15 = HS1-HS6
;       cnts            Spin-averaged compressed rate scaler counts  
;       err     	Error string, null for no error    
    
; BLOCK: interval   
;       times_rng       Time min and max in seconds from start of day   
;       rec_rng         Record number min and max   
   
common interval, times_rng, rec_rng    
    
; BLOCK: hsrates     
;       possible_chan   Strings identifying channels for menu     
;       index           Index of which channels selected for listing     
     
common hsrates, possible_chan, index     
     
if n_elements( index) eq 0L then hsrates_init

;   Output channels selected
chans = index       

;      Check if any channnels were selected   
if index(0) ne -1 then begin   
   
;   Number of channels to extract   
    nchans = n_elements( index)   

;   Number of possible sectors, spins, and points 
    nrecs = rec_rng(1) - rec_rng(0) + 1    
    nsectors = 50L * nrecs 
    nspins = nsectors / 32L + 2L 
    npoints = nchans * nspins   
   
;   Setup output and work arrays    
    tims = lonarr( npoints)
    chs = bytarr( npoints)
    cnts = bytarr( npoints)

;   Initialize number of points stored    
    ipoint = 0L    
	    
;   Process one major frame at a time    
    for irec = rec_rng(0), rec_rng(1) do begin    
    
;	Unpack a single record    
	unpack_cam, irec, err    
    
;	Test for read error    
	if err ne '' then begin    
    
;	Output one empty set of counts
	    times = -1L
	    counts = -1   
	    return    
    
	endif    
    
;	Get DPU status structure    
	get_htmstat_LZ, tmstat    

;	Check for DPU in sensor data mode
	if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;	    Get sector numbers    
	    get_hsect_LZ, ts1, sector1    

;	    Get HIT sectored rate scaler counts for this record    
	    get_htotrts_LZ, index, sector1, tims1, chs1, cnts1
    
;   	    Check if good data
	    if tims1(0) ne -1L then begin

;		Load data into full array          
    		cnts( ipoint) = cnts1    
    		chs( ipoint) = chs1 
    		tims( ipoint) = tims1
    
;		Increment total number of sectors and points   
    		ipoint = ipoint + n_elements( cnts1) 

    	    endif	; good data block

    	endif       	; tm status block
    
    endfor 		; record loop
       
endif else err = 'get_htotrts: no channels selected.'

;       Check if any data received
if ipoint gt 0L then begin

;   Trim arrays to size of data received 
    cnts = cnts( 0:ipoint-1) 
    chs = chs( 0:ipoint-1) 
    tims = tims( 0:ipoint-1) 

;   Channel identifiers for decommutating
    chanindex = intarr( n_elements( possible_chan))
    chanindex( index) = indgen( nchans)
    chan_id = chanindex( chs)
 
;   Reformat counts and times to align channels in 2-d array
    blkwid = nchans
    build_index, chan_id, blkwid, indx

    decomm_data, blkwid, indx, tims, times
    decomm_data, blkwid, indx, cnts, counts

;   Reformat time array to give one value per spin
    times = transpose( times)
    reform_data, -1L, times

endif else begin   
   
    err = 'get_htotrts: no valid data was read.'
    times = -1L
    counts = -1   
   
endelse   
 
end    
;-------------------------------------------------------------------------            
pro get_hmux, tmstr, tsect, mframe, mspin, sector, mux, err
  
;  Get HIT multiplex data from a range of records in a level zero file
  
; Inputs: None    
    
; Outputs:  all arrays approximately dim (50 * nrec)
;	tmstr		Universal time of master frame accumulation start
;	tsect		Sector period in MICROSECONDS
;	mframe		Master frame counter
;	mspin		Spin counter, 0-17
;	sector		Sector number, 0-32
;	mux		Multiplex data
    
; BLOCK: interval   
;       times_rng       Time min and max in seconds from start of day   
;       rec_rng         Record number min and max   
   
common interval, times_rng, rec_rng    
   
;      Number of possible sectors, spins, and points 
nrecs = rec_rng(1) - rec_rng(0) + 1    
nsectors = 50L * nrecs 
   
;      Setup output and work arrays    
tmstr = replicate( -1L, nsectors)
tsect = replicate( -1L, nsectors)
mframe = replicate( -1, nsectors)
mspin = replicate( 255B, nsectors)
sector = replicate( 255B, nsectors)
mux = replicate( 255B, nsectors)
valid = replicate( 255B, nsectors)

;	Number of records and sectors actually stored
isector = 0L

;	Experiment master frame counter
frame_cnt = 0
    
;      Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    

;		Output one empty sector of data
		sector = 255B
		mframe = -1
		mspin = 255B
	        tmstr = -1L
		tsect = -1L
		mux = 255B
		return    
    
        endif    
    
;       Get DPU status structure    
        get_htmstat_LZ, tmstat    

;       Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;           Get sector numbers    
            get_hsect_LZ, times1, sector1    

;           Get sector numbers    
            get_hvalid_LZ, valid1

;           Get all HIT mux data for this record    
            get_hmux_LZ, mux1
    
;           Obtain master frame times and sector interval 
	    get_htimes, mfrm1, tmaster1, tfirst1, tsect1

;	    Start new master frames if necessary
	    if mfrm1 eq 0B then frame_cnt = frame_cnt + 1

;	    Number of sectors received
	    nsect1 = n_elements( sector1) 
    
;           Load data into full array
            mux( isector) = mux1
            valid( isector) = valid1
            sector( isector) = sector1
	    mframe( isector) = replicate( frame_cnt, nsect1)
	    mspin( isector) = byte((indgen( nsect1) + mfrm1*50) / 32)
	    tmstr( isector) = replicate( tmaster1, nsect1)
	    tsect( isector) = replicate( tsect1, nsect1)
    
;           Increment total number of sectors and points   
            isector = isector + nsect1

        endif           ; tm status block
    
endfor 			; record loop

;       Check if any data read
if isector gt 0L then begin
 
;	Trim arrays to size of data received 
	mux = mux( 0:isector-1) 
	sector = sector( 0:isector-1) 
	valid = valid( 0:isector-1) 
	tmstr = tmstr( 0:isector-1) 
	tsect = tsect( 0:isector-1) 
	mspin = mspin( 0:isector-1) 
	mframe = mframe( 0:isector-1) 
 
;	Output only valid data
	jvalid = where( valid eq 1B, nvalid)
	if nvalid ne 0L then begin

		mux = mux( jvalid)
		sector = sector( jvalid)
		tmstr = tmstr( jvalid)
		tsect = tsect( jvalid)
		mspin = mspin( jvalid)
		mframe = mframe( jvalid)

	endif else begin

;		Output one empty sector of data
		err = 'get_hmux: No valid data was read.'
		sector = 255B
		mframe = -1
		mspin = 255B
		tmstr = -1L
		tsect = -1L
		mux = 255B

	endelse

endif else begin

;	Output one empty sector of data
	err = 'get_hmux: No valid data was read.'
	sector = 255B
	mframe = -1
	mspin = 255B
	tmstr = -1L
	tsect = -1L
	mux = 255B

endelse
 
end    
;------------------------------------------------------------------------------ 
pro get_hsing, timesa, sectora, countsa, timesb, sectorb, countsb, err
  
;  Get HIT spin-averaged rate scaler counts from a range of records           
  
; Inputs: None    
    
; Outputs:    
;       timesa		Universal times of perpendicular sectors in long msec 
;	sectora		Sector number of perp singles counts
;       countsa		Compressed singles counts for sectora
;       timesb		Universal times of parallel sectors in long msec 
;	sectorb		Sector number of parallel singles counts
;       countsb		Compressed singles counts for sectorb
;       err     	Error string, null for no error    
    
; BLOCK: interval   
;       times_rng       Time min and max in seconds from start of day   
;       rec_rng         Record number min and max   
   
common interval, times_rng, rec_rng    
    
;   Number of possible sectors, spins, and points 
nrecs = rec_rng(1) - rec_rng(0) + 1    
nsectors = 50L * nrecs 
nspins = nsectors / 32L + 2L 
npoints = nspins  * 10L
   
;   Setup output and work arrays    
tspns = lonarr( npoints)
tsects = lonarr( npoints)
chs = bytarr( npoints)
dat = bytarr( npoints)

;   Initialize number of points stored    
ipoint = 0L    
	    
;   Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;	Unpack a single record    
	unpack_cam, irec, err    
    
;	Test for read error    
	if err ne '' then begin    
    
;	Output one empty set of counts
	    timesa = -1L
	    sectora = 255B
	    countsa = 255B  
	    timesb = -1L
	    sectorb = 255B
	    countsb = 255B  
	    return    
    
	endif    
    
;	Get DPU status structure    
	get_htmstat_LZ, tmstat    

;	Check for DPU in sensor data mode
	if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;	    Get sector numbers    
	    get_hsect_LZ, ts1, sector1    

;	    Get HIT sectored rate scaler counts for this record    
	    get_hsing_LZ, sector1, tspns1, tsects1, chs1, dat1
    
;   	    Check if good data
	    if tspns1(0) ne -1L then begin

;		Load data into full array          
    		tspns( ipoint) = tspns1
    		tsects( ipoint) = tsects1
    		chs( ipoint) = chs1 
    		dat( ipoint) = dat1
    
;		Increment total number of sectors and points   
    		ipoint = ipoint + n_elements( dat1) 

    	    endif	; good data block

    	endif       	; tm status block
    
endfor 			; record loop

;       Check if any data received
if ipoint gt 0L then begin

;   Trim arrays to size of data received 
    tspns = tspns( 0:ipoint-1) 
    tsects = tsects( 0:ipoint-1) 
    chs = chs( 0:ipoint-1) 
    dat = dat( 0:ipoint-1) 

;   Reformat counts and times to align channels in 2-d array
    blkwid = 10
    build_index, chs, blkwid, indx

    decomm_data, blkwid, indx, tspns, tspins
    decomm_data, blkwid, indx, tsects, tsectors
    decomm_data, blkwid, indx, dat, data

;   Reformat spin times and sector period array to give 1 value/spin
    tspins = transpose( tspins)
    reform_data, -1L, tspins

    tsectors = transpose( tsectors)
    reform_data, -1L, tsectors

;   Separate counts from sectors
    data = transpose( data)
    sa = data(*, 2)
    sectora = (sa ne 255B) * ((sa and 31B) / 2B) + 255B * (sa eq 255B)
    countsa = transpose( data( *, [0, 3, 6, 8]))

    sb = data(*, 5)
    sectorb = (sb ne 255B) * ((sb and 31B) / 2) + 255B * (sb eq 255B)
    countsb = transpose( data( *, [1, 4, 7, 9]))

;   Calculate times for centers of sectors a and b
    ta = tspins + long( tsectors * (sectora + 0.5)) / 1000L
    timesa = ta * ((tspins ne -1L) and (sectora ne 255B)) - $
	((tspins eq -1L) or (sectora eq 255B))

    tb = tspins + long( tsectors * (sectorb + 0.5)) / 1000L
    timesb = tb * ((tspins ne -1L) and (sectorb ne 255B)) - $
	((tspins eq -1L) or (sectorb eq 255B))

endif else begin   
   
    timesa = -1L
    sectora = 255B
    countsa = 255B
    timesb = -1L
    sectorb = 255B
    countsb = 255B
    err = 'get_hsing: no valid data was read.'
   
endelse   
 
end    
;-------------------------------------------------------------------------            
pro get_hsun, times, hsun_stat, err
  
;  Get HIT sun sector and other information from a range of records           
  
; Inputs: None    
    
; Outputs:
;	times		Times of starts of spins
;	hsun_stat	HIT Sun Status Structure defined in HIT_SUN.PRO
;       err     	Error string, null for no error    
    
;	Obtain all multiplex data
get_hmux, tmaster, tsector, mframe, mspin, sector, mux, err       

;       Test for read error    
if err ne '' then begin    

;	Output one empty set of singles    
	def_hsunstat_strc, hsun_stat, nbytes
	times = -1L
	return    
    
endif    
    
;	Number of sectors
nsectors = n_elements( sector)

;      Provide all status bytes
nchans = 2

;       Locations of singles rates in MUX array
sect_index = [0, 24]

;       Setup channel id arrays
chan = replicate( 255B, nsectors)

;       Mark data one channel at a time
for ichan = 0, nchans - 1 do begin
        jindex = where( sector eq sect_index( ichan))
        if jindex(0) ne -1 then chan( jindex) = ichan
endfor

;       Which data were marked?
chans = where( chan ne 255B, nc)

;       Check if any data available
if nc ne 0L then begin

;       Extract the Sector-A data which were marked
        channels = chan( chans)
        stats = mux( chans)
	m_spins = mspin( chans)
	t_mastr = tmaster( chans)
	t_sector = tsector( chans)

;	Compute universal times of selected sectors
	timspins = t_mastr + $
		long( (16L * m_spins) * t_sector ) / 1000L

;	Decommutate count arrays to align counts as 2 channels x nspins
	blkwid = 2
	build_index, channels, blkwid, indx
	decomm_data, blkwid, indx, stats, status
	decomm_data, blkwid, indx, timspins, times

;	Pick out individual status words from array
	status = transpose( status)

;	Put status bits into output structure
	load_hsunstat, status, hsun_stat

;	Reformat data to give one value per spin
	times = transpose( times)
	reform_data, -1L, times

endif else begin

        err = 'get_hsun: no valid status data was read.'
	def_hsunstat_strc, hsun_stat, nbytes
	times = -1L

endelse

 
end    
;-----------------------------------------------------------------------------
pro get_hdpustat, times, hdpu_stat, err

;       Get HIT DPU Status for listing or plot
;       from a specified number of LZ records

; Inputs: None

; Outputs:
;	times		Universal times in msec
;       hdpu_stat	HDPU Status Structure
;       err		Error string, null for no error   

; BLOCK: interval
;      times_rng    Time min and max in seconds from start of day
;      rec_rng     Record number min and max

common interval, times_rng, rec_rng

;       Number of possible sectors
nrecs = rec_rng(1) - rec_rng(0) + 1

;       Initialize output record count
jrec = 0L

;       Setup arrays to hold raw data
times = lonarr( nrecs)
word11 = intarr(17, nrecs)
word15 = intarr(11, nrecs)

;       Process one major frame at a time
for irec = rec_rng(0), rec_rng(1) do begin

;       Unpack a single record
        unpack_cam, irec, err

;       Test for read error
        if err ne '' then begin

;		If error then output one empty structure
                times = -1L
                def_hdpustat_strc, hdpu_stat, nbytes
                return

        endif

;       Get HIT status data from word 11 for this record
        get_hword11_LZ, t11, w11

;       Get HIT status data from word 15 for this record
        get_hword15_LZ, t15, w15

;       Load data into full array
        times( jrec) = t11
        word11(0, jrec) = w11
        word15(0, jrec) = w15

;       Increment total number output structures
        jrec = jrec + 1L

endfor   ; End of LZ record loop

;       Load status data into structure array
times = times(0:jrec-1)
word11 = transpose( word11(*, 0:jrec-1))
word15 = transpose( word15(*, 0:jrec-1))
load_hdpustat, word11, word15, hdpu_stat

end
;-------------------------------------------------------------------------     
pro get_hdpuhk, times, hhk, err

;       Get HIT DPU Housekeeping for listing or plot    
;       from a specified number of LZ records         
    
; Inputs: None    
    
; Outputs:    
;	times		Universal times in msec
;	hhk		HDPU housekeeping array
;       err		Error string, null for no error   

; BLOCK: interval   
;	times_rng	Time min and max in seconds from start of day   
;	rec_rng		Record number min and max   
   
common interval, times_rng, rec_rng    
    
;       Number of possible sectors   
nrecs = rec_rng(1) - rec_rng(0) + 1    
   
;       Initialize output record count 
jrec = 0L 
 
;  Initialize good data found flag to false
data_found = 0   
   
;	Setup arrays to hold raw data
times11 = lonarr( nrecs)
word11 = intarr(17, nrecs)
    
;       Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    
    
;		Output flagged data
                times = -1L
                hhk = -1.
                return    
    
        endif    
    
;       Get DPU status structure   
        get_htmstat_LZ, tmstat   
   
;	Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin

;           Get HIT status data from word 11 for this record    
            get_hword11_LZ, t11, w11
    
;           Load data into full array          
            times11( jrec) = t11
            word11(0, jrec) = w11
       
;	    Set good data found flag
	    data_found = 1

;           Increment total number output structures 
            jrec = jrec + 1    

	endif 	; tm status block
    
endfor   	; End of LZ record loop 

;	Check if any good data
if data_found ne 0 then begin

;	Transpose data to nmajor x 17
	word11 = transpose(word11)

;	Load Housekeeping data into array
	hdpu_hk_anlg, times11, word11, times, hhk 

endif else begin

	err = 'get_hdpuhk: no valid data was read.'
	times = -1
	fill = -1.e31
	hhk = replicate( fill, 16)

endelse

end    
