;------------------------------------------------------------------------------ 
; File: GET_MICS.PRO    CAMMICE Level Zero File Parsing Routines             
; Revision: 06/20/97    J L Roeder             
;------------------------------------------------------------------------------ 
; Routines:             
;      mcomb_de        Combine MICS direct events from normal and cross modes 
;      esa_devents     Compute ESA step number for each sector and event    
;      get_mde         Get MICS direct event tof's and energies 
;      get_msrates     Get MICS sectored rate scaler counts    
;      get_msrts       Get MICS sectored rate scaler counts with pitch angles
;      get_mmatrx      Get MICS matrix scalers from major frame      
;      get_mtotrts     Get MICS non-sectored spin-averaged rates counts    
;      get_mmux	       Get all MICS multiplex data 
;      get_msing       Get MICS selected-sector singles compressed counts  
;      get_msun        Get MICS sun sector and other status info
;      get_mdpustat    Get MICS DPU status structure
;      get_mdpuhk      Get MICS DPU Analog Housekeeping 
;      get_micshk      Get MICS Sensor Analog Housekeeping 
;      get_micsstat    Get MICS Sensor status bits
;------------------------------------------------------------------------------ 
pro mcomb_de, mdpu_tmstat, tof, eng, m, moq    
    
;       Get MICS direct events combining Normal and Cross Telemetry into     
;       single arrays    
    
; Inputs:    
;       mdpu_tmstat       	MICS DPU TM Status, 0-15
    
; Outputs: All dimensioned 250    
;       tof                     Time-of-flights in channel number, 0-1023    
;       eng                     Energy in channel number, 01023    
;       m                       Mass group number, 0-15    
;       moq                     Mass/Charge group number, 0-31    
    
;       Construct fill word array    
fill = replicate( -1, 50)    
      
;       Get Direct Events in normal telemetry    
get_mfde_LZ, tof1, eng1    
    
;       Number of events returned    
nev = n_elements( tof1)    
    
;       Check for good data    
if (nev gt 1L) or (tof1(0) ne -1) then begin    
    
;       In cross-mode telemetry?    
        if (mdpu_tmstat and '08'XB) ne 0B then begin    
    
;               Get Direct Events in cross-mode telemetry     
                get_mcde_LZ, tof2, eng2, m2, moq2    
    
;               Combine Normal and cross mode direct events into     
;               a single array    
;               First the time-of-flights    
                tof = intarr( nev, 5)    
                tof(0, 0) = tof1    
                tof(0, 1) = tof2    
                tof = reform( transpose( tof), nev * 5)    
    
;               The energies     
                eng = intarr( nev, 5)    
                eng(0, 0) = eng1    
                eng(0, 1) = eng2    
                eng = reform( transpose( eng), nev * 5)    
    
;               The masses    
                m = intarr( nev, 5)    
                m(0, 0) = fill(0:nev-1)    
                m(0, 1) = m2    
                m = reform( transpose( m), nev * 5)    
    
;               The masses per charge    
                moq = intarr( nev, 5)    
                moq(0, 0) = fill(0:nev-1)    
                moq(0, 1) = moq2     
                moq = reform( transpose( moq), nev * 5)    
    
        endif else begin    
    
;               Use only normal mode telemetry    
                tof = tof1    
                eng = eng1    
                m = fill(0:nev-1)    
                moq = fill(0:nev-1)    
                    
        endelse    
    
endif else begin    
    
;               No events found    
                tof = -1    
                eng = -1    
                m = -1    
                moq = -1    
    
endelse    
    
end    
;-------------------------------------------------------------------------             
pro esa_devents, recnumber, sectors, esa_steps, esa_sector, esa_cross
    
;       Compute ESA step number for each direct event from one record
    
; Inputs:
;	recnumber	Record number of this data    
;       sectors         Sector numbers, 0-31    
;       esa_steps       ESA step for each sector, valid    
;                       only when sector is zero    
;       esa_sector      ESA step for each sector valid for all sectors    
;       esa_cross       ESA step for each cross mode direct event    
    
common last_step, $	; Storage of esa step for record processed previously
	last_record, $	; Previous record number, 1-9391
	last_esa	; Previous esa step number, 0-31

;	Initialize last esa step common block if first execution
if n_elements( last_record) eq 0L then begin

	last_record = -1L
	last_esa = 255B

endif

;       Number of sectors in data arrays    
nsect = n_elements( sectors)    
    
;       Find begining of spins at sector zero    
jsect0 = where ( sectors eq 0B, njsect0)    
js0 = jsect0(0)    

;	Check if any spins found
if njsect0 ne 0 then begin

;	ESA steps at sector zero crossings
	esa0 = esa_steps( jsect0)

;	Number of sectors with zero crossings
	nesa = (njsect0 * 32) < nsect
	esa_values = esa0( lindgen( nesa) / 32L)
	loc_esa = (lindgen( nesa) mod 32L) + jsect0( lindgen( nesa) / 32L)
	esa_sector = replicate( 255B, nsect)
	esa_sector( loc_esa) = esa_values

;	Fill in esa values from last spin in previous record
	if (jsect0(0) ne 0L) and (recnumber eq last_record+1L) then $
		esa_sector(0) = replicate( last_esa, jsect0(0)<31L)

endif else begin

	if recnumber eq last_record + 1L then $
		esa_sector = replicate( last_esa, nsect) else $
		esa_sector = replicate( 255B, nsect)

endelse

;	Save last esa step from this record for next time
last_record = recnumber
last_esa = esa_sector( nsect - 1L)

;       Compute ESA step for each cross mode direct event by replicating    
esa_cross = bytarr( nsect, 5)    
esa_cross(0) = esa_sector(lindgen(nsect*5L) mod nsect)
esa_cross = reform( transpose( esa_cross), nsect * 5L)

get_mtimes, mfrm, tmaster, tfirst, tsector

end    
;-------------------------------------------------------------------------             
pro get_mde, times, eng, tof, esa, m, moq, err
    
;       Get MICS direct events for listing or plot for specified number of    
;       LZ records         
    
; Inputs: None    
    
; Outputs: All dimensioned 250 by number of records    
;       times           Universal times in long msec 
;       tof             Time-of-flights in channel number, 0-1023    
;       eng             Energy in channel number, 01023    
;       m               Mass group number, 0-15    
;       moq             Mass/Charge group number, 0-31    
;       err             Error string, null for no error    
    
; BLOCK: interval   
;      times_rng    Time min and max in seconds from start of day   
;      rec_rng     Record number min and max   
   
common interval, times_rng, rec_rng    
    
;       Number of possible direct events if in CROSS mode    
nrecs = rec_rng(1) - rec_rng(0) + 1    
ndevnts = 250L * nrecs    
    
;       Setup output arrays    
times = lonarr( ndevnts)    
eng = intarr( ndevnts)    
tof = intarr( ndevnts)    
esa = bytarr( ndevnts)    
m = bytarr( ndevnts)    
moq = bytarr( ndevnts)    
    
;       Setup working arrays for sectors and esa_steps    
nsectors = nrecs * 50L    
    
;       Initialize number of events and sectors stored    
idevnt = 0L    
    
;       Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    

		times = -1L
                tof = -1    
                eng = -1    
                esa = 255B    
                m = -1    
                moq = -1    
                return    
    
        endif    
    
;       Get DPU status structure    
        get_mtmstat_LZ, tmstat    

;       Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;           Get MICS combined direct events    
            mcomb_de, tmstat, tof1, eng1, m1, moq1    
    
;           Check if any events found    
            if (n_elements( tof1) gt 1L) or (tof1(0) ne -1) then begin    
    
;               Get sector numbers    
                get_msect_LZ, times1, sector1   

;               Present number of sectors    
	        nsector1 = n_elements( sector1)
    	        if (nsector1 eq 1) and (sector1(0) eq 255B) then nsector1 = 0L
    	    
;	        Any good sectors?    	    
    	        if nsector1 ne 0L then begin
    	    
;                   Get ESA steps     
                    get_mesa_LZ, sector1, esa1    
    
;       	    Fill in ESA step numbers for each event
		    esa_devents, irec, sector1, esa1, esa_sector, esa_cross
    
;	            One event per sector in normal mode or five events 
;		    per sector in cross mode
        	    if (tmstat and '08'XB) ne 0B then $
			esa( idevnt) = esa_cross else esa( idevnt) = esa_sector

;                   Number of events should be 250    
                    nde = n_elements( tof1)    
    
;	    	    Number of events per sector
		    mde = nde / nsector1

;		    Replicate times to get one for each event
		    timde1 = times1( lindgen(nde) / mde)

;                   Load output array with events from this record    
		    times( idevnt) = timde1
                    tof( idevnt) = tof1    
                    eng( idevnt) = eng1    
                    m( idevnt) = m1    
                    moq( idevnt) = moq1    
    
;                   Increment total number of events    
                    idevnt = idevnt + nde

                endif	; good sectors block
    
            endif	; good events block

	endif		; tm status block

endfor    		; record loop
    
;       Reform output arrays to reduce memory use    
if idevnt gt 0 then begin

;	Eliminate unused storage
	tof = tof(0: idevnt-1)    
	eng = eng(0: idevnt-1)    
	m = m(0: idevnt-1)    
	moq = moq(0: idevnt-1)    
	esa = esa(0: idevnt-1)    
	times = times(0: idevnt-1)

endif else begin

	if err eq '' then err='get_mde: no valid data was read.'
        tof = -1    
        eng = -1    
        esa = 255B    
        m = -1    
        moq = -1    
	times = -1L

endelse

end    
;------------------------------------------------------------------------- 
pro get_msrates, times, delta, sectors, esa, chans, cnts, err
    
;       Get MICS sectored rate scaler counts for listing or plot    
;       from a specified number of LZ records         
    
; Inputs: None    
    
; Outputs:    
;       times	Universal times in long msec 
;       delta   Accumulation times in integer msec
;       sectors Sector number, 0-31, dim 50 * nrec   
;       esa     ESA step, 0-31, dim 50 * nrec   
;       chans   Channel numbers, 0-8 are MR's and 9, 10 are DCR and FSR
;	cnts	Compressed rate scaler counts
;       err     Error string, null for no error    
    
; BLOCK: interval   
;      times_rng    Time min and max in seconds from start of day   
;      rec_rng     Record number min and max   
   
common interval, times_rng, rec_rng    
    
; BLOCK: mrates     
;       possible_chan   Strings identifying channels for menu     
;       index           Index of which channels selected for listing     
     
common mrates, possible_chan, index     

if n_elements( index) eq 0L then mrates_init

;	Initialize number of sectors accumulated
nsectors = 0L

;      Check if any channnels were selected   
if index(0) ne -1 then begin   
   
;       Number of possible sectors   
       nrecs = rec_rng(1) - rec_rng(0) + 1    
       nsectors = 50L * nrecs    
   
;      Number of channels to extract   
       nchan = n_elements( index)

;      Output index of channels
       chans = index
       
;      Setup output arrays    
       times = lonarr( nsectors)    
       delta = intarr( nsectors)    
       sectors = bytarr( nsectors)    
       esa = bytarr( nsectors)    
       cnts = intarr( nchan, nsectors)    
    
;      Initialize number of events and sectors stored    
       isector = 0L    
    
;      Process one major frame at a time    
       for irec = rec_rng(0), rec_rng(1) do begin    
	    
;          Unpack a single record    
           unpack_cam, irec, err    
    
;          Test for read error    
           if err ne '' then begin    
    
               mf = -1B   
               cnts = -1   
               esa = -1B   
               times = -1L
               sectors = -1B   
               return    
    
           endif    
    
;          Get DPU status structure    
           get_mtmstat_LZ, tmstat    

;          Check for DPU in sensor data mode
           if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;             Get sector numbers    
              get_msect_LZ, times1, sector1, delta1
    
;             Present number of sectors    
	      nsect1 = n_elements( sector1)
    	      if (nsect1 eq 1) and (sector1(0) eq 255B) then nsect1 = 0L
    	    
;	      Any good sectors?    	    
    	      if nsect1 ne 0L then begin
    	    
;                Get MICS sectored rate scaler counts for this record    
                 get_msrts_LZ, index, cnts1    
    
;                Get ESA steps     
                 get_mesa_LZ, sector1, esa1    
    
;      		 Fill in ESA step numbers for all sectors   
       		 esa_devents, irec, sector1, esa1, esa_sectors, esa_cross

;                Load data into full array          
                 cnts(0, isector) = cnts1    
                 times( isector) = times1    
                 delta( isector) = delta1    
                 sectors( isector) = sector1    
                 esa( isector) = esa_sectors
    
;                Increment total number of sectors
                 isector = isector + nsect1
              
              endif    	; good sector block
    
	   endif 	; tmstat block

       endfor    	; record loop
    
       nsectors = isector

endif else err = 'get_msrates: no channels selected.'

;       Check if any sectors read
if nsectors gt 0L then begin

;      Close excess storage
       times = times( 0:nsectors-1)
       delta = delta( 0:nsectors-1)
       sectors = sectors( 0:nsectors-1)
       cnts = cnts( *, 0:nsectors-1)
       esa = esa( 0:nsectors-1)

endif else begin   
   
       if err eq '' then err = 'get_msrates: No valid data was read.'
       delta = -1B   
       cnts = -1   
       esa = -1B   
       times = -1L
       sectors = -1B   
   
endelse   
 
;	Output channel numbers selected
if n_elements( index) ne 0 then chans = index

end    
;------------------------------------------------------------------------------    
pro get_msrts, pa_mode, times, delta, sectors, esa, chans, cnts, pang, err   
       
;       Get MICS sectored rate scaler counts with pitch angles    
;       for listing or plot from a specified number of LZ records            
       
; Inputs: 
;	pa_mode	Pitch angle mode, 0=0-180 degrees, 1=0-360 degrees
       
; Outputs:       
;       times   Universal times in long msec    
;       delta   Accumulation times in integer msec   
;       sectors Sector number, 0-31, dim 50 * nrec      
;       esa     ESA step, 0-31, dim 50 * nrec      
;       chans   Channel numbers, 0-8 are MR's and 9, 10 are DCR and FSR   
;       cnts    Compressed rate scaler counts   
;       pang    Pitch angles in degrees   
;       err     Error string, null for no error       
       
; BLOCK: interval      
;      times_rng    Time min and max in seconds from start of day      
;      rec_rng     Record number min and max      
      
common interval, times_rng, rec_rng       
       
; BLOCK: mrates        
;       possible_chan   Strings identifying channels for menu        
;       index           Index of which channels selected for listing        
        
common mrates, possible_chan, index        
   
if n_elements( index) eq 0L then mrates_init   
   
;       Initialize number of sectors accumulated   
nsectors = 0L   
   
;      Check if any channnels were selected      
if index(0) ne -1 then begin      
      
;       Number of possible sectors      
       nrecs = rec_rng(1) - rec_rng(0) + 1       
       nsectors = 50L * nrecs       
      
;      Number of channels to extract      
       nchan = n_elements( index)   
   
;      Output index of channels   
       chans = index   
          
;      Setup output arrays       
       times = lonarr( nsectors)       
       delta = intarr( nsectors)       
       sectors = bytarr( nsectors)       
       esa = bytarr( nsectors)       
       magaz = bytarr( nsectors)       
       magel = bytarr( nsectors)  
       revoff = intarr( nsectors)  
       auxoff = intarr( nsectors)  
       cnts = intarr( nchan, nsectors)       
       
;      Initialize number of events and sectors stored       
       isector = 0L       

;      Process one major frame at a time       
       for irec = rec_rng(0), rec_rng(1) do begin       

;          Unpack a single record       
           unpack_cam, irec, err       
       
;          Test for read error       
           if err ne '' then begin       
       
               mf = -1B      
               cnts = -1      
               esa = -1B      
               pa = -1.      
               times = -1L   
               sectors = -1B      
               return       
       
           endif       
       
;          Get DPU status structure       
           get_mtmstat_LZ, tmstat       
   
;          Check for DPU in sensor data mode   
           if (tmstat gt 3B) and (tmstat le 15B) then begin   
       
;             Get sector numbers       
              get_msect_LZ, times1, sector1, delta1   
       
;             Present number of sectors       
              nsect1 = n_elements( sector1)   
              if (nsect1 eq 1) and (sector1(0) eq 255B) then nsect1 = 0L   
               
;             Any good sectors?                
              if nsect1 ne 0L then begin   
               
;                Get MICS sectored rate scaler counts for this record       
                 get_msrts_LZ, index, cnts1       
       
;                Get ESA steps        
                 get_mesa_LZ, sector1, esa1       
       
;                Fill in ESA step numbers for all sectors      
                 esa_devents, irec, sector1, esa1, esa_sectors, esa_cross   
   
;                Get MUX data   
                 get_mmux_LZ, mux_data   
   
;                Fill in MagEl and MagAz for all sectors   
                 mmagsect, irec, sector1, mux_data, magaz1, magel1   
  
;                Get offsets for MICS  
                 get_moff_LZ, revoff1, auxoff1  
   
;                Load data into full array             
                 cnts(0, isector) = cnts1       
                 times( isector) = times1       
                 delta( isector) = delta1       
                 sectors( isector) = sector1       
                 esa( isector) = esa_sectors   
                 magaz( isector) = magaz1   
                 magel( isector) = magel1  
                 revoff( isector) = replicate( revoff1, nsect1)   
                 auxoff( isector) = replicate( auxoff1, nsect1)   
       
;                Increment total number of sectors   
                 isector = isector + nsect1   
                 
              endif     ; good sector block   
       
           endif        ; tmstat block   
   
       endfor           ; record loop   
       
       nsectors = isector   
   
endif else err = 'get_msrates: no channels selected.'   
   
;       Check if any sectors read   
if nsectors gt 0L then begin   
   
;      Close excess storage   
       times = times( 0:nsectors-1)   
       delta = delta( 0:nsectors-1)   
       sectors = sectors( 0:nsectors-1)   
       cnts = cnts( *, 0:nsectors-1)   
       esa = esa( 0:nsectors-1)   
       magaz = magaz( 0:nsectors-1)   
       magel = magel( 0:nsectors-1)   
       revoff = revoff( 0:nsectors-1)  
       auxoff = auxoff( 0:nsectors-1)  
    
;      Convert magaz and magel to radians   
       azimuth = magaz * ( !Pi / 128.)    
       degrad = !Pi / 180.   
       elev = magel * degrad   
   
;      MICS azimuths in radians from sectors and offsets   
       micsaz = degrad * (126.1 + 11.25 * ( sectors + revoff + auxoff / 128.))
       micsaz = micsaz mod (2 * !Pi)
   
;      MICS field of view unit vector   
       mics_x = cos( micsaz)   
       mics_y = sin( micsaz)   
   
;      Magnetic unit vector   
       sinel = sin( elev)   
       Bx = cos( azimuth) * sinel    
       By = sin( azimuth) * sinel   
   
;      Compute pitch angles in degrees
       pang = 180. - acos( Bx * mics_x + By * mics_y) / degrad

;      Full 360 degree angles if requested
       if pa_mode ne 0 then begin

	   micsaz180 = (micsaz + !Pi) mod (2. * !Pi)
           jpang1 = where( (micsaz lt !Pi) and (azimuth ge micsaz) and $
			(azimuth lt micsaz180))
	   jpang2 = where( (micsaz ge !Pi) and (azimuth ge micsaz))
           jpang3 = where( (micsaz ge !Pi) and (azimuth lt micsaz180))
           if jpang1(0) ne -1L then pang( jpang1) = 360. - pang( jpang1)
           if jpang2(0) ne -1L then pang( jpang2) = 360. - pang( jpang2)
           if jpang3(0) ne -1L then pang( jpang3) = 360. - pang( jpang3)
  
	endif

       jbad = where( (sectors eq 255B) or (magaz eq -1) or (magel eq -1) or $  
               (revoff eq -1) or (auxoff eq -1))  
       if jbad(0) ne -1L then pang( jbad) = -1.  
   
endif else begin      
      
       if err eq '' then err = 'get_msrates: No valid data was read.'   
       delta = -1B      
       cnts = -1      
       esa = -1B      
       pa = -1.   
       times = -1L   
       sectors = -1B      
      
endelse      
    
;       Output channel numbers selected   
if n_elements( index) ne 0 then chans = index   
   
end       
  
;-------------------------------------------------------------------------             
pro get_mmatrx, times, cnts, odd, err
    
;       Get MICS compressed matrix scaler counts for listing    
;       or plot from a specified number of LZ records         
    
; Inputs: None    
    
; Outputs:    
;       times   Universal times in long msec 
;       cnts    Matrix compressed counts
;	odd	Even or odd major frame
;       err     Error string, null for no error    
    
; BLOCK: interval   
;      times_rng    Time min and max in seconds from start of day   
;      rec_rng     Record number min and max   
   
common interval, times_rng, rec_rng    
   
;  Number of possible experiment master frames   
   nrecs = rec_rng(1) - rec_rng(0) + 1    
   nmaster =  nrecs / 11 + 2    
  
;  Always same number of matrix scalers in a master frame  
   nmatrix = 512L  
      
;  Setup output arrays    
   cnts = replicate( 255B, nmatrix, nmaster)
   times = replicate( -1L, nmaster)
   odd = replicate( 255B, nmaster)
    
;  Initialize number of master frames stored    
   imaster = 0L    
    
;  Initialize good data found flag to false
   data_found = 0

;  Initialize value of mfc for previous record processed  
   last_mfc = 255B  
  
;  Process one major frame at a time    
   for irec = rec_rng(0), rec_rng(1) do begin    
    
;      Unpack a single record    
       unpack_cam, irec, err    
    
;      Test for read error    
       if err ne '' then begin    
    
	   err = 'get_mmatrx: Input File Read Error'
           cnts = -1B   
           times = -1L
	   odd = 255B
           return    
    
       endif    
    
;      Get DPU status structure    
       get_mtmstat_LZ, tmstat    

;      Check for DPU in sensor data mode
	if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;          Get MICS matrix scaler counts from one record    
           get_mmatrx_LZ, mfc, cnts1, odd1
 
;          Check if good data was returned 
           if mfc ne 255B then begin 
 
;               Index into matrix array is found from the Master Frame Counter  
                indx = 48 * mfc 
 
;               Increment master frames based on last mfc or mfc to improve  
;               noise immunity  
                if (last_mfc eq 10) and (mfc eq 0) then imaster = imaster + 1L  
  
;               Set good data found flag
                data_found = 1

;               Load data into full array          
                cnts( indx, imaster) = cnts1
                odd( imaster) = odd1

;	        Obtain UT of start of master frame
                get_mtimes, mfrm, tmaster, tfirst, tsector

;	        Accumulation over 16 spins beginning at start of master frame
	        times(imaster) = tmaster + 256L * tsector / 1000L

;               Keep of last mfc  
                last_mfc = mfc   
  
             endif else last_mfc = 255B 	; Master frame counter block

	endif else last_mfc = 255B 		; tm status block
 
endfor    					; Record block
 
;       Check if any data accumulated
if data_found ne 0 then begin

;       Delete any unused master frame storage 
	nmaster = imaster+1L 
	cnts = cnts( *, 0:nmaster-1L) 
	times = times(0:nmaster-1L)
	odd = odd(0:nmaster-1L)
 
;       Format output array for mass and mass/charge groups    
	nm = 16 
	nmoq = 32 
	cnts = reform( cnts, nm, nmoq, nmaster) 

endif else begin

;       Output flagged data with error
        err = 'get_mmatrx: No valid data read.'
        times = -1L
        cnts = 255B
	odd = 255B

endelse
 
end    
;-------------------------------------------------------------------------             
pro get_mtotrts, times, delta, esa, chans, counts, err
  
;  Get MICS spin-averaged rate scaler counts from a range of records           
  
; Inputs: None    
    
; Outputs:    
;       times           Universal times in long msec 
;	delta		Accumulation times (spin periods) in msec
;       esa             ESA step, 0-31, dim 50 * nrec   
;       chans           Channel numbers, 0-15 are MR Totals 
;       cnts            Spin-averaged compressed rate scaler counts  
;       err     	Error string, null for no error    
    
; BLOCK: interval   
;       times_rng       Time min and max in seconds from start of day   
;       rec_rng         Record number min and max   
   
common interval, times_rng, rec_rng    
    
; BLOCK: msrates     
;       possible_chan   Strings identifying channels for menu     
;       index           Index of which channels selected for listing     
     
common msrates, possible_chan, index     
     
if n_elements( index) eq 0L then msrates_init

;   Output channels selected
chans = index       

;      Check if any channnels were selected   
if index(0) ne -1 then begin   
   
;   Number of channels to extract   
    nchans = n_elements( index)   

;   Number of possible sectors, spins, and points 
    nrecs = rec_rng(1) - rec_rng(0) + 1    
    nsectors = 50L * nrecs 
    nspins = nsectors / 32L + 2L 
    npoints = nchans * nspins   
   
;   Setup output and work arrays    
    tims = lonarr( npoints)
    delt = lonarr( npoints)
    chs = bytarr( npoints)
    esa_steps = bytarr( npoints)
    cnts = bytarr( npoints)

;   Initialize number of points stored    
    ipoint = 0L    
	    
;   Process one major frame at a time    
    for irec = rec_rng(0), rec_rng(1) do begin    
    
;	Unpack a single record    
	unpack_cam, irec, err    
    
;	Test for read error    
	if err ne '' then begin    
    
;	Output one empty set of counts
	    times = -1L
	    delta = -1L
	    counts = -1   
	    esa = -1B   
	    return    
    
	endif    
    
;	Get DPU status structure    
	get_mtmstat_LZ, tmstat    

;	Check for DPU in sensor data mode
	if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;	    Get sector numbers    
	    get_msect_LZ, ts1, sector1    

;	    Get MICS sectored rate scaler counts for this record    
	    get_mtotrts_LZ, index, sector1, tims1, delt1, chs1, esa1, cnts1
    
;   	    Check if good data
	    if tims1(0) ne -1L then begin

;		Load data into full array          
    		cnts( ipoint) = cnts1    
    		chs( ipoint) = chs1 
    		tims( ipoint) = tims1
    		delt( ipoint) = delt1
    		esa_steps( ipoint) = esa1    
    
;		Increment total number of sectors and points   
    		ipoint = ipoint + n_elements( cnts1) 

    	    endif	; good data block

    	endif       ; tm status block
    
    endfor 		; record loop
       
endif else err = 'get_mtotrts: no channels selected.'

;       Check if any data received
if ipoint gt 0L then begin

;   Trim arrays to size of data received 
    esa_steps = esa_steps( 0:ipoint-1) 
    cnts = cnts( 0:ipoint-1) 
    chs = chs( 0:ipoint-1) 
    tims = tims( 0:ipoint-1) 
    delt = delt( 0:ipoint-1) 

;   Channel identifiers for decommutating
    chanindex = intarr( n_elements( possible_chan))
    chanindex( index) = indgen( nchans)
    chan_id = chanindex( chs)
 
;   Reformat counts and times to align channels in 2-d array
    blkwid = nchans
    build_index, chan_id, blkwid, indx

    decomm_data, blkwid, indx, tims, times
    decomm_data, blkwid, indx, delt, delta
    decomm_data, blkwid, indx, esa_steps, esa
    decomm_data, blkwid, indx, cnts, counts

;   Reformat time, delta time, and esa array to give one value per spin
    times = transpose( times)
    reform_data, -1L, times

    delta = transpose( delta)
    reform_data, -1L, delta

    esa = transpose( esa)
    reform_data, 255B, esa
 
endif else begin   
   
    err = 'get_mtotrts: no valid data was read.'
    times = -1L
    delta = -1L
    counts = -1   
    esa = -1B   
   
endelse   
 
end    
;-----------------------------------------------------------------------------
pro get_mmux, tmstr, tsect, mframe, mspin, sector, mux, err
  
;  Get MICS multiplex data from a range of records in a level zero file
  
; Inputs: None    
    
; Outputs:  all arrays approximately dim (50 * nrec)
;	tmstr		Universal time of master frame accumulation start
;	tsect		Sector period in MICROSECONDS
;	mframe		Master frame counter
;	mspin		Spin counter, 0-17
;	sector		Sector number, 0-32
;	mux		Multiplex data
    
; BLOCK: interval   
;       times_rng       Time min and max in seconds from start of day   
;       rec_rng         Record number min and max   
   
common interval, times_rng, rec_rng    
   
;      Number of possible sectors, spins, and points 
nrecs = rec_rng(1) - rec_rng(0) + 1    
nsectors = 50L * nrecs 
   
;      Setup output and work arrays    
tmstr = replicate( -1L, nsectors)
tsect = replicate( -1L, nsectors)
mframe = replicate( -1, nsectors)
mspin = replicate( 255B, nsectors)
sector = replicate( 255B, nsectors)
mux = replicate( 255B, nsectors)

;	Number of records and sectors actually stored
isector = 0L

;	Experiment master frame counter
frame_cnt = 0
    
;      Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    

;		Output one empty sector of data
		sector = 255B
		mframe = -1
		mspin = 255B
	        tmstr = -1L
		tsect = -1L
		mux = 255B
		return    
    
        endif    
    
;       Get DPU status structure    
        get_mtmstat_LZ, tmstat    

;       Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;           Get sector numbers    
            get_msect_LZ, times1, sector1

;           Get all MICS mux data for this record    
            get_mmux_LZ, mux1
    
;           Obtain master frame times and sector interval 
	    get_mtimes, mfrm1, tmaster1, tfirst1, tsect1

;	    Start new master frames if necessary
	    if mfrm1 eq 0B then frame_cnt = frame_cnt + 1

;	    Number of sectors received
	    nsect1 = n_elements( sector1) 
    
;           Load data into full array
            mux( isector) = mux1
            sector( isector) = sector1
	    mframe( isector) = replicate( frame_cnt, nsect1)
	    mspin( isector) = byte((indgen( nsect1) + mfrm1*50) / 32)
	    tmstr( isector) = replicate( tmaster1, nsect1)
	    tsect( isector) = replicate( tsect1, nsect1)
    
;           Increment total number of sectors and points   
            isector = isector + nsect1

        endif           ; tm status block
    
endfor 			; record loop

;       Check if any data read
if isector gt 0L then begin
 
;	Trim arrays to size of data received 
	mux = mux( 0:isector-1) 
	sector = sector( 0:isector-1) 
	tmstr = tmstr( 0:isector-1) 
	tsect = tsect( 0:isector-1) 
	mspin = mspin( 0:isector-1) 
	mframe = mframe( 0:isector-1) 
 
endif else begin

;	Output one empty sector of data
	err = 'get_mmux: No valid data was read.'
	sector = 255B
	mframe = -1
	mspin = 255B
	tmstr = -1L
	tsect = -1L
	mux = 255B

endelse
 
end    
;------------------------------------------------------------------------------ 
pro get_msing, esa, timesa, dlta, sectora, countsa, $
	timesb, dltb, sectorb, countsb, err
  
;  Get MICS spin-averaged rate scaler counts from a range of records           
  
; Inputs: None    
    
; Outputs:    
;       esa             ESA step, 0-31, dim 50 * nrec   
;       timesa		Universal times of perpendicular sectors in long msec 
;	dlta		Accumulation time in long msec for sectora
;	sectora		Sector number of perp singles counts
;       countsa		Compressed singles counts for sectora
;       timesb		Universal times of parallel sectors in long msec 
;	dltb		Accumulation time in long msec for sectorb
;	sectorb		Sector number of parallel singles counts
;       countsb		Compressed singles counts for sectorb
;       err     	Error string, null for no error    
    
; BLOCK: interval   
;       times_rng       Time min and max in seconds from start of day   
;       rec_rng         Record number min and max   
   
common interval, times_rng, rec_rng    
    
;   Number of possible sectors, spins, and points 
nrecs = rec_rng(1) - rec_rng(0) + 1    
nsectors = 50L * nrecs 
nspins = nsectors / 32L + 5L 
npoints = nspins  * 10L
   
;   Setup output and work arrays    
tspns = lonarr( npoints)
tsects = lonarr( npoints)
chs = bytarr( npoints)
esa_steps = bytarr( npoints)
dat = bytarr( npoints)

;   Initialize number of points stored    
ipoint = 0L    
	    
;   Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;	Unpack a single record    
	unpack_cam, irec, err    
    
;	Test for read error    
	if err ne '' then begin    
    
;	Output one empty set of counts
	    esa = 255B   
	    timesa = -1L
	    dlta = -1L
	    sectora = 255B
	    countsa = 255B  
	    timesb = -1L
	    dltb = -1L
	    sectorb = 255B
	    countsb = 255B  
	    return    
    
	endif    
    
;	Get DPU status structure    
	get_mtmstat_LZ, tmstat    

;	Check for DPU in sensor data mode
	if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;	    Get sector numbers and accumulation times
	    get_msect_LZ, ts1, sector1

;	    Get MICS sectored rate scaler counts for this record    
	    get_msing_LZ, sector1, tspns1, tsects1, chs1, esa1, dat1
    
;   	    Check if good data
	    if tspns1(0) ne -1L then begin

;		Load data into full array          
    		tspns( ipoint) = tspns1
    		tsects( ipoint) = tsects1
    		chs( ipoint) = chs1 
    		esa_steps( ipoint) = esa1    
    		dat( ipoint) = dat1
    
;		Increment total number of sectors and points   
    		ipoint = ipoint + n_elements( dat1) 

    	    endif	; good data block

    	endif       	; tm status block
    
endfor 			; record loop

;       Check if any data received
if ipoint gt 0L then begin

;   Trim arrays to size of data received 
    tspns = tspns( 0:ipoint-1) 
    tsects = tsects( 0:ipoint-1) 
    chs = chs( 0:ipoint-1) 
    esa_steps = esa_steps( 0:ipoint-1) 
    dat = dat( 0:ipoint-1) 

;   Reformat counts and times to align channels in 2-d array
    blkwid = 10
    build_index, chs, blkwid, indx

    decomm_data, blkwid, indx, tspns, tspins
    decomm_data, blkwid, indx, tsects, tsectors
    decomm_data, blkwid, indx, esa_steps, esa
    decomm_data, blkwid, indx, dat, data

;   Reformat spin times, sector period, accumulation times and esa array 
;   to give 1 value/spin
    tspins = transpose( tspins)
    reform_data, -1L, tspins

    tsectors = transpose( tsectors)
    reform_data, -1L, tsectors

    esa = transpose( esa)
    reform_data, 255B, esa
 
;   Separate counts from sectors
    data = transpose( data)
    sectora = data( *, 2)
    countsa = transpose( data( *, [0, 3, 5, 8]))
    sectorb = data( *, 7)
    countsb = transpose( data( *, [1, 4, 6, 9]))

;   Calculate times for centers of sectors a and b
    ta = tspins + long( tsectors * (sectora + 0.5)) / 1000L
    timesa = ta * ((tspins ne -1L) and (sectora ne 255B)) - $
	((tspins eq -1L) or (sectora eq 255B))

    tb = tspins + long( tsectors * (sectorb + 0.5)) / 1000L
    timesb = tb * ((tspins ne -1L) and (sectorb ne 255B)) - $
	((tspins eq -1L) or (sectorb eq 255B))

;   Compute accumulation times of sectors a and b
    tdead = 58		; Dead time in milliseconds for sector zero
    dlta = byte((tsectors /1000L) * (sectora ne 255B) $
		- tdead * (sectora eq 0B) + 255 * (sectora eq 255B))

    dltb = byte((tsectors /1000L) * (sectorb ne 255B) $
		- tdead * (sectorb eq 0B) + 255 * (sectorb eq 255B))

endif else begin   
   
    esa = 255B
    timesa = -1L
    dlta = -1L
    sectora = 255B
    countsa = 255B
    timesb = -1L
    dltb = -1L
    sectorb = 255B
    countsb = 255B
    err = 'get_msing: no valid data was read.'
   
endelse   
 
end    
;-----------------------------------------------------------------------------
pro get_msun, times, msun_stat, err
  
;  Get MICS sun sector and other information from a range of records           
  
; Inputs: None    
    
; Outputs:
;	times		Times of starts of spins
;	msun_stat	MICS Sun Status Structure defined in MICS_SUN.PRO
;       err     	Error string, null for no error    
    
;	Obtain all multiplex data
get_mmux, tmaster, tsector, mframe, mspin, sector, mux, err       

;       Test for read error    
if err ne '' then begin    

;	Output one empty set of singles    
	def_msunstat_strc, msun_stat, nbytes
	times = -1L
	return    
    
endif    
    
;	Number of sectors
nsectors = n_elements( sector)

;      Provide all status bytes
nchans = 3

;       Locations of singles rates in MUX array
sect_index = [8, 16, 24]

;       Setup channel id arrays
chan = replicate( 255B, nsectors)

;       Mark data one channel at a time
for ichan = 0, nchans - 1 do begin
        jindex = where( sector eq sect_index( ichan))
        if jindex(0) ne -1 then chan( jindex) = ichan
endfor

;       Which data were marked?
chans = where( chan ne 255B, nc)

;       Check if any data available
if nc ne 0L then begin

;       Extract the Sector-A data which were marked
        channels = chan( chans)
        stats = mux( chans)
	m_spins = mspin( chans)
	t_mastr = tmaster( chans)
	t_sector = tsector( chans)

;	Compute universal times of selected sectors
	timspins = t_mastr + $
		long( (32L * m_spins) * t_sector ) / 1000L

;	Decommutate count arrays to align counts as 3 channels x nspins
	blkwid = 3
	build_index, channels, blkwid, indx
	decomm_data, blkwid, indx, stats, status
	decomm_data, blkwid, indx, timspins, times

;	Pick out individual status words from array
	status = transpose( status)

;	Put status bits into output structure
	load_msunstat, status, msun_stat

;	Reformat data to give one value per spin
	times = transpose( times)
	reform_data, -1L, times

endif else begin

        err = 'get_msun: no valid status data was read.'
	def_msunstat_strc, msun_stat, nbytes
	times = -1L

endelse

 
end    
;-----------------------------------------------------------------------------
pro get_mdpustat, times, mstat_struc, err
    
;       Get MICS DPU Status for listing or plot    
;       from a specified number of LZ records         
    
; Inputs: None    
    
; Outputs:
;	times		Universal times of status readout, msec
;	mstat_strc	MDPU Status Structure  
;       err     	Error string, null for no error    
    
; BLOCK: interval   
;      times_rng    Time min and max in seconds from start of day   
;      rec_rng     Record number min and max   
   
common interval, times_rng, rec_rng    
    
;       Number of possible sectors   
nrecs = rec_rng(1) - rec_rng(0) + 1    
   
;       Initialize output record count 
jrec = 0L 
 
;	Setup arrays to hold raw data
times = lonarr( nrecs)
word12 = intarr(21, nrecs)
word15 = intarr(8, nrecs)
    
;       Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    
    
;		If error then output one empty structure
                times = -1L
		def_mdpustat_strc, mstat_struc, nbytes   
                return    
    
        endif    
    
;       Get MICS status data from word 12 for this record    
        get_mword12_LZ, t12, w12
    
;       Get MICS status data from word 15 for this record    
        get_mword15_LZ, t15, w15
    
;       Load data into full array
	times( jrec) = t12
        word12(0, jrec) = w12
        word15(0, jrec) = w15
       
;       Increment total number output structures 
        jrec = jrec + 1    
    
endfor   ; End of LZ record loop 
    
;	Load status data into structure array
times = times(0:jrec-1)
word12 = transpose( word12(*, 0:jrec-1))
word15 = transpose( word15(*, 0:jrec-1))
load_mdpustat, word12, word15, mstat_struc

end    
;-------------------------------------------------------------------------             
pro get_mdpuhk, times, mhk, err
    
;       Get MICS DPU Housekeeping for listing or plot    
;       from a specified number of LZ records         
    
; Inputs: None    
    
; Outputs:
;	times		Universal times of housekeeping readout, msec
;	mhk		MDPU housekeeping array
;       err     	Error string, null for no error    

; BLOCK: interval   
;	times_rng	Time min and max in seconds from start of day   
;	rec_rng		Record number min and max   
   
common interval, times_rng, rec_rng    
    
;       Number of possible sectors   
nrecs = rec_rng(1) - rec_rng(0) + 1    
   
;       Initialize output record count 
jrec = 0L 
 
;  Initialize good data found flag to false
data_found = 0

;	Setup arrays to hold raw data
times12 = lonarr( nrecs)
word12 = intarr(21, nrecs)
    
;       Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    
    
;		Output one array with fills
		times = -1L
		fill = -1.e31
                mhk = replicate( fill, 16) 
                return    
    
        endif    
    
;       Get DPU status structure    
        get_mtmstat_LZ, tmstat    

;       Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;           Get MICS status data from word 12 for this record    
            get_mword12_LZ, t12, w12
    
;           Load data into full array          
	    times12( jrec) = t12
            word12(0, jrec) = w12
       
;           Set good data found flag
            data_found = 1

;           Increment total number output structures 
            jrec = jrec + 1    

        endif   ; tm status block
    
endfor   	; End of LZ record loop 
    
;       Check if any good data
if data_found ne 0 then begin

;	Transpose data to nrec x 21
	word12 = transpose(word12)

;	Reformat data into housekeeping array
	mdpu_hk_anlg, times12, word12, times, mhk 

endif else begin

        err = 'get_mdpuhk: no valid data was read.'
        times = -1
        fill = -1.e31
        mhk = replicate( fill, 16)

endelse

end    
;-------------------------------------------------------------------------             
pro get_micshk, times, micshk, err
    
;       Get MICS Sensor Housekeeping for listing or plot    
;       from a specified number of LZ records         
    
; Inputs: None    
    
; Outputs:
;	times		Universal times of housekeeping readout, msec
;	micshk		MICS housekeeping array
;       err     	Error string, null for no error    

; BLOCK: interval   
;	times_rng	Time min and max in seconds from start of day   
;	rec_rng		Record number min and max   
   
common interval, times_rng, rec_rng    
    
;       Number of possible sectors   
nrecs = rec_rng(1) - rec_rng(0) + 1    
   
;       Initialize output record count 
jrec = 0L 
 
;	Initialize good data found flag to false
data_found = 0

;	Setup arrays to hold raw data
times12 = lonarr( nrecs)
word12 = intarr(21, nrecs)
    
;       Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    
    
                times = -1L
	        fill = -1.e31
		micshk = replicate( fill, 7)
                return    
    
        endif    
    
;       Get DPU status structure    
        get_mtmstat_LZ, tmstat    

;       Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;           Get MICS status data from word 12 for this record    
            get_mword12_LZ, t12, w12
    
;           Load data into full array          
            times12( jrec) = t12
            word12(0, jrec) = w12
       
;           Set good data found flag
            data_found = 1

;           Increment total number output structures 
            jrec = jrec + 1    
    
        endif   ; tm status block
    
endfor   	; End of LZ record loop 
    
;       Check if any good data
if data_found ne 0 then begin

;	Transpose data to nmajor x 21
	word12 = transpose(word12)

;	Reformat data into housekeeping array
	mics_hk_anlg, times12, word12, times, micshk 

endif else begin

        err = 'get_micshk: no valid data was read.'
        times = -1
        fill = -1.e31
        micshk = replicate( fill, 7)

endelse

end    
;-------------------------------------------------------------------------             
pro get_micsstat, times, micsstat, err

;       Get MICS Sensor status for listing or plot    
;       from a specified number of LZ records         
    
; Inputs: None    
    
; Outputs:
;	times		Universal times of housekeeping readouts, msec
;	micsstat	MICS sensor status longword 
;       err     	Error string, null for no error    

; BLOCK: interval   
;	times_rng	Time min and max in seconds from start of day   
;	rec_rng		Record number min and max   
   
common interval, times_rng, rec_rng    
    
;       Number of possible sectors   
nrecs = rec_rng(1) - rec_rng(0) + 1    
   
;       Initialize output record count 
jrec = 0L 
 
;	Initialize good data found flag to false
data_found = 0

;	Setup arrays to hold raw data
times12 = lonarr( nrecs)
word12 = intarr(21, nrecs)
    
;       Process one major frame at a time    
for irec = rec_rng(0), rec_rng(1) do begin    
    
;       Unpack a single record    
        unpack_cam, irec, err    
    
;       Test for read error    
        if err ne '' then begin    
    
                micsstat = -1   
                return    
    
        endif    
    
;       Get DPU status structure    
        get_mtmstat_LZ, tmstat    

;       Check for DPU in sensor data mode
        if (tmstat gt 3B) and (tmstat le 15B) then begin
    
;       	Get MICS status data from word 12 for this record    
	        get_mword12_LZ, t12, w12
    
;       	Load data into full array          
	        times12( jrec) = t12
	        word12(0, jrec) = w12
       
;		Set good data found flag
	        data_found = 1

;       	Increment total number output structures 
	        jrec = jrec + 1    

	endif	; TM status block
    
endfor   ; End of LZ record loop 
    
;       Check if any good data
if data_found ne 0 then begin

;	Transpose data to nmajor x 21
	word12 = transpose(word12)

;	Reformat data into sensor status array
	mics_hk_stat, times12, word12, times, micsstat

endif else begin

        err = 'get_micsstat: no valid data was read.'
        times = -1
        micsstat = -1L

endelse

end    
