;------------------------------------------------------------------------------      
; File: HEAD_LZ.PRO     Level Zero File Header Parsing Routines               
; Revision: 08/22/96    J L Roeder               
;------------------------------------------------------------------------------      
; Routines:         
;       sc_clock        Parses S/C clock bytes          
;       def_flr_strc    Defines structure for LZ File Label Record         
;       LZ_flr          Parses the File Label Record (FLR)               
;       def_drh_strc    Defines structure for Level Zero Data Record Header      
;       LZ_drh          Parses the Data Record Header (DRH)               
;------------------------------------------------------------------------------      
pro sc_clock, data_bytes, clk_words              
           
;  Parses Spacecraft clock bytes per DFCD instructions              
           
; Inputs:           
;  data_bytes  Array of eight bytes containing S/C clock information           
;              from the Level Zero file           
         
; Outputs:         
;  clk_words   Longword array consisting of a truncated Julian date,          
;              seconds from the start of the day, leftover millsec,         
;              and leftover tenths of microsec         
           
;  Number of clocks to process              
nclocks = n_elements( data_bytes) / 8              
           
;  Setup output array              
clk_words = lonarr( nclocks, 5)              
           
;               Work with transposed input array, dim nclocks x 8              
dbt = transpose( data_bytes)           
         
;  Extract tenths of microseconds           
microsec10_lsb = fix( dbt(*, 0) and 224B)           
microsec10_msb = fix( dbt(*, 1) and 127B)          
clk_words(0, 4) = ishft( microsec10_lsb, -3) + ishft( microsec10_msb, 1)            
           
;  Extract microseconds           
microsec_lsb = fix( dbt(*, 1) and 'F0'XB)           
microsec_msb = fix( dbt(*, 2) and '3F'XB)           
clk_words(0, 3) = microsec_lsb + ishft( microsec_msb, 4)            
           
;  Extract milliseconds           
millisec_lsb = fix( dbt(*, 2) and 'C0'XB)           
millisec_msb = fix( dbt(*, 3))         
clk_words(0, 2) = ishft( millisec_lsb, -6) + ishft( millisec_msb, 2)            
           
;  Extract seconds           
sec_lsb = long( dbt(*, 4))         
sec_mid = long( dbt(*, 5))           
sec_msb = long( dbt(*, 6) and '01'XB)           
clk_words(0, 1) = sec_lsb + ishft( sec_mid, 8) + ishft( sec_msb, 16)            
           
;  Extract truncated julian date           
tjd_lsb = long( dbt(*, 6) and 'FE'XB)           
tjd_msb = long( dbt(*, 7) and '7F'XB)           
clk_words(0, 0) = ishft( tjd_lsb, -1) + ishft( tjd_msb, 7)            
          
;  Output tranposed array          
clk_words = transpose( clk_words)          
           
end               
;------------------------------------------------------------------------------      
pro def_flr_strc, nbytes, strc              
         
;       Defines a structure to hold the Level Zero File Label Record         
         
; Inputs: none         
         
; Outputs:         
;       nbytes  Size of structure in bytes         
;       strc    Structure to hold data         
         
strc = {File_Label_Rec,                 $              
        spacecraft_id:  0L,             $              
        instrument_num:  0L,            $              
        instrument_nam:  '   ',         $              
        phys_rec_posit:  0L,            $              
        phys_rec_mjfrm:  0L,            $              
        phys_recs_file:  0L,            $              
        beg_mjfrm_cnt:  0L,             $              
        end_mjfrm_cnt:  0L,             $              
        beg_spc_clock:  lonarr(5),      $              
        end_spc_clock:  lonarr(5),      $              
        atc_beg:        lonarr(4),      $              
        atc_end:        lonarr(4),      $              
        mjfrm_expct:    0L,             $              
        mjfrm_file:     0L,             $              
        mjfrm_lev_gap:  0L,             $                  
        data_covage_typ: '    '         $              
}              
         
nbytes = n_tags( strc, /length)              
         
end               
;------------------------------------------------------------------------------      
pro Lz_flr, flr_strc, err              
         
;       Parses the File Label Record (FLR) from the Level Zero File         
;       Define a structure to hold a FLR            
              
; Inputs:         
;       None         
         
; Outputs:         
;       flr_strc        File label record structure         
;       err             Error string is null if no error         
         
; BLOCK: LZfile                      
;       pth         Path to LZ file including device and all directories      
;       fname       LZ filename including path info                      
;       unit        Unit number if file is open, else -1                
;       spacecraft  Name string should be one of 'POLAR'                
;       instrument  Name string should be one of ['CAMMICE','CEPPAD',              
;                   'MFE', or 'PIXIE']                
;       tm_mode     Must be one of ['SCIENCE','MANUEVER','CONTINGENCY']       
;       nbytes      Number of bytes per record                
                
common LZfile, pth, fname, unit, spacecraft, instrument, tm_mode, nbytes      
                
def_flr_strc, nbytes_flr, flr_strc            
              
;       Get instrument name read from Level Zero file header      
spacecraft_lz, name      
      
;       Return with error if not a Level Zero file      
if name eq 'UNKNOWN SPACECRAFT' then begin      
      
        err = 'LZ_flr: input data file is not level zero'      
        return      
      
endif      
      
;               Read fist data record from file in non-buffered mode             
rec_numb = 0L         
rd_LZ, rec_numb, data, err            
               
;       Test if reading error              
if err eq '' then begin              
            
;       Load structure with data              
        long_data = long( data, 0, n_elements(data)/4)               
            
;       Swap bytes if necessary for this platform              
        swap_bytes, long_data           
              
        flr_strc.spacecraft_id = long_data(0)               
        flr_strc.instrument_num = long_data(1)              
        flr_strc.instrument_nam = strtrim( string( data(8:11)), 2)               
        flr_strc.phys_rec_posit = long_data(3)               
        flr_strc.phys_rec_mjfrm = long_data(4)               
        flr_strc.phys_recs_file = long_data(5)               
        flr_strc.beg_mjfrm_cnt = long_data(6)               
        flr_strc.end_mjfrm_cnt = long_data(7)               
        flr_strc.atc_beg = long_data(12:15)               
        flr_strc.atc_end = long_data(16:19)               
        flr_strc.mjfrm_expct = long_data(20)               
        flr_strc.mjfrm_file = long_data(21)              
        flr_strc.mjfrm_lev_gap = long_data(22)               
        flr_strc.data_covage_typ = strtrim( string( data(92:95)), 2)              
          
        sc_clock, data(32:39), clk_words          
        flr_strc.beg_spc_clock = clk_words               
          
        sc_clock, data(40:47), clk_words          
        flr_strc.end_spc_clock = clk_words              
         
; ************************************************************************         
;       Get file size from fstat structure instead of header         
;       if file was made by Mac GSE program         
         
fs = fstat( unit)         
if flr_strc.mjfrm_file eq 0L then $         
        flr_strc.mjfrm_file = (fs.size / nbytes) - 1L         
         
; ************************************************************************         
         
endif              
             
end               
;------------------------------------------------------------------------------      
pro def_drh_strc, nbytes, strc              
         
;       Defines a structure to hold the Level Zero Data Record Header              
         
; Inputs: none         
         
; Outputs:         
;       nbytes  Size of structure in bytes         
;       strc    Structure to hold data         
         
strc =  { Data_Rec_Head,                $         
        inst_num:  0L,                  $         
        rec_num:  0L,                   $         
        mfrm_cnt: 0L,                   $         
        spc_clk:  lonarr(5),            $         
        absolute_time: lonarr(4),       $         
        mnfr_fil:  0L,                  $         
        mnfr_syn:  0L,                  $         
        tm_mode :  0L,                  $         
        mnfr_qua:  bytarr(250)          $         
        }              
         
nbytes = n_tags( strc, /length)              
         
end               
;------------------------------------------------------------------------------      
function Lz_drh, headr_bytes              
         
;  Parses the Data Record Headers (DRH) for any number of records      
;  from the Level Zero File      
         
; Inputs:              
;       headr_bytes     First 300 bytes of Level Zero Record containing the      
;                       Data Record Header      
      
;       Number of records to process      
nrecs = n_elements( headr_bytes) / 300L      
      
;       Reformat input bytes as records      
hdr_bytes = reform( headr_bytes, 300, nrecs)      
      
;       Define a structure to hold all the Level Zero Data Record Headers      
def_drh_strc, nbytes, struct      
strc = replicate( struct, nrecs)      
      
;       Reformat input bytes as longs      
hdr_longs = transpose( long( hdr_bytes( 0:47, *), 0, 12, nrecs))      
            
;       Swap bytes if necessary for this platform              
swap_bytes, hdr_longs        
              
strc.inst_num = hdr_longs( *, 0)              
strc.rec_num = hdr_longs( *, 1)              
strc.mfrm_cnt = hdr_longs( *, 2)              
strc.absolute_time = transpose( hdr_longs( *, 5:8))      
strc.mnfr_fil = hdr_longs( *, 9)               
strc.mnfr_syn = hdr_longs( *, 10)               
strc.tm_mode  = hdr_longs( *, 11)               
strc.mnfr_qua = hdr_bytes( 48:297, *)              
          
sc_clock, hdr_bytes(12:19, *), clk_words          
strc.spc_clk = clk_words              
      
return, strc              
end               
     
