;------------------------------------------------------------------------------ 
; File: HIT_HK.PRO   	CAMMICE HIT Housekeeping Routines
; Revision: 02/21/96    J L Roeder             
;------------------------------------------------------------------------------ 
; Routines:             
;	hpass_temp_lut	Computes Look-up table for Passive Temp Monitors 
;	hpass_temp	Convert S/C Generated Passive Temp Monitors to degrees
;	hdpu_hk_anlg    Extracts HDPU Housekeeping Analogs  
;	hdpuhk_lut	Computes look-up table for HDPU Housekeeping words 
;	conv_hdpuhk	Converts HDPU Housekeeping words to physical units 
;	rd_hhk_cal	Reads calibration factors for HIT housekeeping data
;------------------------------------------------------------------------------ 
pro hpass_temp_lut, lut

;	Convert S/C Generated Passive Temp Monitors to degrees

; Inputs: None

; Outputs:
;	lut		Look-up table for two passive temp monitors
;			dim 256 x 2

; BLOCK: hhk_cals	Calibration coefficients for:
;	hdpucal		HDPU Housekeeping words
;	hpasscal	Passive Temperature Monitors
;	rtempcal	R Temperature Coefficients
;	ftempcal	F Temperature Coefficients

common hhk_cals, hdpucal, hpasscal, rtempcal, ftempcal

;	Check if calibration file has been read
if n_elements( hpasscal) eq 0 then rd_hhk_cal

;	Start with all possible counts
counts = indgen(256)

;	Compute monitor voltages from counts for HDPU monitor
volts = (counts + hpasscal(0,0)) * hpasscal(1,0)

;	Select coefficient sets from calibrations for HDPU Temp
rcoef = rtempcal(*, 1)
fcoef = ftempcal(*, 1)

;       Compute relative temp
rt = rt_pass( rcoef, volts)

;	Convert voltage to temps
deg_hdpu = temp_cent( fcoef, rt)

;	Fill in special cases
deg_hdpu(0) = 300.
deg_hdpu(251) = replicate(-90.,5)

;	Compute monitor voltages from counts for HIT Sensor monitor
volts = (counts + hpasscal(0,1)) * hpasscal(1,1)

;	Select coefficient sets from calibrations for HIT Sensor Temp
rcoef = rtempcal(*, 2)
fcoef = ftempcal(*, 2)

;       Compute relative temp
rt = rt_pass( rcoef, volts)

;	Convert voltage to temps
deg_hsen = temp_cent( fcoef, rt)

;	Fill in special cases
deg_hsen(0) = 200.

;	Combine two tables to form one look-up array
lut = fltarr(256, 2)
lut(0, 0) = deg_hdpu
lut(0, 1) = deg_hsen

end
;------------------------------------------------------------------------------ 
pro hpass_temp, counts, degrees

;	Convert S/C Generated Passive Temp Monitors to degrees

; Inputs:
;	counts		Housekeeping data from TM Word 12 for both DPU and
;			Sensor Temperatures

; Outputs:
;	degrees		Temperature in degrees centigrade

; BLOCK: hpass_lut
;	lut		Look-up table for S/C Generated temp monitors for HIT

common hpass_lut, lut

;	Calculate look-up table on first execution
if n_elements( lut) eq 0 then begin

	hpass_temp_lut, lut0
	lut = lut0

endif

;	HDPU Temp First
;	Data must be within range
cnts = fix(counts(*, 0))
c = cnts > 0
c = c < 255

;	Look-up
degrees_hdpu = lut( c, 0)

;	Fill bad data
fill = -1.e31
jbad = where(cnts lt 0, nbad)
if nbad gt 0 then degrees_hdpu( jbad) = fill

;	Sensor Temp Second
;	Data must be within range
cnts = fix(counts(*, 1))
c = cnts > 0
c = c < 255

;	Look-up
degrees_hit = lut( c, 1)

;	Fill bad data
fill = -1.e31
jbad = where(cnts lt 0, nbad)
if nbad gt 0 then degrees_hit( jbad) = fill

;	Combine temperatures into one array
degrees = float( counts)
degrees(0, 0) = degrees_hdpu
degrees(0, 1) = degrees_hit

end
;------------------------------------------------------------------------------ 
pro hdpu_hk_anlg, t11, w11, times, housekp
 
;       Converts HDPU Housekeeping Analogs into physical units for plots 
 
; Inputs: 
; 	t11	Universal times of major frames, dim nmajor
;       w11	Raw integer array of data from TM Word 11, dim nmajor x 17
 
; Outputs: 
;	times	Universal times of housekeeping analogs, dim N
;       housekp	Housekeeping Analogs in physical units, dim 16 x N 
 
;       HIT DPU Status from loose bytes
hdpustat= (w11(*,0) and '0F'XB) * (w11(*,0) ne -1)

;       Convert only frames with DPU outputing sensor data
jgood = where( hdpustat gt 3)
if jgood(0) ne -1L then begin

        t11 = t11( jgood)
        tw11 = transpose( w11)
        tw11 = tw11( *, jgood)
        w11 = transpose( tw11)

;       Extract Subcom counter from data 
	sb = w11(*,5) 
	subctr = (sb ne -1) * (sb and '1F'X) + 255 * (sb eq -1) 
 
;       Number of points
	npts = n_elements( sb) * 2
 
;       Extract data from input array 
	dt = w11(*, 6:7)
	dt = transpose(dt)
	dt = fix( dt, 0, npts) 

;       Replicate to get time for each data point
        tt = t11( lindgen( npts) / 2L)

;       Index arrays 
;	HDPU Housekeeping has 16 channels
	blkwid = 16
	cntr2 = 2 * (subctr mod (blkwid/2))
	cntr = intarr(npts/2,2)
	cntr(0, 0) = cntr2
	cntr(0, 1) = cntr2 + 1
	cntr = fix( transpose(cntr), 0, npts)
 
;	Create index to reformat data
	build_index, cntr, blkwid, index

;       Decommutate data and times using index
        decomm_data, blkwid, index, dt, hskp
        decomm_data, blkwid, index, tt, tims

;       Reformate data and times for further processing
        times = transpose( tims)
        hskp = transpose( hskp)

;	Converts HDPU Housekeeping words to physical units for plots
	conv_hdpuhk, hskp, housekp

;	Reformat time array to give one value per set
	reform_data, -1L, times

endif else begin

        times = -1L
        housekp = replicate( -1.e31, 16)

endelse

end 
;------------------------------------------------------------------------------ 
pro hdpuhk_lut, lut

;	Computes look-up table for HDPU Housekeeping words 

; Inputs: None

; Outputs:
;	lut		Floating point look-up array of housekeeping values
;			dim 256 x 16

; BLOCK: hhk_cals	Calibration coefficients for:
;	hdpucal		HDPU Housekeeping words
;	hpasscal	Passive Temperature Monitors
;	rtempcal	R Temperature Coefficients
;	ftempcal	F Temperature Coefficients

common hhk_cals, hdpucal, hpasscal, rtempcal, ftempcal

;	Check if calibration file has been read
if n_elements( hdpucal) eq 0 then rd_hhk_cal

;	Start with all possible counts
cnts = findgen(256)

;	Monitor voltage array for all channels
nchan = 16
volts = fltarr( 256, 16)

;	Conversion algorithm for each channel
for ichan = 0, nchan - 1 do $
	volts(0, ichan) = (cnts + hdpucal(0, ichan)) * $
		hdpucal(1, ichan)	

;	Place results in look-up table
lut = volts

;	Convert HDPU temp to degrees
rcoef = rtempcal(*, 0)
fcoef = ftempcal(*, 0)
v = volts(*, 9)
degrees = norm_temp( rcoef, fcoef, v)
degrees(0) = -90.
degrees(228) = replicate(252., 28)
lut(0, 9) = degrees

end
;------------------------------------------------------------------------------ 
pro conv_hdpuhk, counts, hkpg

;	Converts HDPU Housekeeping words to physical units for plots

; Inputs:
;	counts		Integer data word array, dim 16 x nmajor/16
;			bad data flagged by -1's

; Outputs:
;	hkpg		Floating point array of housekeeping values

; BLOCK: hdpu_lut
;	lut		Look-up table for HIT DPU HK monitors 

common hdpu_lut, lut

;	Calculate look-up table on first execution
if n_elements( lut) eq 0 then begin

	hdpuhk_lut, lut0
	lut = lut0

endif

;	Size of input data array
sz = size( counts)
nsets = sz(1)
nchans = sz(2)

;	Setup output array
hkpg = fltarr( nsets, nchans)

;	Perform conversion one channel at a time
for ichan = 0, nchans - 1 do begin

;	Extract data for this channel
	cnts = counts( *, ichan)

;	Data must be within range
	cnt = fix(cnts) > 0
	cnt = cnt < 255

;	Extract look-up table for this channel
	lut_ichan = lut(*, ichan)

;	Perform look-up
	hk = lut_ichan( cnt)

;	Fill bad data
	fill = -1.e31
	jbad = where(cnts lt 0, nbad)
	if nbad gt 0 then hk( jbad) = fill

;	Transpose data array
	hkpg(0, ichan) = hk

endfor

;	Transpose to 16 x nmajor/16 for listing
hkpg = transpose( hkpg)

end
;------------------------------------------------------------------------------ 
pro rd_hhk_cal, dummy
 
;       Reads calibration factors for CAMMICE HIT Housekeeping data
;	into common block hhk_cals

; Inputs: none
 
; BLOCK: hhk_cals	Calibration coefficients for:
;	hdpucal		HDPU Housekeeping words
;	hpasscal	Passive Temperature Monitors
;	rtempcal	R Temperature Coefficients
;	ftempcal	F Temperature Coefficients

common hhk_cals, hdpucal, hpasscal, rtempcal, ftempcal

if n_elements(intp) eq 0 then intp = 0 
 
;	Get path to calibration files
path_lz, path_prog, path_data, path_cals

fname = 'hhk_cals.dat'
fn = path_cals + fname
errno = -1
while errno ne 0 do begin

	openr, unit, fn, /GET_LUN, ERROR=errno
	if errno ne 0 then begin
	
		fn = pickfile(TITLE='Find file '+fname, PATH=path_cals, GET_PATH=pth)
		path_cals = pth
		path_lz, path_prog, path_data, path_cals
		
	endif
		
endwhile		
 
;	Read and print header string in file
hdr1 = ' '
readf, unit, hdr1
print,'Reading: '+hdr1
 
;	Number of housekeeping words for HDPU
hdr = ''
readf, unit, hdr
nhk = 0
readf, unit, nhk

;	Read coefficients for HDPU HK 
hdpucal=fltarr(3, nhk) 
readf,unit, hdpucal
hdpucal = hdpucal(0:1, *)

;	Number of HIT Passive Temperature Monitors
readf, unit, hdr
npmt = 0
readf, unit, npmt

;	Read coefficients for Passive Temp Monitors
hpasscal=fltarr(2, npmt) 
readf,unit, hpasscal

;	Total number of Temp Monitors
readf, unit, hdr
ntmp = 0
readf, unit, ntmp

;	Read r coefficients for HDPU temp mons
rtempcal=fltarr(2, ntmp) 
readf,unit, rtempcal

;	Total number of Temp Monitors
readf, unit, hdr
ntmp = 0
readf, unit, ntmp

;	Read f coefficients for HDPU temp mons
ftempcal=fltarr(4, ntmp) 
readf,unit, ftempcal

end 
