;------------------------------------------------------------------------------hset
; File: HIT_LZ.PRO	CAMMICE HIT Level Zero File Parsing Routines           
; Revision: 08/22/96    J L Roeder           
;------------------------------------------------------------------------------
; Routines:           
;	get_htmaster		Major frame counter and master frame start time 
;	get_htimes		Extract HIT sector period and first time
;       get_hsrts_LZ    	Extract HIT sectored rate scaler counts
;	get_hxrts_LZ		Extract HIT X Scaler counts 
;	get_hyrts_LZ		Extract HIT Y Scaler counts 
;       get_hfde_LZ     	Extract HIT normal telemetry direct events
;       get_hcde_LZ     	Extract HIT cross-mode direct Events     
;       get_hmatrx_LZ   	Extract HIT matrix scalers from major frame    
;       get_hsect_LZ    	Extract HIT sector information          
;       get_hvalid_LZ   	Extract HIT sector validity bits          
;       get_hmag_LZ      	Extract HIT MagAz and MagEl          
;       get_htotrts_LZ  	Extract HIT spin-averaged rates counts
;	get_hsing_LZ		Extract HIT single sector singles counts
;	get_hmux_LZ		Extract all HIT multiplex data 
;	get_htmstat_LZ		Extract HIT DPU TM status only 
;	get_hxystat_LZ		Extract HIT X and Y scaler type and number 
;	get_hword11_LZ		Extract HDPU Status from Word 11
;	get_hword15_LZ		Extract HIT Status from Word 15
;------------------------------------------------------------------------------ 
pro get_htmaster, mfrm, tmaster, mframe, majort

;	Obtain major frame counter and start time of master frame

; Inputs: none

; Outputs:
;	mfrm		HIT DPU Major Frame Counter, 0-10
;	tmaster		UT of start of first sector of present master frame

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
    
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
    
;	Major and minor frame period in long MICROseconds
tmajor = 9200000L
tminor = tmajor / 250L

;       Get minor frame quality bytes from header  
qual = drh.mnfr_qua  
  
;      Extract major frame counter    
if qual(61) eq 0B then mfrm = subcom(61, 0) and 15B else mfrm = 255B
    
;	Minor frame offset of master period
if qual(66) eq 0B then mframe = subcom( 66, 0) else mframe = 255B

;	Time of start of first sector in this master frame, millisec
if (mfrm ne 255B) and (mframe ne 255B) then $
	tmaster = drh.absolute_time(2) - (tmajor*mfrm + $
		tminor*(250 - mframe)) / 1000L else tmaster = -1L

majort = drh.absolute_time(2)

end
;------------------------------------------------------------------------------
pro get_htimes, mfrm, tmaster, tfirst, tsector

;	Obtain sector period and time of the first sector in the present 
;	major frame, and start time of master frame

; Inputs: none

; Outputs:
;	mfrm		HIT DPU Major Frame Counter, 0-10
;	tmaster		UT of start of first sector of present master frame
;	tfirst		UT of center of first sector in this major frame
;			in long integer milliseconds
;	tsector		Length of sector in long MICROseconds
;			Spin Period / 16 in spin sync mode,
;			Minor Frame Period * 10 in TM sync mode

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
    
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
    
;	Time of start of first sector in this master frame, millisec
;	and CAMMICE HIT Major Frame Counter, 0-10
get_htmaster, mfrm, tmaster

;       HIT TM status from two sources 
;       word 15 mf 61, and word 11 mf 8
;	note that the word 11 byte is in the lower half, different than MICS
tm1 = ishft( subcom( 61, 0) and 'F0'XB, -4)
tm2 = hk( 8, 0) and '0F'XB

;	HIT DPU synchronization mode, 0=Spin, 1=PCM from three sources
;       sm1 from TM Status in word 15 mf 61,
;       sm2 from HDPU Status in word 11 mf 8, and
;       sm3 from then SM bit in word 15 mf 64
sm1 = ishft( subcom( 61, 0) and '10'XB, -4)
sm2 = hk( 8, 0) and '01'XB
sm3 = ishft( subcom( 64, 0) and '90'XB, -7)

;       DPU must be in sensor data mode and all three of these 
;	bits must be zero to go into spin sync mode
if (tm1 gt 3B) and (tm2 gt 3B) and (sm1 eq 0B) and (sm2 eq 0B) and $
	(sm3 eq 0B) then sync_mode = 0B else sync_mode = 1B

;	Sector period tied to spin only in spin syncronized mode
if sync_mode eq 0 then begin

;	Spacecraft spin period in floating point MICROseconds from HDPU
;	housekeeping subcom or spacecraft attitude file
	hspin_period, tmaster, tspin

;	Sixteen sectors per spin
	tsector = tspin / 16L

endif else tsector = 368000L		; ten minor frame periods

;	Time of center of first sector in the this major frame, millisec
if (tmaster ne -1L) and (tsector ne -1L) then $
	tfirst = tmaster + long( tsector * (25 * mfrm + 0.5)) / 1000L $
	else tfirst = -1L

end
;------------------------------------------------------------------------------
pro get_hsrts_LZ, ichannels, cnts  
          
;       Extract HIT Sectored Rate Scaler Compressed Counts from Level
;       Zero data arrays
  
; Inputs:          
;       ichannels       Array of rate scaler channel numbers for sectored 
;                       rates only, 0-11 for HR0-11, 12-17 for HID0,2,5-8
; Outputs:          
;       cnts            Compressed accumulated counts          
   
; BLOCK: camLZ_data           
;       recnumb         Record number of data           
;       drh             Data record header           
;       hk              Housekeeping bytes dim 250 x 2          
;       subcom          Subcom (Loose) bytes dim 250 x 2          
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15
   
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
   
;               Get Validity bits
get_hvalid_LZ, valid

;	Reformat for HIT Telemetry, 10 mf / sector
nval = n_elements(valid)
indx = lindgen( nval)
keven = where( indx mod 2 eq 0, nkeven)
kodd = where( indx mod 2 ne 0, nkodd)
if nkeven ne 0 then valid_even = valid( keven) else valid_even = 255B
if nkodd ne 0 then valid_odd = valid( kodd) else valid_odd = 255B

;	Determine frames with both valid even and odd sectors
jeven = where( valid_even ne 0B, neven)
jodd = where( valid_odd ne 0B, nodd)
nvalid = min( nodd, neven)
if nvalid gt 0 then jvalid = jodd( 0:nvalid-1)

;       Number of channels to extract          
nchan = n_elements( ichannels)

;	Check if any valid sectors in this frame
if nvalid ne 0L then begin

;       Setup output array
	cnts = bytarr( nvalid, nchan)

;	Reformat fixed TM words to 10 minor frames
	fixed_hit = transpose( hit_fixed)
	fixed_hit = byte( fixed_hit, 0, 30, 25)
	fixed_hit = transpose( fixed_hit)

;       Index array for channels
	index = [indgen(6)+6, indgen(6)+21, indgen(3)+3, indgen(3)+18]
   
;       Extract channels one at a time          
	for jchan = 0, nchan-1 do $
                cnts( 0, jchan) = fixed_hit( jvalid, index( ichannels(jchan)))
 
;      Ouput array is dimensioned nchannels x nsectors/2
	cnts = transpose( cnts) 

endif else cnts = replicate( 255B, nchan, 1)
         
end          
;------------------------------------------------------------------------------
pro get_hxrts_LZ, cnts
          
;       Extract HIT X Scaler Compressed Counts from Level Zero data arrays
  
; Inputs: None

; Outputs:          
;       cnts            Compressed accumulated counts          
   
; BLOCK: camLZ_data           
;       recnumb         Record number of data           
;       drh             Data record header           
;       hk              Housekeeping bytes dim 250 x 2          
;       subcom          Subcom (Loose) bytes dim 250 x 2          
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15
   
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
   
;               Get Validity bits
get_hvalid_LZ, valid

;	Reformat for HIT Telemetry, 10 mf / sector
nval = n_elements(valid)
indx = lindgen( nval)
keven = where( indx mod 2 eq 0, nkeven)
kodd = where( indx mod 2 ne 0, nkodd)
if nkeven ne 0 then valid_even = valid( keven) else valid_even = 255B
if nkodd ne 0 then valid_odd = valid( kodd) else valid_odd = 255B

;	Validity bits for all 32 sectors
jvalid = where( valid ne 0B, nvalid)

;	Determine valid frames for even and odd sectors
jeven = where( valid_even ne 0B, neven)
jodd = where( valid_odd ne 0B, nodd)
nsect = min( [neven, nodd])

;	Check if any valid sectors in this frame
if nsect ne 0L then begin

;	Flatten frame numbers to be equal even and odd sectors
	jeven = jeven(0:nsect-1)

;       Setup output array
	counts = bytarr( nsect, 2)

;	Reformat fixed TM words to 10 minor frames
	fixed_hit = transpose( hit_fixed)
	fixed_hit = byte( fixed_hit, 0, 30, 25)
	fixed_hit = transpose( fixed_hit)

;       Indices for xscaler nibbles are 0 and 16
;       Extract channels one at a time          
	counts( 0, 0) = fixed_hit( jeven, 0) and '0F'XB
	counts( 0, 1) = fixed_hit( jeven, 16) and '0F'XB

;	Combine nibbles into bytes
	cnts = byte( 16 *  counts( *, 1) + counts(*, 0))

endif else cnts = 255B
         
end          
;------------------------------------------------------------------------------
pro get_hyrts_LZ, cnts
          
;       Extract HIT Y Scaler Compressed Counts from Level Zero data arrays
  
; Inputs: None

; Outputs:          
;       cnts            Compressed accumulated counts          
   
; BLOCK: camLZ_data           
;       recnumb         Record number of data           
;       drh             Data record header           
;       hk              Housekeeping bytes dim 250 x 2          
;       subcom          Subcom (Loose) bytes dim 250 x 2          
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15
   
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
   
;               Get Validity bits
get_hvalid_LZ, valid

;	Validity bits for all 32 sectors
jvalid = where( valid ne 0B, nvalid)

;	Check if any valid sectors in this frame
if nvalid ne 0L then begin

;       Setup output array
	counts = bytarr( nvalid, 2)

;	Indices for yscaler nibbles are 1 and 2
;	Extract channels one at a time from mics arrays
	counts( 0, 0) = mics_fixed( jvalid, 2) and 'F0'XB
	counts( 0, 1) = mics_fixed( jvalid, 1) and 'F0'XB

;	Combine nibbles into bytes
	cnts = byte( 16 *  counts( *, 1) + counts(*, 0))

endif else cnts = 255B

end          
;------------------------------------------------------------------------------
pro get_hfde_LZ, deng, eng, eprime, sample

;       Extract HIT Direct Events from HIT Fixed-Column data array only 
;       This does not include those events in the MICS Fixed-Column array
;       when in Cross mode          
   
; Inputs: none          
   
; Outputs:          
;       deng		Energy Loss in counts, 0-1023
;       eng             Energy in counts, 0-1023
;	eprime		Energy prime in counts, 0-255
;	sample		Time slice, 0-15
   
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 
        
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
        
;       Extract energy loss least significant bits
deng_lsb = hit_fixed( *, 1)

;	Extract most significant two bits of energy loss
deng_msb = hit_fixed( *, 12) and 12B

;	Combine lower and upper parts of energy loss into int
deng = fix( deng_lsb) + 64 * deng_msb

;       Extract energy least significant bits
eng_lsb = hit_fixed(*, 2)          
        
;	Extract most significant two bits of energy
eng_msb = hit_fixed( *, 12) and 3B

;	Combine lower and upper parts of energy into int
eng = fix( eng_lsb) + 64 * eng_msb

;	Extract sample time slice
sample = ishft( hit_fixed( *, 12) and 'F0'XB, -4)

;	Extract energy prime values
eprime = hit_fixed( *, 13)

;               Get Validity bits  
get_hvalid_LZ, valid  
jvalid = where( valid ne 0B)  
  
;               Truncate data at end of master frame  
if jvalid(0) ne -1 then begin  
        deng = deng( jvalid)  
        eng = eng( jvalid)
	sample = sample( jvalid)
	eprime = eprime( jvalid)
endif else begin  
        deng = -1  
        eng = -1
	sample = 255B
	eprime = 255B
endelse  
  
end           
;------------------------------------------------------------------------------
pro get_hcde_LZ, deng, eng, eprime, sample, mgrp
     
;       Extract HIT Direct Events from MICS Fixed-Column data array only
;       when in Cross mode. Does not include events from HIT Fixed-     
;       Column array     
     
; Inputs: none          
   
; Outputs:          
;       deng		Energy Loss in counts, 0-1023
;       eng             Energy in counts, 0-1023
;	eprime		Energy prime in counts, 0-255
;	sample		Time slice, 0-15
;	mgrp		Mass group number     
     
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 
        
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
        
;               Get Validity bits  
get_hvalid_LZ, valid  
jvalid = where( valid eq 1B, nvalid)  
  
;	Check if any valid events
if nvalid gt 0 then begin

;       Extract least significant eight bits of energy loss
	deng_lsb = mics_fixed( *, 3:5)     
     
;       Extract most significant two bits of energy loss
	deng_msb = mics_fixed( *, 9:11) and 3B     
        
;       Combine lower and upper parts of energy loss into int          
	deng = fix( deng_lsb) + 256 * deng_msb          
        
;       Extract lower significant bytes of energy
	eng_lsb = mics_fixed( *, 6:8)     
     
;       Extract most significant two bits of energy
	eng_msb = mics_fixed(*, 9:11) and 12B          
           
;       Combine lower and upper parts of TOF into int          
	eng = fix( eng_lsb) + 64 * eng_msb          
     
;       Extract the mass group numbers     
	mgrp = mics_fixed( *, 0:2) and 15B
     
;	Extract the sample time slices
	sample = ishft( mics_fixed( *, 9:11) and 'F0'XB, -4)

;	Extract the Energy primes
	eprime = mics_fixed( *, 12:14)

;       Replicate Validity bits  by 3
	valid3 = bytarr( 50, 3)
	for i=0,2 do valid3(0, i) = valid
	valid3 = reform( valid3, 150)

;	Reformat events to be one dimensional
	deng = reform( deng, 150)
	eng = reform( eng, 150)
	eprime = reform( eprime, 150)
	sample = reform( sample, 150)
	mgrp = reform( mgrp, 150)

;	Eliminate invalid events
	jvalid = where( valid3 ne 0B, nvalid3)  

	deng = deng( jvalid)
	eng = eng( jvalid)
	eprime = eprime( jvalid)
	sample = sample( jvalid)
	mgrp = mgrp( jvalid)

;	Reformat arrays to be dimension nvalid x 3
	kvalid = where( valid ne 0B, nvalid)  
	deng = reform( deng, nvalid, 3)
	eng = reform( eng, nvalid, 3)
	eprime = reform( eprime, nvalid, 3)
	sample = reform( sample, nvalid, 3)
	mgrp = reform( mgrp, nvalid, 3)

endif  else begin

	deng = -1
	eng = -1
	eprime = 255B
	sample = 255B
	mgrp = 255B

endelse
  
end           
;------------------------------------------------------------------------------
pro get_hmatrx_LZ, mfc, hms    
    
;      Extracts the contents of the HIT matrix scalers from major frame    
;      arrays    
    
; Inputs: None    
    
; Outputs:    
;      mfc         Major frame counter, 0-10     
;      hms         HIT Matrix scaler contents, compressed
    
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk 		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 
        
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
        
;       Get minor frame quality bytes from header  
qual = drh.mnfr_qua  
  
;      Extract major frame counter    
if qual(0) eq 0B then mfc = (subcom(61, 0) and 15B) else mfc = 255B
    
;	Check if major frame counter is good
if mfc ne 255B then begin

;	Setup output array
	hms = bytarr( 48)

;	Extract matrix scaler contents from word 15
	hms(0) = subcom( 74:89, 0) * (qual(74:89) eq 0B) + $
		255B * (qual(74:89) ne 0B)   
    
;	Extract matrix scaler contents from word 16
	hms(16) = subcom( 210:241, 1) * (qual(210:241) eq 0B) + $
		255B * (qual(210:241) ne 0B)   
    
;	Truncate array if end of experiment master frame    
	if mfc eq 10B then hms = hms(0:31)    

endif else hms = 255B
    
end     
;------------------------------------------------------------------------------
pro get_hsect_LZ, times, sector          
        
;       Extract HIT sector information from Level Zero record
;	Note that this routine returns all 32 values per spin, instead of
;	the 16 unique values per spin.  Divide by 2 to obtain unique values
        
; Inputs: None   
          
; Outputs:        
;       times   	Universal times of center of each sector, msec
;	sector		Sector identification number, 0-31,   
;			set to 255 if invalid or filled  
        
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 
        
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
        
;      Get minor frame quality bits for first and last minor frame in sector
qs = transpose( reform( drh.mnfr_qua, 5, 50))  
qual_sect0 = qs( *, 0)  
qual_sect4 = qs( *, 4)  
             
;               Get Validity bits  
get_hvalid_LZ, valid  
jinvalid = where( valid eq 0B)  
  
;	No valid sectors?
jvalid = where( valid ne 0B, nvalid)
if nvalid eq 0L then begin
	times = -1L
	sector = 255B
	return
endif

;       Extract lower significant three bits of sector number          
sect_lsb = ishft( hit_fixed(*, 0) and 112B, -4)
sect_lsb = sect_lsb * (qual_sect0 eq 0B) + 255B * (qual_sect0 ne 0B)
if jinvalid(0) ne -1 then sect_lsb( jinvalid) = 255B
       
;       Find index where the lsb are zero, this is where the msb may          
;       be accessed          
jsect0 = where((sect_lsb eq 0B) and (valid ne 0B))
njsect0 = n_elements( jsect0)      
        
;       Check if any good data found  
if jsect0(0) ne -1L then begin  
  
;       Extract upper two bits of sector number from MUX data          
        mux = hit_fixed(*, 14)          
        sect_msb = ishft( mux( jsect0) and 192B, -3)
	sect_msb = sect_msb * (qual_sect4(jsect0) eq 0B) + $
		255B * (qual_sect4(jsect0) ne 0B)
        
;       Replicate to get msb for each sector separately
	nsm = (njsect0 * 8) < (50 - jsect0(0))
        msbs = sect_msb( lindgen( nsm) / 8L)
	loc_msb = (lindgen( nsm) mod 8) + jsect0( lindgen( nsm) / 8L)
	sector_msb = replicate( 255B, 50)
	sector_msb( loc_msb) = msbs
        
;       Combine lower and upper parts of sector number into int          
        sector = sect_lsb + sector_msb
	sector = sector * ((sect_lsb ne 255B) and (sector_msb ne 255B)) + $
		255B * ((sect_lsb eq 255B) or (sector_msb eq 255B))
  
;       Set the first few sector msb's to the first sector - 1          
        if jsect0(0) ne 0 then begin  
  
                jfirst_sect = jsect0(0)  
                first_sect = sector( jfirst_sect)        
                last_sect = first_sect - 1B        
                if last_sect lt 0 then last_sect = 31B         
                sector(0) = (indgen( jfirst_sect) + last_sect $  
                        - jfirst_sect + 1)  mod 32B       
  
        endif  
  
endif else sector = replicate( 255B, 50)  
         
;	Any valid sectors
if nvalid ne 0L then begin

;	Eliminate invalid sectors
	sector = sector( jvalid)

;       Obtain sector interval and first time
	get_htimes, mfrm, tmaster, tfirst, tsector

;       Compute time of accumulation for each valid sector
	if (tfirst ne -1L) and (tsector ne -1L) then begin

	   times = ((lindgen(nvalid)/2*tsector) / 1000L + tfirst) * $
		(sector ne 255B) - long( sector eq 255B)

	endif else times = replicate( -1L, nvalid)

endif else begin

	sector = 255B
	times = -1L

endelse

end           
;------------------------------------------------------------------------------
pro get_hvalid_LZ, valid          
        
;       Extract HIT validity bits from Level Zero record          
;	Note that all 32 values per spin are returned, not just the even
;	or odd sector values
        
; Inputs: none          
        
; Outputs:          
;       valid   Valid data bit is zero if sector data is invalid          
        
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 
        
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
        
;	Quality flags for first minor frame of every sector
qual = transpose( reform( drh.mnfr_qua, 5, 50))
qual = qual( *, 0)

;       Extract most significant bit and right shift          
valid = ishft( hit_fixed(*, 0) and 128B, -7)

;	If quality is bad then assume data is valid
valid = valid * byte( qual eq 0B) + byte( qual ne 0B)

end           
;------------------------------------------------------------------------------
pro get_hmag_LZ, sector, magaz, magel          
        
;       Extract HIT MagAz and MagEl from Level Zero record          
        
; Inputs:           
;       sector  Sector identification number, 0-31 to index MUX data          
        
; Outputs:          
;       magaz   Magnetic field azimuth          
;       magel   Magnetic field elevation          
        
; BLOCK: camLZ_data           
;       recnumb     Record number of data           
;       drh         Data record header           
;       hk              Housekeeping bytes dim 250 x 2          
;       subcom  Subcom (Loose) bytes dim 250 x 2          
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
        
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
        
;       Extract MUX data from fixed-column data          
mux = hit_fixed(*, 14)          
        
;       Indexes for positions in MUX data          
index_az = 17          
index_el = 1          
        
;       Find data for those sectors in this major frame          
jmagaz = where( sector eq index_az)          
jmagel = where( sector eq index_el)          
        
;       Extract the MUX data for these positions          
if jmagaz(0) ne -1 then magaz = mux( jmagaz) else magaz = -1B         
if jmagel(0) ne -1 then magel = mux( jmagel) else magel = -1B          
        
end           
;------------------------------------------------------------------------------
pro get_htotrts_LZ, ichannels, sector, times, chans, cnts
        
;       Extract HIT non-sectored spin-averaged rate scaler compressed counts
;       from a Level Zero record          
        
; Inputs:           
;	ichannels	Array of rate scaler channel numbers for non-sectored
;			spin-averaged rates in telemetry order
;			0 = HR12 Total		1 = HID4 Total
;			2 = HS1 Total		3 = HS5 Total
;			4 = HID1 Total		5 = HR13 Total
;			6 = HID9 Total		7 = HS2 Total
;			8 = HS6 Total		9 = HR14 Total
;			10 = HID10 Total	11 = HS3 Total
;			12 = HID3 Total		13 = HR15 Total
;			14 = HID12 Total	15 = HS4 Total
;	sector		Sector identification number, 0-31
        
; Outputs:          
;	times		Universal times of spin centers, long msec
;	chans		Channel identification numbers          
;	cnts		Compressed accumulated counts
        
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 
        
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          
        
common last_htot,       $ ; BLOCK for selected items from previous record
        lstrec,         $ ; record number of previous record accessed
        lstmstr,        $ ; Start time of master frame for previous record
        maxlstspin        ; Last spin processed in last record

;       Initialize common block first time through
if n_elements( lstrec) eq 0L then begin

        lstrec = -99L
        lstmstr = -1L
        maxlstspin = 0L

endif

;       Quality flags for last minor frame in every sector
qual_mux = transpose( reform( drh.mnfr_qua, 5, 50))
qual_mux = qual_mux( *, 4)

;       Extract MUX data from fixed-column data          
mux = hit_fixed(*, 14) * (qual_mux eq 0B) + 255B * (qual_mux ne 0B)
        
;       Locations of spin-averaged rates in MUX array          
sect_index = [4, 5, 6, 7, 9, 12, 13, 14, 15, 20, 21, 22, 25, 28, 29, 30]
        
;       Setup channel id array          
chans = replicate( -1B, 50)          
        
;       How many channels to extract          
nchans = n_elements( ichannels)          
        
;       Mark data one channel at a time          
for ichan = 0, nchans - 1 do begin          
        jchan = ichannels( ichan)          
        jindex = where( sector eq sect_index( jchan))          
        if jindex(0) ne -1 then chans( jindex) = jchan          
endfor          
        
;       Which data were marked?          
kchans = where( chans ne 255B,  nkchans)
if nkchans ne 0L then begin          
        
;       Extract the data which were marked          
        chans = chans( kchans)          
        cnts = mux( kchans)          

;	Obtain sector interval and first time
	get_htimes, mfrm, tmaster, tfirst, tsector

;	Spin index for marked data
	kspins = (kchans + mfrm*50) / 32

;	Universal times of spin centers delayed by one spin
;	note 16 sectors/spin instead of 32
	times = tmaster + long( 16L * tsector * (kspins - 0.5)) / 1000L
	times = times * ((tmaster ne -1L) and (tsector ne -1L)) - $
		long((tmaster eq -1L) or (tsector eq -1L)) 

;       At beginning of master must calculate time with previous master time
        zspins = where( kspins lt 1L, nzspins)
        if nzspins gt 0L then begin

                if (recnumb eq lstrec+1) and (lstmstr ne -1L) then $
			times( zspins) = lstmstr + $
                        long( 16L * tsector * (maxlstspin + 0.5) ) / 1000L $
                        else times( zspins) = -1L

        endif

	maxlstspin = max( kspins)
	lstmstr = tmaster
	lstrec = recnumb

endif else begin

;       If no data marked then output -1's          
	times = -1L
        chans = 255B          
        cnts = 255B

endelse

end          
;------------------------------------------------------------------------------ 
pro get_hsing_LZ, sector, tspins, tsects, chans, data   
         
;       Extract HIT selected-sector singles rate scaler compressed counts 
;       from a Level Zero record           
         
; Inputs:            
;       sector       Sector identification number, 0-31           
         
; Outputs:
;	tspins	Universal times of spin starts, long msec
;	tsects	Sector period, long MICROSEC
;       ids	Data identification numbers, 0-9B, in telemetry order:
;		0 = HS1-A,
;		1 = HS1-B TCR,
;		2 = Sector-A
;		3 = HS2-A, 
;		4 = HS2-B,
;		5 = Sector-B,
;		6 = HS3-A, 
;		7 = HS3-B,
;		8 = HS4-A,
;		9 = HS4-B
;	data	Sectors and compressed accumulated counts, 0-255B    

common camLZ_data, 	$ ; BLOCK for raw cammice lz data
	recnumb,	$ ; Record number of data
	drh,       	$ ; Data record header
	hk, 	   	$ ; Housekeeping bytes dim 250 x 2
	subcom,    	$ ; Subcom (Loose) bytes dim 250 x 2
	mics_fixed,	$ ; Fixed-Column MICS Science Allocation, dim 50 x 15
	hit_fixed  	  ; Fixed-Column HIT Science Allocation, dim 50 x 15
         
common last_hsing,   	$ ; BLOCK for selected items from previous record
	lstrec,    	$ ; record number of previous record accessed
	lstmstr,	$ ; Start time of master frame for previous record
	maxlstspin	  ; Last spin processed in previous record

;	Initialize common block first time through
if n_elements( lstrec) eq 0L then begin

	lstrec = -99L
	lstmstr = -1L
	maxlstspin = 0L

endif

;	Quality flags for last minor frame in every sector
qual_mux = transpose( reform( drh.mnfr_qua, 5, 50))
qual_mux = qual_mux( *, 4)

;       Extract MUX data from fixed-column data           
mux = hit_fixed(*, 14) * (qual_mux eq 0B) + 255B * (qual_mux ne 0B)          
         
;       Locations of singles rates in MUX array           
sect_index = [2,3,8,10,11,16,18,19,26,27]
         
;       Setup channel id array           
chans = replicate( 255B, 50)           
         
;       Mark data one channel at a time           
for ichan = 0, 9 do begin           
        jindex = where( sector eq sect_index( ichan))           
        if jindex(0) ne -1 then chans( jindex) = ichan           
endfor           
         
;       Which data were marked?           
kchans = where( chans ne 255B, nkchans) 
if nkchans ne 0L then begin           
         
;       Extract the data which were marked           
        chans = chans( kchans)           
        data = mux( kchans)           

;	Obtain sector interval and first time
	get_htimes, mfrm, tmaster, tfirst, tsector

;	Spin index for marked data delayed by one spin
;	First spin in master frame is number zero
	kspins = (kchans + mfrm*50L) / 32L

;	Universal times of spin starts delayed by one spin
;	note 16 sectors/spin instead of 32
	tspins = tmaster + long( 16L * tsector * (kspins-1) ) / 1000L
        tspins = tspins * ((tmaster ne -1L) and (tsector ne -1L)) - $
                long((tmaster eq -1L) or (tsector eq -1L))

;	At beginning of master must calculate time with previous master time
	zspins = where( kspins lt 1L, nzspins)
	if nzspins gt 0L then begin

		if (recnumb eq lstrec+1) and (lstmstr ne -1L) then $
			tspins( zspins) = lstmstr + $
			long( 16L * tsector * maxlstspin) / 1000L else $
			tspins( zspins) = -1L

	endif

	maxlstspin = max( kspins)
	lstmstr = tmaster
	lstrec = recnumb

;	Sector periods constant for entire record
	tsects = replicate( tsector, nkchans)

endif else begin

;       If no data marked then output flags
	tspins = -1L
	tsects = -1L
        chans = 255B           
        data = 255B

endelse

end           
;------------------------------------------------------------------------------
pro get_hmux_LZ, mux_data
         
;       Extract HIT multiplex data from a Level Zero record           
         
; Inputs: None
         
; Outputs:           
;	mux_data	Data from multiplex channel in fixed column words
;			dim 50 bytes

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;      Get minor frame quality bits for last minor frame in sector
qual_sect = transpose( reform( drh.mnfr_qua, 5, 50))
qual_sect = qual_sect( *, 4)

;       Extract MUX data from fixed-column data           
mux = hit_fixed(*, 14) * (qual_sect eq 0B) + 255B * (qual_sect ne 0B)
         
;               Get Validity bits
get_hvalid_LZ, valid

;       Number of valid and invalid sectors
jvalid = where( valid ne 0B, nvalid)

;	Output only valid sectors
if nvalid ne 0 then mux_data = mux( jvalid) else mux_data = 255B

end           
;------------------------------------------------------------------------------
pro get_htmstat_LZ, tmstat
         
;       Extract HIT DPU TM status only from a Level Zero record           
         
; Inputs: None.
         
; Outputs:
;       tmstat          HDPU TM status, 0-15

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;	Minor frame quality flags
qual = drh.mnfr_qua

;	HIT DPU telemetry status and quality for word 11 mf 8
q = qual(8)
tmstat = (q eq 0B) * (hk(8, 0) and '0F'XB)

end           
;------------------------------------------------------------------------------
pro get_hxystat_LZ, xstype, xsnum, ystype, ysnum
         
;       Extract HIT X and Y scaler type and number from a Level Zero record
         
; Inputs: None.
         
; Outputs:
;       stype          	Scaler type, HR=0, HS=1, HID=2
;	snum		Scaler number, HR=0-15, HS=0-7, HID=0-10,12

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;	Minor frame quality flags
qual = drh.mnfr_qua

;	HIT X scaler id byte
imf = 67
q = qual(imf)
xsid = subcom( imf, 0)
xstype = (q eq 0B) * (xsid and 'F0'XB) + 255B * (q ne 0B)
xsnum = (q eq 0B) * (xsid and '0F'XB) + 255B * (q ne 0B)

;	HIT Y scaler id byte
imf = 68
q = qual(imf)
ysid = subcom( imf, 0)
ystype = (q eq 0B) * (ysid and 'F0'XB) + 255B * (q ne 0B)
ysnum = (q eq 0B) * (ysid and '0F'XB) + 255B * (q ne 0B)

end           
;------------------------------------------------------------------------------
pro get_hword11_LZ, time, hdpu_hk
  
;       Extract contents of Word 11 from a Level Zero Record  

; Outputs:
;	time		Universal time in long msec of the major frame
;	hdpu_hk 	Array of relevant bytes from Word 11 converted to
;			ints with bad data flagged as -1's, dim 17
  
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 

common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          

;       Universal time of this Mf from header
time = drh.absolute_time(2)

;       Get minor frame quality bytes from header  
qual = drh.mnfr_qua  
  
;       Get TM Word 11 from common block  
hk0 = hk(*, 0)  

;	Index of relevant minor frames; note re-ordering to group together
;	like items, and put multi-byte integers into VAX order
index = [8, 116, 128, 140,152, $
	164, 176, 188, $
	56, 200, 92, 80, 68, 236, 224, 212, 248]

;	Extract and re-order data and quality bytes
hk = hk0( index)
q = qual( index)

;	Flag bad minor frames with -1 in output array
hdpu_hk = fix(hk) * (q eq 0B) - fix(q ne 0B)

end  
;------------------------------------------------------------------------------
pro get_hword15_LZ, time, hdpu_stat
  
;       Extract HIT Status of Word 15 from a Level Zero Record  

; Outputs:
;	time		Universal time in long msec of the major frame
;	hdpu_stat 	Array of relevant bytes from Word 15 converted to
;			ints with bad data flagged as -1's
  
; BLOCK: camLZ_data           
;	recnumb		Record number of data           
;	drh		Data record header           
;	hk		Housekeeping bytes dim 250 x 2          
;	subcom		Subcom (Loose) bytes dim 250 x 2          
;	mics_fixed	Fixed-Column MICS Science Allocation, dim 50 x 15
;	hit_fixed	Fixed-Column HIT Science Allocation, dim 50 x 15 

common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed          

;       Universal time of this Mf from header
time = drh.absolute_time(2)

;       Get minor frame quality bytes from header  
qual = drh.mnfr_qua  
  
;       Get TM Word 11 from common block  
sb0 = subcom(*, 0)  

;	Index of relevant minor frames, 12 bytes long
index = [61, 63, 64, 65, 66, 67, 68, 70, 71, 72, 73]

;	Extract and re-order data and quality bytes
sb = sb0( index)
q = qual( index)

;	Flag bad minor frames with -1 in output array
hdpu_stat = fix(sb) * (q eq 0B) - fix(q ne 0B)

end  
