;------------------------------------------------------------------------------ 
; File: HIT_SPN.PRO    CAMMICE HIT Spin Period Routines            
; Revision: 05/15/96	J L Roeder            
;------------------------------------------------------------------------------ 
; Routines:            
;	hspin_period	Spacecraft spin period from master frame times
;	hspin_clk_init	Initialize common block for spin period average
;	hspin_avg	Average spacecraft spin period from common block data
;	next_hmaster	Obtains start time of next master frame 
;------------------------------------------------------------------------------ 
pro hspin_period, tmaster, tspin

;	Get spacecraft spin period from successive master times
;	HIT housekeeping subcom data

; Inputs: 
;	tmaster		Present HIT master time start in long milliseconds

; Outputs:
;	tspin		Spin period in long MICROseconds, nominally 6,000,000

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
    
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           

;       Get minor frame quality bytes from header
qual = drh.mnfr_qua

;       CAMMICE HIT Major Frame Counter, 0-10
if qual(61) eq 0B then mfrm = subcom( 61, 0) and '0F'XB else mfrm = 255B

;	HDPU Subcom Counter
if qual(164) eq 0B then subctr = hk(164, 0) and '1F'XB else subctr = 255B

;       Clock data is in HDPU HK A for subcom zero
if (subctr eq 0B) and (qual(176) eq 0B) then sclock = hk( 176, 0) else $
	sclock = 255B

;	Compute average spin period from master times or clock data
hspin_avg, tmaster, mfrm, sclock, tspin

end
;------------------------------------------------------------------------------ 
pro hspin_avg_init

;	Initialize common block for spin clock data

; BLOCK:  hspin_average
;	ndat		Number of valid HDPU clock data points in array
;	first_master	Flags occurence of first master frame
;	last_sclock	Value of spin clock last received
;	mstr_times	Univeral times of successive master frame starts

common hspin_average, ndat, first_master, last_sclock, mstr_times, spint

ndat = 0L
first_master = 0
last_sclock = 255B
mstr_times = replicate( -1L, 10)
if n_elements(spint) eq 0 then spint = -1L

end
;------------------------------------------------------------------------------ 
pro hspin_avg, tmaster, mfrm, sclock, tspin

;	Computes spacecraft spin period as average of HDPU clock data

; Inputs:
;	tmaster		Universal Time of present experiment master frame
;	sclock		Spin clock data to be used if no previous master

; Output:
;	tspin		Average spin period, long MICROsec, ~6 million

; BLOCK:  hspin_average
;	ndat		Number of valid HDPU clock data points in array
;	first_master	Flags occurence of first master frame
;	last_sclock	Value of spin clock last received
;	mstr_times	Univeral times of successive master frame starts

common hspin_average, ndat, first_master, last_sclock, mstr_times, spint

;	Initialize common block if first execution
if n_elements( ndat) eq 0 then hspin_avg_init

;	Spin period from spin clock
if (sclock gt 0B) and (sclock ne 255B) then begin

;	Incoming value must be good
	spinp = 1.2058624d6 / sclock
	last_sclock = sclock 

endif else begin 

;	Use last value if available in common 
	if last_sclock ne 255 then spinp = 1.2058624d6 / last_sclock else $
		spinp = 6000.

endelse

;	Difference between new master and last old master
delt = tmaster - mstr_times( (ndat-1)>0)

;	Initialize common block if time jumps more than 19 spins or less than
;	15 spins at previous spin period 
if ndat gt 0 then begin

    if delt gt 20L then begin

;	Compute delta times to test input
	if spint ne -1L then begin

;		Use last saved spin period
		dt19 = (19L * spint) / 1000L	; Eighteen spins
		dt15 = (15L * spint) / 1000L	; Fifteen spins

	endif else begin

;		Use spin clock value
		dt19 = long( 19. * spinp)
		dt15 = long( 15. * spinp)

	endelse

;	Initial common block if new master does not meet criteria
	if (delt lt 0L) or ((delt gt dt19) and (first_master eq 0)) or $
	   ((delt lt dt15) and (first_master eq 0)) then hspin_avg_init
		

    endif

endif

;	Insert only new data into array
if (delt gt 50L) or (ndat eq 0) then begin

;	If common block arrays are full then make room for another data point
	if ndat eq n_elements( mstr_times) then begin

;		Make room for another data point by shifting data down
		mstr_times = shift( mstr_times, -1)
		mstr_times(ndat-1) = -1L
		ndat = ndat - 1

	endif

;	Insert new data by adding point to tail of array
	mstr_times(ndat) = tmaster
	ndat = ndat + 1
	first_master = 0

endif

;	Search for next master frame to get period at beginning of data
if ndat eq 1 then begin

;	Start time of next master frame
	next_hmaster, mstr_times( 0), tmaster_next
	
;	Add to average array if found
	if tmaster_next ne -1L then begin

		mstr_times(ndat) = tmaster_next
		ndat = ndat + 1
		first_master = 1

	endif

endif

;	Check if enough data to compute period
if ndat gt 1 then begin

;	Experiment master frame periods in milliseconds
	times = mstr_times(0:ndat-1)
	diff = shift( times, -1) - times
	diff = diff(0:ndat-2)

;	Number of spins per master frame should be 16, 17, or 18
	nspins = fix( diff / spinp + 0.5) > 16

;	Average spin period in MICROseconds, nominally six million
	t_spin = long( total( diff / nspins) / (ndat-1) * 1000.)

;	Is old average available
	if spint ne -1L then begin

;	    New average spin period is valid if not more than 10% different 
;	    from old average
            if abs(t_spin - spint)/spint le 0.1 then tspin = t_spin else $
	    	tspin = spint

;	Must use new average
        endif else tspin = t_spin

;	Save newest value of spin period in common
	spint = tspin

;	Use spin clock value
endif else tspin = long( spinp * 1000.)

end
;------------------------------------------------------------------------------ 
pro next_hmaster, tmaster_old, tmaster_next

;	Obtains the start time of the HIT master frame after the present one

; Inputs: 
;	tmaster_old	Start time of present master frame

; Outputs:
;	tmaster_next	Start time of next master frame

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
    
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           

;	Save original record number
orig_record = recnumb

;	Output flag if data not found
tmaster_next = -1L

;	Try the next twelve records
irecord = orig_record + 1
last_record = orig_record + 13
done = 0
while (irecord le last_record) and (done eq 0) do begin

;	Unpack next record
	unpack_cam, irecord, err

;	Time of start of master frame accumulation
	get_htmaster, mfrm, tmstr

;	If different than present master then done or else try next record
	if tmstr gt tmaster_old then begin

		tmaster_next = tmstr
		done = 1

	endif else irecord = irecord + 1

endwhile

;	Restore original record to buffer
unpack_cam, orig_record, err

end
