;------------------------------------------------------------------------------ 
; File: HIT_STAT.PRO		CAMMICE HIT Status Routines             
; Revision: 02/12/96		J L Roeder             
;------------------------------------------------------------------------------ 
; Routines:             
;      def_hdpustat_strc	Create structure for HIT DPU Status
;      load_hdpustat		Loads structure with HIT status data 
;------------------------------------------------------------------------------ 
pro def_hdpustat_strc, strc, nbytes   
   
;       Define the structure to hold the HIT DPU Status bits   
   
strc = { HIT_Status,           $          
        HDPU_Status: 0B,        $ ; Status of HIT DPU from Word 11, 0-15
        Subctr: 0B,             $ ; Subframe Counter, 0-31
	Spares: bytarr(2),	$ ; Space to align on 4-byte boundaries
        VHHV_Status: 0,		$ ; Power Supply Status (S/C Generated)
        HDPU_Pwr_Status: 0,     $ ; Analog Status (S/C Generated)
        HDPU_Temp: 0.,          $ ; HDPU Temp (S/C Generated) 
        HIT_Temp: 0.,           $ ; HIT Sensor Temp (S/C Generated) 
        CMD: lonarr(2),         $ ; Last commands received 
        CMDCTR: intarr(2),      $ ; Command counter, 0-255 
        DUMP_ADDRESS: intarr(2),$ ; Memory dump addresses
        DUMP_VALUE: bytarr(2),  $ ; Memory dump values
        TM_Checksum: 0,         $ ; Checksum for previous major frame 

        TM_status: 0B,          $ ; HDPU Status from Word 15   
        MFC:    0B,             $ ; Major Frame Counter, 0-10   
        RE:  0B,                $ ; SYSR Error Flag, 0=No error   
        TE:  0B,                $ ; SYST Error Flag, 0=No error   
        FPE: 0B,                $ ; SYSR Floating Point Error Flag, 0=No error
        REVOFF: 0B,             $ ; Sector Offset   
        SM: 0B,                 $ ; TM Synchronization Mode, 0=Spin, 1=PCM   
        AUXOFF: 0B,             $ ; Fine Tuning Offset   
        SDEP:   0B,             $ ; Software Direct Event Priority Buffer, 0-3
        HDEP:   0B,             $ ; Hardware Direct Event Priority Buffer, 0-3
        RSB:    0B,             $ ; Rate Scaler Select Buffer, 0-1   
        TBL:    0B,             $ ; Mass and M/Q Table Coeff Set, 0-7   
	X_STYPE: 0B,		$ ; X scaler type, 0-2
	X_SNUM: 0B,		$ ; X scaler number, 0-15
	Y_STYPE: 0B,		$ ; Y scaler type, 0-2
	Y_SNUM: 0B,		$ ; Y scaler number, 0-15
	HIT_Instr_Status: 0,	$ ; HIT Instrument Status Bits
        Master_Frame: 0         $ ; Minor frame number of start of Master    
        }                         ; Period, 0-249   
   
nbytes = 56
   
end   
;------------------------------------------------------------------------------ 
pro load_hdpustat, w11, w15, hdpu_stat
 
;       Loads structure with HIT housekeeping data 
 
; Inputs: 
;	w11		Int array of data from TM Word 11 of LZ file 
;                       bad data set to -1's 
;	w15		Int array of data from TM Word 15 of LZ file 
;                       bad data set to -1's 
 
; Outputs: 
;	hdpu_stat	Structure filled with calibrated data 
 
;       Number of major frames to process
;	assume w11 array is nmajor x 17
sz = size( w11) 
nmajor = sz( 1) 
 
;       Check that word 15 data has same numbers of major frames
;	assume w11 array is nmajor x ??
sz = size( w15) 
nmj15 = sz( 1) 
if nmajor ne nmj15 then stop,'Error load_hdpustat: wrong number of Mf'
 
;       Define output structure 
def_hdpustat_strc, struc, nbytes 
 
;       Create output array 
hdpu_stat = replicate( struc, nmajor) 
 
;       Load data into structure array 
;       HIT DPU Status 
tmp = w11(*, 0) 
status = ( tmp ne -1) * ( tmp and '0F'X) + 255 * ( tmp eq -1) 
hdpu_stat. HDPU_Status = byte( status) 
 
;       Subframe Counter 
tmp = w11(*, 5) 
sctr = ( tmp ne -1) * ( tmp and '1F'X) + 255 * ( tmp eq -1) 
hdpu_stat.SUBCTR = byte( sctr) 
 
;       VHHV Power Status (SCG) 
hdpu_stat.VHHV_Status = w11(*, 1) 
 
;       HDPU Power Status (SCG) 
hdpu_stat.HDPU_Pwr_Status = w11(*, 2) 
 
;       HDPU Temperature (SCG) 
counts = w11(*, 3:4)
hpass_temp, counts, degrees
hdpu_stat.HDPU_Temp = degrees(*, 0)
 
;       HIT Sensor Temperature (SCG) 
hdpu_stat.HIT_Temp = degrees(*, 1)
 
;       Command Counter 
tmp = w11(*, 8:9) 
tmp = transpose( tmp) 
cmd_ctr = (tmp ne -1) * (tmp and '7F'X) - (tmp eq -1)
hdpu_stat.CMDCTR = cmd_ctr 
 
;       Last Commands 
cmds = bytarr( nmajor, 8) 
cmds( 0, 0) = w11(*, 10:12) 
cmds( 0, 4) = w11(*, 13:15) 
cmds = transpose( cmds) 
cmds = long( cmds, 0, 2, nmajor) 

;       Swap bytes of longs on some platforms
swap_bytes, cmds

;       Last Commands valid only when msb of counter byte is zero
valid_cmd = (tmp and '80'X)
hdpu_stat.CMD = (valid_cmd eq 0) * cmds

;       Dump data is output when commands not valid
hdpu_stat.DUMP_ADDRESS = (valid_cmd ne 0) * fix( cmds / 256L)
hdpu_stat.DUMP_VALUE = (valid_cmd ne 0) * byte( cmds and 'FF'XL)
 
;       Telemetry Checksum 
hdpu_stat.TM_Checksum = w11(*, 16) 

;       Load data into structure array from Word 15
;       HITDPU Status 
tmp = w15(*, 0) 
hdpu_stat.TM_status = (tmp ne -1) * ishft( tmp and 'F0'XB, -4) + $
	(tmp eq -1) * 255B
hdpu_stat.MFC = (tmp ne -1) * (tmp and '0F'XB) + (tmp eq -1) * 255
 
tmp = w15(*, 1)
hdpu_stat.RE = (tmp ne -1) * ishft( tmp and '80'XB, -7) + (tmp eq -1) * 255B
hdpu_stat.TE = (tmp ne -1) * ishft( tmp and '40'XB, -6) + (tmp eq -1) * 255B
hdpu_stat.FPE = (tmp ne -1) * ishft( tmp and '20'XB, -5) + (tmp eq -1) * 255B
hdpu_stat.REVOFF = (tmp ne -1) * (tmp and '1F'XB) + (tmp eq -1) * 255B
   
;	Synchronization Mode and Auxillary Offset
tmp = w15(*, 2)
hdpu_stat.SM = (tmp ne -1) * ishft( tmp and '80'XB, -7) + (tmp eq -1) * 255B
hdpu_stat.AUXOFF = (tmp ne -1) * (tmp and '7F'XB) + (tmp eq -1) * 255B
   
;	Table numbers
tmp = w15(*, 3)
hdpu_stat.SDEP = (tmp ne -1) * ishft( tmp and 'C0'XB, -6) + (tmp eq -1) * 255B
hdpu_stat.HDEP = (tmp ne -1) * ishft( tmp and '30'XB, -4) + (tmp eq -1) * 255B
hdpu_stat.RSB = (tmp ne -1) * ishft( tmp and '04'XB, -3) + (tmp eq -1) * 255B
hdpu_stat.TBL = (tmp ne -1) * (tmp and '07'XB) + (tmp eq -1) * 255B
   
;	Minor frame number of start of master period
hdpu_stat.Master_frame = w15(*, 4)
   
;	X Scaler Identification
tmp = w15(*, 5)
hdpu_stat.X_STYPE = (tmp ne -1) * ishft( tmp and 'F0'XB, -4) + $
	(tmp eq -1) * 255B
hdpu_stat.X_SNUM = (tmp ne -1) * (tmp and '0F'XB) + (tmp eq -1) * 255B

;	Y Scaler Identification
tmp = w15(*, 6)
hdpu_stat.Y_STYPE = (tmp ne -1) * ishft( tmp and 'F0'XB, -4) + $
	(tmp eq -1) * 255B
hdpu_stat.Y_SNUM = (tmp ne -1) * (tmp and '0F'XB) + (tmp eq -1) * 255B

;	HIT Instrument status bits
tmp = byte( w15(*, 7:8) and 'FF'X)
tmp = fix( transpose( tmp), 0, nmajor )
hdpu_stat.HIT_Instr_Status = tmp

end 
