;------------------------------------------------------------------------------
; File: INDEX.PRO       Telemetry Decommutating Routines              
; Revision: 02/20/95    J L Roeder              
;------------------------------------------------------------------------------
; Routines:              
;	consolid_blks	Consolidates blocks of identical values and length one
;       build_index     Index for decommutating data  
;	decomm_data	Decommutates data from subcom of fixed width
;	reform_data	Reformat data array to give single value/set with flag
;------------------------------------------------------------------------------
pro consolid_blks, cntr, block1, block2

;       Consolidates blocks of identical values and length one

; Inputs:
;       cntr    Values of subcom counter for given blocks
;       block1  Index of beginnings of blocks
;       block2  Index of ends of blocks

; Outputs:
;       block1 and block2 are modified in place

;       Lengths of blocks and number of blocks
blklen = block2 - block1 + 1
nblocks = n_elements( block1)

;       Working arrays
b1 = lonarr( nblocks)
b2 = b1

;       Always transfer first block
b1( 0) = block1( 0)
b2( 0) = block2( 0)
nb = 1L

;      Loop on remaining blocks
ib = 1L
while ( ib lt nblocks) do begin

       value = cntr( block1( ib))
       last_value = cntr( block2( nb - 1L))
       if (value eq last_value) and (blklen( ib) eq 1) then begin

;          Consolidate degenerate block with previous block
           b2( nb - 1) = block2( ib)

       endif else begin

;          Transfer non-degenerate block to ouput 
           b1( nb) = block1( ib)
           b2( nb) = block2( ib)
           nb = nb + 1L

       endelse
       ib = ib + 1L

endwhile

;      Copy results to input/output arrays
block1 = b1( 0:nb - 1)
block2 = b2( 0:nb - 1)

end
;------------------------------------------------------------------------------  
pro build_index, cntr, blkwid, index 
 
;       Index for blocking data  
 
; Inputs: 
;       cntr    Counter ascends from zero to blkwid-1 and restarts 
;       blkwid  Block width, usually 8 or 16 
 
; Output: 
;       index   Index for reformatting data array 
 
;       Number of points in array 
npts = n_elements( cntr) 
 
;	Check for data loss flags and fill with zeros
countr = cntr
jloss = where(countr lt 0)
if jloss(0) ne -1L then countr( jloss) = 0

;       Check for jumps in counter sequence 
wid1 = blkwid - 1 
diff = shift( countr, -1) - countr 
diff = diff(0:n_elements(diff)-2) 
 
;       Eliminate normal jumps at end of pattern 
jmp = where( (diff eq -wid1) and (countr eq wid1)) 
if jmp(0) ne -1 then diff(jmp) = 1 
 
;       Check for anomalous jumps 
jneg = where( diff ne 1) 
 
;       Any bad jumps found? 
if jneg(0) ne -1 then begin 
 
        nblocks = n_elements(jneg) + 1 
 
;       Beginning and end index for each block of data 
        blk1 = lonarr( nblocks) 
        blk2 = lonarr( nblocks) 
        for iblk = 0, nblocks - 1 do begin 
 
            if (iblk ne 0) and (iblk ne nblocks-1) then begin 
                blk1( iblk) = jneg(iblk-1)+1 
                blk2( iblk) = jneg(iblk) 
            endif 
 
;           Last block 
            if iblk eq nblocks-1 then begin 
                blk1( iblk) = jneg(iblk-1)+1 
                blk2( iblk) = npts - 1 
            endif 
 
;           First block 
            if iblk eq 0 then begin 
                blk1( iblk) = 0 
                blk2( iblk) = jneg(iblk) 
            endif 
 
        endfor  
 
;       Consolidate degenerate blocks
        consolid_blks, countr, blk1, blk2

endif else begin

;	A single block including all data
        blk1 = [0L]
	blk2 = [npts]

endelse

;       Number of blocks remaining
nblocks = n_elements( blk1)

;       Block lengths in points 
blklen = blk2 - blk1 + 1 
 
;       Construct index one block at a time 
off_ndx = lindgen( blkwid)
index = lindgen( npts)
for iblk = 0L, nblocks - 1L do begin

        value = countr( blk1( iblk))
	location = long( index( blk1( iblk)) mod blkwid)

        offset_index = (off_ndx + location) mod blkwid
        offset = long( where( offset_index eq value))

        index( blk1( iblk)) = index( blk1( iblk):*) + offset(0)

endfor

end 
;------------------------------------------------------------------------------  
pro decomm_data, blkwid, index, in_data, out_data

;	Decommutates data in subcom of width blkwid into an array
;	dimensioned blkwid x nset

; Inputs:
;	blkwid		Width of subcom in words, usually 8 or 16
;	index		Index built from subcom counter by the routine build_index
;	in_data		Input data to be decommutated

; Outputs:
;	out_data	This array is modified in place

;       Number of subcom sets in input array
nset = max(index)/blkwid
if (nset * blkwid) lt max(index) then nset = nset + 1

;	Output array must be same type as input with a value of -1
sz = size( in_data)
nsz = n_elements( sz)
intype = sz( nsz - 2)
out_data = make_array( blkwid * nset, value=-1, TYPE=intype)

;       Load good data into output arrays
out_data( index) = in_data

;       Recast times as two-d array
out_data = reform( out_data, blkwid, nset)

end
;------------------------------------------------------------------------------
pro reform_data, flag, data

;	Reformat universal time array to give a single value per set

; Input:
;	flag	Marks bad or missing data points
;	data	Data points 

; Outputs:	Data array is modified in place

;	Input array should be dimensioned as (nset, blklen ) where blklen is
;	the width of the subcom and nset is the total number of sets
sz = size( times)
nset = sz( 1)
if sz(0) gt 1 then blkwid = sz( 2) else blkwid = 1

;	First set of times from first channel
dat = data(*, 0)

;	Compare with other channels one at a time
if blkwid gt 1 then begin

    for i = 1, blkwid - 1 do begin

;	New candidate times
	dd = data(*, i)

;	Replace old times with new if new times are not negative
	dat = dat*(dd eq flag) + dd*(dd ne flag)

    endfor

endif 

;	Place results in input array for output
data = dat

end
