;------------------------------------------------------------------------------     
; File: INTERVAL.PRO    Interval routines for reading CAMMICE LZ files  
; Revision: 05/16/97    J L Roeder     
;------------------------------------------------------------------------------     
; Routines:  
;       interval_init   Initializes the input data interval to be entire file  
;       set_records     Sets the range of LZ records to be processed  
;       set_times       Sets the range of Universal Times to be processed  
;------------------------------------------------------------------------------     
pro interval_init, record_range     
     
;       Initializes the interval common block to the beginning and end of   
;       the input LZ file  
  
; Inputs: none  
  
; Outputs: 
;       record_range    Record number range, [min,max], first record is 1  
  
; BLOCK: interval  
;       times_rng       Universal time range, [min, max], in seconds  
;       rec_rng         Record number range, [min, max], with first record is 1  
  
common interval, times_rng, rec_rng     
     
;       Read File Label Record from LZ file  
LZ_flr, flr, err     
     
;       Check for possible read error  
if err eq '' then begin     
  
;       Set record range to entire file     
        rec_rng = [1, flr.mjfrm_file]  
  
;       Set time range to beginning and end times of file     
        atc1 = flr.atc_beg     
        atc2 = flr.atc_end      
        times_rng = [atc1(2), atc2(2)] / 1000L
	daynumb = atc1(1)
	year = atc1(0)

	day_date, daynumb, year, month, day
	datestr = date_str( month, day, year)
  
        fmt='("Records: ",i1,"-",i4,a15," Day: ",i3,"  UT: ",i5.5,"-",i5.5)'  
        print, rec_rng,datestr,daynumb,times_rng, format=fmt
         
        record_range = rec_rng 
     
endif else begin     
  
;       Set ranges to nonsense values     
        times_rng = [-1L, -1L]     
        rec_rng = [-1L, -1L]     
 
        record_range = rec_rng 
     
endelse     
     
end   
;------------------------------------------------------------------------------     
pro set_records, range  
  
;       Sets the range of LZ records to be processed  
  
; Inputs:  
;       range   Min and max record number, first record is 1.  
;               Value of [-1, -1] resets to beginning and end of file  
  
; BLOCK: interval  
;       times_rng       Universal time range, [min, max], in seconds  
;       rec_rng         Record number range, [min, max], with first record is 1  
  
common interval, times_rng, rec_rng     
     
;       Read File Label Record from LZ file  
LZ_flr, flr, err     
  
if err eq '' then begin  
     
        if (range(0) ge 1) and (range(1) le flr.mjfrm_file) and $  
                (range(0) le range(1)) then begin  
  
                rec_rng = range  
                times_rng = [-1, -1]  
  
        endif else print,'set_record error: invalid record range'  
          
endif else print,'set_record error: LZ file not open.'  
          
end  
;------------------------------------------------------------------------------     
pro set_times, range  
  
;       Sets the range of Universal Times to be processed  
  
; Inputs:  
;       range   Min and max universal time in seconds  
;               Value of [-1, -1] resets to beginning and end of file  
  
; BLOCK: interval  
;       times_rng       Universal time range, [min, max], in seconds  
;       rec_rng         Record number range, [min, max], with first record is 1  
  
common interval, times_rng, rec_rng     
     
if (range(0) ge 0L) and (range(1) ge 0L) and $  
        (range(0) le range(1)) then begin  
  
        times_rng = long(range)  
  
;       Search for start and stop times in file  
        search_LZ, times_rng(0), rec1, err  
        search_LZ, times_rng(1), rec2, err  
  
        if err eq '' then begin  
  
;               Validate record numbers  
                rec1 = rec1 > 1L  
                rec2 = rec2 > rec1  
  
                rec_rng = [rec1, rec2]  
                print,rec_rng, FORMAT='("Set Record Range: ",i4,"-",i4)'  
  
        endif else print,'set_times error: LZ file not open.'  
  
endif else print,'set_times error: invalid time range'  
          
end  

