;------------------------------------------------------------------------------  
; File: MICS_HK.PRO     CAMMICE MICS Housekeeping Routines              
; Revision: 02/21/96    J L Roeder              
;------------------------------------------------------------------------------  
; Routines:              
;       rt_conv         Relative temperature conversion for active monitors 
;       rt_pass         Relative temperature conversion for passive monitors 
;       ref_correct     Corrects MICS sensor monitor voltages for drift of +5V 
;       temp_cent       Converts corrected monitor voltage to temperature  
;       msensor_temp    Converts MICS sensor HK temperature monitor to degrees 
;       norm_temp       Converts normal HK temperature monitor to degrees 
;       mpass_temp_lut  Computes Look-up table for Passive Temp Monitors  
;       mpass_temp      Convert S/C Generated Passive Temp Monitors to degrees 
;       mdpu_hk_anlg    Extracts MDPU Housekeeping Analogs   
;       mdpuhk_lut      Computes look-up table for MDPU Housekeeping words  
;       conv_mdpuhk     Converts MDPU Housekeeping words to physical units  
;       mics_hk_stat    Extracts MICS Sensor status bits 
;       mics_hk_anlg    Extracts MICS Sensor Housekeeping Analogs   
;       conv_micshk     Converts MICS Sensor Housekeeping to physical units  
;       rd_mhk_cal      Reads calibration factors for MICS housekeeping data 
;       vvps_lut        Computes look-up table for VVPS monitor conversion 
;       vvps_mon        Converts VVPS monitor counts into kilovolts  
;------------------------------------------------------------------------------  
function rt_conv, coef, vin 
 
;       Relative temperature conversion function for temperature monitors 
 
vout = coef(0) * ((coef(1) / vin) - 1.) 
 
return, vout 
end 
;------------------------------------------------------------------------------  
function rt_pass, coef, vin 
 
;       Relative temperature conversion function for temperature monitors 
 
vout = coef(0) / ((coef(1) / vin) - 1.) 
 
return, vout 
end 
;------------------------------------------------------------------------------  
function ref_correct, ref5v, vin 
 
;       Corrects MICS sensor monitor voltages for drift of +5V reference 
 
; Inputs: 
;       ref5V   Calibrated value of MICS sensor +5V monitor 
;       vin     Voltage to correct 
 
;       Simply scale input voltage to ratio of reference to its correct value 
vout = vin * (5. / ref5v) 
 
return, vout 
end 
;------------------------------------------------------------------------------  
function temp_cent, coef, rt 
 
;       Converts corrected monitor voltage to temperature  
;       in degrees centigrade 
 
; Inputs: 
;       rt      Relative temperature, corresponds to variable by same name 
;               in GSE code by Patty Lew, dim npts x 2 
;       coef    Coefficients from calibration file, dim 4 
 
;       Compute temp 
temperature = coef(0) + 1./((1./coef(1)) + (alog((rt>1.e-5)/coef(2)))/coef(3)) 
 
return, temperature 
end 
;------------------------------------------------------------------------------  
function msensor_temp, monp5v, volts 
 
;       Converts MICS sensor housekeeping temperature monitor voltages to 
;       temperatures in degrees 
 
; Inputs: 
;       monp5v  +5V Uncorrected reference monitor voltage from sensor hk 
;       volts   Voltage from temperature monitor 
 
; BLOCK: mhk_cals       Calibration coefficients for: 
;       mdpucal         MDPU Housekeeping words 
;       micscal         MICS Sensor Housekeeping 
;       mpasscal        Passive Temperature Monitors 
;       rtempcal        R Temperature Coefficients 
;       ftempcal        F Temperature Coefficients 
 
common mhk_cals, mdpucal, micscal, mpasscal, rtempcal, ftempcal 
 
;       Correct monitor voltage for drifting reference 
v = volts(*, 0) 
vintermed = ref_correct( monp5v, v) 
 
;       Compute relative electroinics temp using coefficients  
rcoef = rtempcal(*, 1) 
rt = rt_conv( rcoef, vintermed) 
 
;       Compute absolute temperature in degrees 
fcoef = ftempcal(*, 1) 
degrees_elect = temp_cent( fcoef, rt) 
 
;       Insert special cases 
vcrit = 0.01888 
j128 = where( ((v lt vcrit) and (v ge 0.)), cnt) 
if cnt gt 0 then degrees_elect(j128) = replicate(-92., cnt) 
 
;       Fill bad data 
fill = -1.e31 
jbad = where((v lt 0.) or (monp5v lt 0.), nbad) 
if nbad gt 0 then degrees_elect( jbad) = fill 
 
;       Correct monitor voltage for drifting reference 
v = volts(*, 1) 
vintermed = ref_correct( monp5v, v) 
 
;       Compute relative BUPS temp using coefficients  
rcoef = rtempcal(*, 2) 
rt = rt_conv( rcoef, vintermed) 
 
;       Compute absolute temperature in degrees 
fcoef = ftempcal(*, 2) 
degrees_bups = temp_cent( fcoef, rt) 
 
;       Insert special cases 
vcrit = 0.01888 
j128 = where( ((v lt vcrit) and (v ge 0.)), cnt) 
if cnt gt 0 then degrees_bups(j128) = replicate(-92., cnt) 
 
;       Fill bad data 
fill = -1.e31 
jbad = where((v lt 0.) or (monp5v lt 0.), nbad) 
if nbad gt 0 then degrees_bups( jbad) = fill 
 
;       Combine both temps into one array 
degrees = float( volts) 
degrees(0, 0) = degrees_elect 
degrees(0, 1) = degrees_bups 
 
return, degrees 
end 
;------------------------------------------------------------------------------  
function norm_temp, rcoef, fcoef, volts 
 
;       Converts normal housekeeping temperature monitor voltages to 
;       temperatures in degrees 
 
; Inputs: 
;       rcoef   Coefficients of relative temperature function from cal file 
;       fcoef   Coefficients of log temperature function from cal file 
;       volts   Voltage from temperature monitor 
 
;       Compute relative temp 
rt = rt_conv( rcoef, volts) 
 
;       Compute absolute temperature in degrees 
degrees = temp_cent( fcoef, rt) 
 
return, degrees 
end 
;------------------------------------------------------------------------------  
pro mpass_temp_lut, lut 
 
;       Convert S/C Generated Passive Temp Monitors to degrees 
 
; Inputs: None 
 
; Outputs: 
;       lut             Look-up table for two passive temp monitors 
;                       dim 256 x 2 
 
; BLOCK: mhk_cals       Calibration coefficients for: 
;       mdpucal         MDPU Housekeeping words 
;       micscal         MICS Sensor Housekeeping 
;       mpasscal        Passive Temperature Monitors 
;       rtempcal        R Temperature Coefficients 
;       ftempcal        F Temperature Coefficients 
 
common mhk_cals, mdpucal, micscal, mpasscal, rtempcal, ftempcal 
 
;       Check if calibration file has been read 
if n_elements( mpasscal) eq 0 then rd_mhk_cal 
 
;       Start with all possible counts 
counts = indgen(256) 
 
;       Compute monitor voltages from counts for MDPU monitor 
volts = (counts + mpasscal(0,0)) * mpasscal(1,0) 
 
;       Select coefficient sets from calibrations for MDPU Temp 
rcoef = rtempcal(*, 3) 
fcoef = ftempcal(*, 3) 
 
;       Compute relative temp 
rt = rt_pass( rcoef, volts) 
 
;       Compute absolute temperature in degrees 
deg_mdpu = temp_cent( fcoef, rt) 
 
;       Fill in special cases 
deg_mdpu(0) = 300. 
deg_mdpu(251) = replicate(-90.,5) 
 
;       Compute monitor voltages from counts for MICS Sensor monitor 
volts = (counts + mpasscal(0,1)) * mpasscal(1,1) 
 
;       Select coefficient sets from calibrations for MICS Sensor Temp 
rcoef = rtempcal(*, 4) 
fcoef = ftempcal(*, 4) 
 
;       Compute relative temp 
rt = rt_pass( rcoef, volts) 
 
;       Compute absolute temperature in degrees 
deg_msen = temp_cent( fcoef, rt) 
 
;       Fill in special cases 
deg_msen(0) = 200. 
deg_msen(251) = replicate(-90.,5) 
 
;       Combine two tables to form one look-up array 
lut = fltarr(256, 2) 
lut(0, 0) = deg_mdpu 
lut(0, 1) = deg_msen 
 
end 
;------------------------------------------------------------------------------  
pro mpass_temp, counts, degrees 
 
;       Convert S/C Generated Passive Temp Monitors to degrees 
 
; Inputs: 
;       counts          Housekeeping data from TM Word 12 for both DPU and 
;                       Sensor Temperatures 
 
; Outputs: 
;       degrees         Temperature in degrees centigrade 
 
; BLOCK: mpass_lut 
;       lut             Look-up table for S/C Generated temp monitors for MICS 
 
common mpass_lut, lut 
 
;       Calculate look-up table on first execution 
if n_elements( lut) eq 0 then begin 
 
        mpass_temp_lut, lut0 
        lut = lut0 
 
endif 
 
;       MDPU Temp First 
;       Data must be within range 
cnts = fix(counts(*, 0)) 
c = cnts > 0 
c = c < 255 
 
;       Look-up 
degrees_mdpu = lut( c, 0) 
 
;       Fill bad data 
fill = -1.e31 
jbad = where(cnts lt 0, nbad) 
if nbad gt 0 then degrees_mdpu( jbad) = fill 
 
;       Sensor Temp Second 
;       Data must be within range 
cnts = fix(counts(*, 1)) 
c = cnts > 0 
c = c < 255 
 
;       Look-up 
degrees_mics = lut( c, 1) 
 
;       Fill bad data 
fill = -1.e31 
jbad = where(cnts lt 0, nbad) 
if nbad gt 0 then degrees_mics( jbad) = fill 
 
;       Combine temperatures into one array 
degrees = float( counts) 
degrees(0, 0) = degrees_mdpu 
degrees(0, 1) = degrees_mics 
 
end 
;------------------------------------------------------------------------------  
pro mdpu_hk_anlg, t12, w12, times, housekp 
  
;       Converts MDPU Housekeeping Analogs into physical units for plots  
  
; Inputs:  
;       t12     Universal times of data from wor12, dim nmajor 
;       w12     Raw integer array of data from Word 12, dim nmajor x 21  
  
; Outputs: 
;       times   Universal times of housekeeping analogs, dim N 
;       housekp Housekeeping Analogs in physical units, dim 16 x N  
  
;       MICS DPU Status from loose bytes 
mdpustat= ishft(w12(*,0) and 'F0'XB, -4) * (w12(*,0) ne -1) 
 
;       Convert only frames with DPU outputing sensor data 
jgood = where( mdpustat gt 3) 
if jgood(0) ne -1L then begin 
 
        t12 = t12( jgood) 
        tw12 = transpose( w12) 
        tw12 = tw12( *, jgood) 
        w12 = transpose( tw12) 
 
;       Extract Subcom counter from data  
        sb = w12(*,1)  
        subctr = (sb ne -1) * (sb and '1F'X) + 255 * (sb eq -1)  
  
;       Number of points 
        npts = n_elements( sb) * 2 
  
;       Extract data from input array  
        dt = w12(*, 8:9) 
        dt = transpose(dt) 
        dt = fix( dt, 0, npts)  
 
;       Replicate to get time for each data point 
        tt = t12( lindgen( npts) / 2L) 
 
;       Index arrays  
;       MDPU Housekeeping has 16 channels 
        blkwid = 16 
        cntr2 = 2 * (subctr mod (blkwid/2)) 
        cntr = intarr(npts/2,2) 
        cntr(0, 0) = cntr2 
        cntr(0, 1) = cntr2 + 1 
        cntr = fix( transpose(cntr), 0, npts) 
  
;       Create index to reformat data 
        build_index, cntr, blkwid, index 
 
;	Decommutate data and times using index
	decomm_data, blkwid, index, dt, hskp
	decomm_data, blkwid, index, tt, tims
 
;       Reformate data and times for further processing
        times = transpose( tims) 
        hskp = transpose( hskp) 
 
;       Converts MDPU Housekeeping words to physical units for plots 
        conv_mdpuhk, hskp, housekp 
 
;	Reformat time array to give one value per set
        reform_data, -1L, times

endif else begin 
 
        times = -1L 
        housekp = replicate( -1.e31, 16) 
 
endelse 
 
end  
;------------------------------------------------------------------------------  
pro mdpuhk_lut, lut 
 
;       Computes look-up table for MDPU Housekeeping words  
 
; Inputs: None 
 
; Outputs: 
;       lut             Floating point look-up array of housekeeping values 
;                       dim 256 x 16 
 
; BLOCK: mhk_cals       Calibration coefficients for: 
;       mdpucal         MDPU Housekeeping words 
;       micscal         MICS Sensor Housekeeping 
;       mpasscal        Passive Temperature Monitors 
;       rtempcal        R Temperature Coefficients 
;       ftempcal        F Temperature Coefficients 
 
common mhk_cals, mdpucal, micscal, mpasscal, rtempcal, ftempcal 
 
;       Check if calibration file has been read 
if n_elements( mdpucal) eq 0 then rd_mhk_cal 
 
;       Start with all possible counts 
cnts = findgen(256) 
 
;       Monitor voltage array for all channels 
nchan = 16 
volts = fltarr( 256, 16) 
 
;       Conversion algorithm for each channel 
for ichan = 0, nchan - 1 do $ 
        volts(0, ichan) = (cnts + mdpucal(0, ichan)) * $ 
                mdpucal(1, ichan)        
 
;       Place results in look-up table 
lut = volts 
 
;       Do special cases 
;       First the VVPS - HSKP2 Voltage 
vvps_lut, lut_kv 
lut(0, 7) = lut_kv 
 
;       Convert MDPU temp to degrees 
rcoef = rtempcal(*, 0) 
fcoef = ftempcal(*, 0) 
v = volts(*, 9) 
degrees = norm_temp( rcoef, fcoef, v) 
degrees(0) = -90. 
degrees(228) = replicate(252., 28) 
lut(0, 9) = degrees 
 
end 
;------------------------------------------------------------------------------  
pro conv_mdpuhk, counts, hkpg 
 
;       Converts MDPU Housekeeping words to physical units for plots 
 
; Inputs: 
;       counts          Integer data word array, dim 16 x nmajor/16 
;                       bad data flagged by -1's 
 
; Outputs: 
;       hkpg            Floating point array of housekeeping values 
 
; BLOCK: mdpu_lut 
;       lut             Look-up table for MICS DPU HK monitors  
 
common mdpu_lut, lut 
 
;       Calculate look-up table on first execution 
if n_elements( lut) eq 0 then begin 
 
        mdpuhk_lut, lut0 
        lut = lut0 
 
endif 
 
;       Size of input data array 
sz = size( counts) 
nsets = sz(1) 
nchans = sz(2) 
 
;       Setup output array 
hkpg = fltarr( nsets, nchans) 
 
;       Perform conversion one channel at a time 
for ichan = 0, nchans - 1 do begin 
 
;       Extract data for this channel 
        cnts = counts( *, ichan) 
 
;       Data must be within range 
        cnt = fix(cnts) > 0 
        cnt = cnt < 255 
 
;       Extract look-up table for this channel 
        lut_ichan = lut(*, ichan) 
 
;       Perform look-up 
        hk = lut_ichan( cnt) 
 
;       Fill bad data 
        fill = -1.e31 
        jbad = where(cnts lt 0, nbad) 
        if nbad gt 0 then hk( jbad) = fill 
 
;       Transpose data array 
        hkpg(0, ichan) = hk 
 
endfor 
 
;       Transpose to 16 x nmajor/16 for listing 
hkpg = transpose( hkpg) 
 
end 
;------------------------------------------------------------------------------  
pro mics_hk_stat, t12, w12, times, status 
  
;       Extracts MICS Sensor status bits from subcom 
  
; Inputs:  
;       t12     Universal times of data from wor12, dim nmajor 
;       w12     Raw integer array of data from Word 12, dim nmajor x 21  
  
; Outputs:  
;       times   Universal times of sensor status words 
;       status  Long integer array of status data from MICS Sensor 
  
;       Extract Subcom counter from data  
sb = w12(*,1)  
subctr = (sb ne -1) * (sb and '1F'X) - (sb eq -1)  
  
;       Number of possible housekeeping points 
npts = n_elements( sb)  
  
;       Extract data from input array  
dt = byte( w12(*, 10:11)) 
 
;       Replicate to get time for each data point 
tt = t12( lindgen( npts) / 2L) 
 
;       Check if any bad data 
jbad0 = where( w12(*, 10) eq -1, cnt0) 
jbad1 = where( w12(*, 11) eq -1, cnt1) 
flag_bad = bytarr( npts) 
if cnt0 ne 0 then flag_bad( jbad0) = 1B 
if cnt1 ne 0 then flag_bad( jbad1) = 1B 
 
;       Use both low and high bytes 
dt1 = fix( transpose( dt), 0, npts) 
  
;       Subcom counter for sensor data 
blkwid = 16             ; MICS Subcom is 16 channels wide for status 
cntr = subctr mod blkwid 
jminus = where( subctr eq -1) 
if jminus(0) ne -1 then cntr(jminus) = -1 
 
;       Create index to reformat data 
build_index, cntr, blkwid, index 
 
;	Decommutate data and times using index
decomm_data, blkwid, index, dt1, hskp
decomm_data, blkwid, index, tt, tims
decomm_data, blkwid, index, flag_bad, flag
 
;       Re-arrange time, data, and flag array to be dim nset x blkwid 
tims = transpose( tims) 
hskp = transpose( hskp) 
flag = transpose( flag) 
 
;       Separate status words from analog housekeeping in time, data, and flags 
times = tims( *, 0) 
 
;	Number of housekeeping sets
sz= size( hskp)
nset = sz(1)
status = intarr(nset, 2) 
status(0, 0) = hskp( *, 0) 
status(0, 1) = hskp( *, 8) 
 
flg = bytarr(nset, 2) 
flg(0, 0) = flag(*, 0) 
flg(0, 1) = flag(*, 8) 
 
;       Reformat status into long words and flags into ints 
status = transpose( status) 
status = long( status, 0, nset) 
flg = transpose( flg) 
flg = fix( flg, 0, nset) 
 
;       Test flags for bad data and edit status if needed 
jflg = where( flg ne 0) 
if jflg(0) ne -1 then status( jflg) = -1L 
 
end  
;------------------------------------------------------------------------------  
pro mics_hk_anlg, t12, w12, times, housekp 
  
;       Converts MICS Sensor Housekeeping Analogs into physical units  
  
; Inputs:  
;       t12     Universal times of data from wor12, dim nmajor 
;       w12     Raw integer array of data from Word 12, dim nmajor x 21  
  
; Outputs:  
;       housekp Housekeeping Analogs in physical units, dim 7 x N  
  
;       MICS DPU Status from loose bytes 
mdpustat= ishft(w12(*,0) and 'F0'XB, -4) * (w12(*,0) ne -1) - (w12(*,0) eq -1) 
 
;       Convert only data with DPU on 
jgood = where( mdpustat gt 0, ngood) 
if ngood gt 0L then begin 
 
        t12 = t12( jgood) 
        tw12 = transpose( w12) 
        tw12 = tw12( *, jgood) 
        w12 = transpose( tw12) 
 
;       Extract Subcom counter from data  
        sb = w12(*,1)  
        subctr = (sb ne -1) * (sb and '1F'X) - (sb eq -1)  
  
;       Number of possible housekeeping points 
        npts = n_elements( sb) * 2 
  
;       Extract data from input array  
        dt = w12(*, 10:11)  
 
;       Use only the high order byte 
        dt1 = dt(*, 0) 
  
;       Subcom counter for sensor data 
        blkwid = 8              ; MICS Subcom is 8 channels wide 
        cntr = subctr mod blkwid 
        jminus = where( subctr eq -1) 
        if jminus(0) ne -1 then cntr(jminus) = -1 
 
;       Create index to reformat data 
        build_index, cntr, blkwid, index 
 
;	Decommutate data and times using index
	decomm_data, blkwid, index, dt1, hskp
	decomm_data, blkwid, index, t12, tt
 
;       Re-arrange data array to be dim nset x blkwid 
        hskp = transpose( hskp) 
        times = transpose( tt) 
 
;       Reformat time array to give one value per set 
	reform_data, -1L, times
 
;       Converts MICS Sensor housekeeping words to physical units for plots 
        conv_micshk, hskp, housekp 
 
;       Housekeeping dimensioned 7 x nset 
        housekp = transpose( housekp) 
         
endif else begin 
 
        times = -1L 
        housekp = replicate( -1.e31, 7) 
 
endelse 
 
end  
;------------------------------------------------------------------------------  
pro micshk_lut, lut 
 
;       Computes Look-up Table to convert MICS Sensor Housekeeping words  
;       to physical units for plots 
 
; Inputs: None 
 
; Outputs: 
;       lut             Floating point look-up table of housekeeping values 
;                       dim 256 x 7 
 
; BLOCK: mhk_cals       Calibration coefficients for: 
;       mdpucal         MDPU Housekeeping words 
;       micscal         MICS Sensor Housekeeping 
;       mpasscal        Passive Temperature Monitors 
;       rtempcal        R Temperature Coefficients 
;       ftempcal        F Temperature Coefficients 
 
common mhk_cals, mdpucal, micscal, mpasscal, rtempcal, ftempcal 
 
;       Check if calibration file has been read 
if n_elements( micscal) eq 0 then rd_mhk_cal 
 
;       Start with all possible counts 
cnts = findgen(256) 
 
;       Output data array 
lut = fltarr(256, 7) 
 
;       Conversion algorithm for each channel 
nchan = 7 
for ichan = 0, nchan - 1 do $ 
        lut(0, ichan) = ((cnts + micscal(0,ichan)) * micscal(1, ichan) + $ 
                micscal(2, ichan)) / 1.e3 
 
end 
;------------------------------------------------------------------------------  
pro conv_micshk, counts, hkpg 
 
;       Converts MICS Sensor Housekeeping words to physical units for plots 
 
; Inputs: 
;       counts          Integer data word array, dim nmajor/16 x 2 x 8 
 
; Outputs: 
;       hkpg            Floating point array of housekeeping values 
 
; BLOCK: mics_lut 
;       lut             Look-up table for MICS Sensor HK monitors 
 
common mics_lut, lut 
 
;       Calculate look-up table on first execution 
if n_elements( lut) eq 0 then begin 
 
        micshk_lut, lut0 
        lut = lut0 
 
endif 
 
;       Number of subcom sets to process 
sz = size( counts) 
nsets = sz(1) 
nchan = sz(2) - 1 
 
;       Output array 
hkpg = fltarr( nsets, nchan) 
 
;       Process each channel separately 
for ichan = 0, nchan - 1 do begin 
 
;       Extract data and look-up table for this channel 
        cnts = counts(*, ichan+1) 
        lut_ichan = lut(*, ichan) 
         
;       Data must be within range 
        c = cnts > 0 
        c = c < 255 
 
;       Perform look-up 
        hk = lut_ichan(c) 
 
;       Fill bad data 
        fill = -1.e31 
        jbad = where(cnts lt 0, nbad) 
        if nbad gt 0 then hk( jbad) = fill 
 
;       Put data in output array 
        hkpg(0, ichan) = hk 
 
endfor 
 
;       Extract +5V monitor to correct temp monitors 
monp5v = hkpg(*, 2) 
 
;       Convert sensor electronics and bups temp to degrees 
volts = hkpg(*, 0:1) 
degrees = msensor_temp( monp5v, volts) 
hkpg(0, 0) = degrees 
 
end 
;------------------------------------------------------------------------------  
pro rd_mhk_cal, dummy 
  
;       Reads calibration factors for CAMMICE MICS Housekeeping data 
;       into common block mhk_cals 
 
; Inputs: none 
  
; BLOCK: mhk_cals       Calibration coefficients for: 
;       mdpucal         MDPU Housekeeping words 
;       micscal         MICS Sensor Housekeeping 
;       mpasscal        Passive Temperature Monitors 
;       rtempcal        R Temperature Coefficients 
;       ftempcal        F Temperature Coefficients 
 
common mhk_cals, mdpucal, micscal, mpasscal, rtempcal, ftempcal 
 
if n_elements(intp) eq 0 then intp = 0  
  
;       Get path to calibration files 
path_lz, path_prog, path_data, path_cals 
 
fname = 'mhk_cals.dat' 
fn = path_cals + fname 
errno = -1 
while errno ne 0 do begin 
 
        openr, unit, fn, /GET_LUN, ERROR=errno 
        if errno ne 0 then begin 
         
                fn = pickfile(TITLE='Find file '+fname, PATH=path_cals, GET_PATH=pth) 
                path_cals = pth 
                path_lz, path_prog, path_data, path_cals 
                 
        endif 
                 
endwhile                 
  
;       Read and print header string in file 
hdr1 = ' ' 
readf, unit, hdr1 
print,'Reading: '+hdr1 
  
;       Number of housekeeping words for MDPU 
hdr = '' 
readf, unit, hdr 
nhk = 0 
readf, unit, nhk 
 
;       Read coefficients for MDPU HK  
mdpucal=fltarr(3, nhk)  
readf,unit, mdpucal 
mdpucal = mdpucal(0:1, *) 
 
;       Number of housekeeping words for MICS Sensor 
readf, unit, hdr 
nhk = 0 
readf, unit, nhk 
  
;       Read coefficients for MICS Sensor HK 
micscal=fltarr(4, nhk)  
readf,unit, micscal 
micscal = micscal(0:2, *) 
 
;       Number of MICS Passive Temperature Monitors 
readf, unit, hdr 
npmt = 0 
readf, unit, npmt 
 
;       Read coefficients for Passive Temp Monitors 
mpasscal=fltarr(2, npmt)  
readf,unit, mpasscal 
 
;       Total number of Temp Monitors 
readf, unit, hdr 
ntmp = 0 
readf, unit, ntmp 
 
;       Read r coefficients for MDPU temp mons 
rtempcal=fltarr(2, ntmp)  
readf,unit, rtempcal 
 
;       Total number of Temp Monitors 
readf, unit, hdr 
ntmp = 0 
readf, unit, ntmp 
 
;       Read f coefficients for MDPU temp mons 
ftempcal=fltarr(4, ntmp)  
readf,unit, ftempcal 
 
end  
;------------------------------------------------------------------------------  
pro vvps_lut, lut 
 
;       Computes Look-up Table for VVPS monitor conversion 
;       using exact equation from P Lew and N Katz 
 
; Inputs: None. 
 
; Outputs:  
;       lut             Potential on ESA Plates for 0-255 counts in monitor 
 
;       Start with possible raw values 
raw = indgen( 256) 
 
;       Compute Norm Katz's n array 
offset = [16, -8, -36, -66, -97, -124, -156, -196, -224] 
multiplier = 2^findgen(9) 
i0=[ 1,32,64, 96,128,151,188,236,252] 
i1=[31,63,95,127,150,187,235,251,255] 
index = intarr(256) 
for i=0,8 do index(i0(i)) = replicate( i, i1(i)-i0(i)+1) 
n = (raw + offset(index)) * multiplier(index) 
n(0) = 0 
 
a = -0.001921473 
b = -0.00856 
lut = a * n + b 
 
end 
;------------------------------------------------------------------------------  
pro vvps_mon, counts, kilovolts 
 
;       Converts VVPS monitor counts into kilovolts between ESA plates 
;       using look-up table from vvps_lut 
 
; BLOCK: lut_vvps 
 
;       voltage         Potential on ESA Plates for 0-255 counts in monitor 
 
common lut_vvps, kvolts 
 
;       Compute look-up table first time through 
if n_elements( kvolts) eq 0 then begin 
 
        vvps_lut, lut0 
        kvolts = lut0 
 
endif 
 
;       Restrict counts to byte range 
cnts = counts > 0 
cnts = cnts < 255 
 
;       Index into look-up table 
kilovolts = kvolts( cnts) 
 
end 
