;------------------------------------------------------------------------------ 
; File: MICS_LZ.PRO     CAMMICE Level Zero File Parsing Routines            
; Revision: 05/06/97    J L Roeder            
;------------------------------------------------------------------------------ 
; Routines:            
;	get_mtimes		Extract sector period and first time
;       get_msrts_LZ            Extract MICS sectored rate scaler counts
;       get_mfde_LZ             Extract MICS normal telemetry direct events 
;       get_mcde_LZ             Extract MICS cross-mode direct Events      
;       get_mmatrx_LZ           Extract MICS matrix scalers from major frame
;       get_msect_LZ            Extract MICS sector information           
;       get_mvalid_LZ           Extract MICS sector validity bits           
;       get_mesa_LZ              Extract MICS ESA step numbers
;       get_mag_LZ              Extract MICS Mag Azimuth and Elevation
;       get_mtotrts_LZ          Extract MICS spin-averaged rates counts 
;	get_mmux_LZ		Extract all MICS multiplex data 
;       get_msing_LZ            Extract MICS selected-sector singles counts   
;	get_mfstat_LZ		Extract MICS status bits from fixed column mux 
;       get_mtmstat_LZ          Extract MDPU TM Status bits from word 15
;       get_mword15_LZ          Extract all MDPU Status bits from Word 15 
;       get_mword12_LZ          Extract all MDPU Housekeeping from Word 12 
;------------------------------------------------------------------------------ 
pro get_mtmaster, mfrm, tmaster, mframe, majort

;	Obtain major frame counter and start time of master frame

; Inputs: none

; Outputs:
;	mfrm		MICS DPU Major Frame Counter, 0-10
;	tmaster		UT of start of first sector of present master frame

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
    
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
    
;	Major and minor frame period in long MICROseconds
tmajor = 9200000L
tminor = tmajor / 250L

;       Get minor frame quality bytes from header  
qual = drh.mnfr_qua  
  
;      Extract major frame counter    
if qual(0) eq 0B then mfrm = subcom(0, 0) and 15B else mfrm = 255B
    
;	Minor frame offset of master period
if qual(5) eq 0B then mframe = subcom( 5, 0) else mframe = 255B

;	Time of start of first sector in this master frame, millisec
if (mfrm ne 255B) and (mframe ne 255B) then $
	tmaster = drh.absolute_time(2) - (tmajor*mfrm + $
		tminor*(250 - mframe)) / 1000L else tmaster = -1L

majort = drh.absolute_time(2)

end
;------------------------------------------------------------------------------
pro get_mtimes, mfrm, tmaster, tfirst, tsector

;	Obtain sector period and time of the first sector in the present 
;	major frame

; Inputs: none

; Outputs:
;	mfrm		MICS DPU Major Frame Counter, 0-10
;	tmaster		UT of start of first sector of present master frame
;	tfirst		UT of center of first sector in this major frame
;			in long integer milliseconds
;	tsector		Length of sector in long MICROseconds
;			Spin Period / 32 in spin sync mode,
;			Minor Frame Period * 5 in TM sync mode

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
    
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
    
;	Major and minor frame periods in long MICROseconds
tmajor = 9200000L
tminor = tmajor / 250L

;	Master frame start time and major frame counter
get_mtmaster, mfrm, tmaster

;	TM status from two sources
tm1 = ishft( subcom( 0, 0) and 'F0'XB, -4)
tm2 = ishft( hk( 8, 1) and 'F0'XB, -4)

;	MICS DPU synchronization mode, 0=Spin, 1=PCM from three sources:
;	sm1 from TM Status in word 15 mf 0, 
;	sm2 from MDPU Status in word 12 mf 8, and
;	sm3 from then SM bit in word 15 mf 3
sm1 = ishft( subcom( 0, 0) and '10'XB, -4)
sm2 = ishft( hk( 8, 1) and '01'XB, -4)
sm3 = ishft( subcom( 3, 0) and '90'XB, -7)

;	To go into spin sync mode the DPU must be in "sensor data" mode
;	and all three of these spin sync bits must be zero
if (tm1 gt 3) and (tm2 gt 3) and (sm1 eq 0B) and (sm2 eq 0B) and $
	(sm3 eq 0B) then sync_mode = 0B else sync_mode = 1B

;	Sector period tied to spin only in spin syncronized mode
if sync_mode eq 0 then begin

;	Spacecraft spin period in long MICROseconds from MDPU
;	housekeeping subcom or spacecraft attitude file
	mspin_period, tmaster, tspin

;	Thirty-two sectors per spin, MICROsec
	tsector = tspin / 32L

endif else tsector = tmajor / 50L	; five minor frame periods, MICROsec

;	Time of center of first sector in the this major frame, millisec
if (tmaster ne -1L) and (tsector ne -1L) then $
	tfirst = tmaster + long( tsector * (50 * mfrm + 0.5)) / 1000L $
	else tfirst = -1L

end
;------------------------------------------------------------------------------ 
pro get_msrts_LZ, ichannels, cnts   
           
;       Extract MICS Sectored Rate Scaler Compressed Counts from Level 
;       Zero data arrays           
   
; Inputs:           
;       ichannels       Array of rate scaler channel numbers for sectored  
;                       rates only, 0-8 for MR0-8 and 9-10 for DCR and FSR 
; Outputs:           
;	times		Universal times of accumulation interval centers, msec
;       cnts            Compressed accumulated counts           
    
; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
    
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
    
;               Get Validity bits   
get_mvalid_LZ, valid   
jvalid = where( valid ne 0B, nvalid)
   
;       Number of channels to extract           
nchan = n_elements( ichannels)           
    
;       Check if any valid sectors in this frame
if nvalid ne 0L then begin

;       Setup output array           
	cnts = bytarr( nvalid, nchan)           
    
;       Index array is offset from channels           
	index = ichannels + 3           
    
;       Extract channels one at a time           
	for jchan = 0, nchan-1 do $
		cnts( 0, jchan) = mics_fixed(jvalid, index( jchan)) 
  
;	Output array is dimensioned nchannels x nvalid
	cnts = transpose( cnts)  

endif else cnts = replicate( 255B, nchan, 1)

end           
;------------------------------------------------------------------------------ 
pro get_mfde_LZ, tof, eng   
           
;       Extract MICS Direct Events from MICS Fixed-Column data array only  
;       This does not include those events in the HIT Fixed-Column array 
;       when in Cross mode           
    
; Inputs: none           
    
; Outputs:           
;       tof             Time-of-flight in channel numbers           
;       eng             Energy in channel numbers           
    
; BLOCK: camLZ_data            
;       recnumb     Record number of data            
;       drh         Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom  Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15  
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
         
;       Extract lower significant bytes of TOF           
tof_lsb = mics_fixed(*, 1)           
         
;       Extract most significant two bits of TOF           
tof_msb = mics_fixed(*, 0) and 12B           
         
;       Combine lower and upper parts of TOF into int           
tof = fix( tof_lsb) + 64 * tof_msb           
         
;       Extract lower significant bytes of ENG           
eng_lsb = mics_fixed(*, 2)           
         
;       Extract most significant two bits of ENG           
eng_msb = mics_fixed(*, 0) and 3B           
         
;       Combine lower and upper parts of TOF into int           
eng = fix( eng_lsb) + 256 * eng_msb           
         
;               Get Validity bits   
get_mvalid_LZ, valid   
jvalid = where( valid ne 0B)   
   
;               Truncate data at end of master frame   
if jvalid(0) ne -1 then begin   
        tof = tof( jvalid)   
        eng = eng( jvalid)   
endif else begin   
        tof = -1   
        eng = -1   
endelse   
   
end            
;------------------------------------------------------------------------------ 
pro get_mcde_LZ, tof, eng, mgrp, moqgrp      
      
;       Extract MICS Direct Events from HIT Fixed-Column data array only 
;       when in Cross mode. Does not include events from MICS Fixed-      
;       Column array      
      
; Inputs: none           
    
; Outputs:           
;       tof         Time-of-flight in channel numbers           
;       eng         Energy in channel numbers      
;       mgrp        Mass group number      
;       moqgrp      Mass/Charge group number      
      
; BLOCK: camLZ_data            
;       recnumb     Record number of data            
;       drh         Data record header            
;       hk          Housekeeping bytes dim 250 x 2           
;       subcom      Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed  Fixed-Column MICS Science Allocation, dim 50 x 15            
;       hit_fixed   Fixed-Column HIT Science Allocation, dim 50 x 15           
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
         
;       Reform cross telemetry dim 250 x 3           
cross = transpose( reform( transpose( hit_fixed), 3, 250))      
         
;               Get Validity bits   
get_mvalid_LZ, valid   
jvalid = where( valid ne 0B)   
nvalid = n_elements( jvalid)   
   
if jvalid(0) ne -1 then begin   
   
;   Construct index for minor frames that contain sector information      
        sect_index = indgen( nvalid) * 5      
        sect_flag = bytarr( 5 * nvalid)      
        sect_flag( sect_index) = 1B      
        nonsect_index = where( sect_flag eq 0B)      
      
;       Extract least significant eight bits of TOF           
        tof_lsb = cross( *, 1)      
        tof_lsb = tof_lsb( nonsect_index)          
      
;       Extract most significant two bits of TOF           
        tof_msb = cross( *, 0) and 12B      
        tof_msb = tof_msb( nonsect_index)           
         
;       Combine lower and upper parts of TOF into int           
        tof = fix( tof_lsb) + 64 * tof_msb           
         
;       Extract lower significant bytes of ENG           
        eng_lsb = cross( *, 2)      
        eng_lsb = eng_lsb( nonsect_index)          
      
;       Extract most significant two bits of ENG           
        eng_msb = cross(*, 0) and 3B           
        eng_msb = eng_msb( nonsect_index)      
            
;       Combine lower and upper parts of TOF into int           
        eng = fix( eng_lsb) + 256 * eng_msb           
      
;       Extract the mass group numbers      
        mgrp = ishft( cross( *, 0) and 240B, -4)      
        mgrp = mgrp( nonsect_index)      
      
;       Extract the mass/charge group numbers bytes      
        mqa = cross( *, 0) and 15B      
        mqa = mqa( sect_index)      
      
        mqb = cross( *, 1)      
        mqb = mqb( sect_index)      
      
        mqc = cross( *, 2)      
        mqc = mqc( sect_index)      
      
;       Setup output array      
        moqgrp = bytarr( nvalid, 4)      
      
;       Combine various bits into mass/charge groups      
        moqgrp(0, 0) = ishft( mqa, 1) + ishft( mqb and 128B, -7)       
        moqgrp(0, 1) = ishft( mqb and 124B, -2)      
        moqgrp(0, 2) = ishft( mqb and 3B, 3) + ishft( mqc and 224B, -5)      
        moqgrp(0, 3) = mqc and 31B      
      
;      Reform tof, eng, and m group for output as dim 50 x 4   
        tof = transpose( reform( tof, 4, nvalid))     
        eng = transpose( reform( eng, 4, nvalid))     
        mgrp = transpose( reform( mgrp, 4, nvalid))   
   
endif else begin   
   
        tof = -1   
        eng = -1   
        mgrp = -1   
        moqgrp = -1   
   
endelse   
   
end            
;------------------------------------------------------------------------------ 
pro get_mmatrx_LZ, mfc, mms, odd
     
;      Extracts the contents of the matrix scalers from major frame     
;      arrays     
     
; Inputs: None     
     
; Outputs:     
;	mfc		Major frame counter, 0-10      
;	mms		Matrix scaler contents, compressed
;	odd		Even-odd flag, 0=first 16 ESA steps, 1=second 16 steps
     
; BLOCK: camLZ_data            
;       recnumb     Record number of data            
;       drh         Data record header            
;       hk          Housekeeping bytes dim 250 x 2           
;       subcom      Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed  Fixed-Column MICS Science Allocation, dim 50 x 15            
;       hit_fixed   Fixed-Column HIT Science Allocation, dim 50 x 15           
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
         
;       Get minor frame quality bytes from header   
qual = drh.mnfr_qua   
   
;      Extract major frame counter     
if qual(0) eq 0B then mfc = (subcom(0, 0) and 15B) else mfc = 255B 
     
;       Check if major frame counter is good 
if mfc ne 255B then begin 
 
;       Extract matrix scaler contents     
        mms = subcom( 13:60, 0) * (qual(13:60) eq 0B) + $ 
                255B * (qual(13:60) ne 0B)    
     
;      Truncate array if end of experiment master frame     
        if mfc eq 10B then mms = mms(0:31)     
 
;	Even-odd flag from VVPS status found in loose bytes
	vvps_stat = subcom( 1, 0)
	vvps_on = (vvps_stat ge '80'XB)
	odd = vvps_on * (vvps_stat and '01'XB) + 255B * (1 - vvps_on)

endif else begin

;	Can't sync up so fill outputs
	odd = 255B
	mms = -1B 

endelse
     
end      
;------------------------------------------------------------------------------ 
pro get_msect_LZ, times, sector, delta, mfrm
         
;       Extract MICS sector information from Level Zero record           
         
; Inputs: None    
           
; Outputs:         
;	times	Universal times of center of each sector
;       sector  Sector identification number, 0-31,    
;		set to 255 if invalid or filled
;	delta	Accumulation times in byte msec
;	mfrm	MICS major frame index, 0-10
         
; BLOCK: camLZ_data            
;       recnumb     Record number of data            
;       drh         Data record header            
;       hk          Housekeeping bytes dim 250 x 2           
;       subcom      Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed  Fixed-Column MICS Science Allocation, dim 50 x 15
;       hit_fixed   Fixed-Column HIT Science Allocation, dim 50 x 15           
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
         
;      Get minor frame quality bits for first and last minor frame in sector
qs = transpose( reform( drh.mnfr_qua, 5, 50))   
qual_sect0 = qs(*, 0)
qual_sect4 = qs(*, 4)
              
;               Get Validity bits   
get_mvalid_LZ, valid   
   
;	Number of valid and invalid sectors
jvalid = where( valid ne 0B, nvalid)
jinvalid = where( valid eq 0B, ninvalid)   

;	No valid sectors?
if nvalid eq 0L then begin
	times = -1L
	sector = 255B
	return
endif

;       Extract lower significant three bits of sector number           
sect_lsb = (ishft( mics_fixed(*, 0) and 112B, -4)) * (qual_sect0 eq 0B) + $
	255B * (qual_sect0 ne 0B)
        
;       Find index where the lsb are zero, this is where the msb may           
;       be accessed           
jsect0 = where((sect_lsb eq 0B) and (valid ne 0B)) 
njsect0 = n_elements( jsect0)       
         
;       Check if any good data found   
if jsect0(0) ne -1L then begin   
   
;       Extract upper two bits of sector number from MUX data           
        mux = mics_fixed(*, 14)           
        sect_msb = ishft( mux( jsect0) and 192B, -3)
	sect_msb = sect_msb * (qual_sect4(jsect0) eq 0B) + $
		255B * (qual_sect4(jsect0) ne 0B)
         
;       Replicate to get msb for each sector separately           
	nsm = (njsect0 * 8) < (50-jsect0(0))
        msbs = sect_msb( lindgen( nsm) / 8L)
	loc_msb = (lindgen( nsm) mod 8) + jsect0( lindgen( nsm) / 8)
	sector_msb = replicate( 255B, 50)
        sector_msb( loc_msb) = msbs
         
;       Combine lower and upper parts of sector number into int           
        sector = (sect_lsb + sector_msb) * ((sect_lsb ne 255B) and $
		(sector_msb ne 255B)) + 255B * ((sect_lsb eq 255B) or $
		(sector_msb eq 255B)) 

;       Set the first few sector msb's to the first sector - 1           
        if jsect0(0) ne 0 then begin   
   
                jfirst_sect = jsect0(0)   
                first_sect = sector( jfirst_sect)         
                last_sect = first_sect - 1B         
                if last_sect lt 0 then last_sect = 31B          
                sector(0) = (indgen( jfirst_sect) + last_sect $   
                        - jfirst_sect + 1)  mod 32B        
   
        endif   
   
endif else sector = replicate( 255B, nvalid)   

;	Any valid sectors
if nvalid ne 0L then begin

;	Eliminate invalid sectors
	sector = sector( jvalid)
          
;	Obtain sector interval and first time
	get_mtimes, mfrm, tmaster, tfirst, tsector

;	Compute time and length of accumulation for each valid sector
	if (tfirst ne -1) and (tsector ne -1L) then begin

		times = (( jvalid * tsector) / 1000L + tfirst) * $
			(sector ne 255B) - long( sector eq 255B)

		tdead = 58	; dead time in milliseconds
		delta = byte( (sector ne 255B) * tsector / 1000 $
			- (sector eq 0B) * tdead + 255 * (sector eq 255B))

	endif else begin

		times = replicate( -1L, nvalid)
		delta = replicate( 255B, nvalid)

	endelse

endif else begin

	sector = 255B
	times = -1L
	delta = 255B

endelse

end            
;------------------------------------------------------------------------------ 
pro get_mvalid_LZ, valid           
         
;       Extract MICS validity bits from Level Zero record           
         
; Inputs: none           
         
; Outputs:           
;       valid   Valid data bit is zero if sector data is invalid           
         
; BLOCK: camLZ_data            
;       recnumb     Record number of data            
;       drh         Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom  Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
         
;      Get minor frame quality bits for last minor frame in sector         
qual_sect = transpose( reform( drh.mnfr_qua, 5, 50))   
qual_sect = qual_sect( *, 0)   
              
;       Extract most significant bit and right shift           
valid = mics_fixed(*, 0) and 128B

;	If quality is bad then assume that data is valid
valid = (ishft( valid, -7)) * (qual_sect eq 0B) + byte( qual_sect ne 0B)
         
end            
;------------------------------------------------------------------------------ 
pro get_mesa_LZ, sector, esa, es           
         
;       Extract MICS ESA step numbers from Level Zero record           
         
; Inputs:            
;       sector      Sector identification number, 0-31           
         
; Outputs:           
;       esa         Electrostatic analyzer step number, 0-31   
;       es	    ESA On = 1, Off = 0       
         
; BLOCK: camLZ_data            
;       recnumb     Record number of data            
;       drh         Data record header            
;       hk          Housekeeping bytes dim 250 x 2           
;       subcom      Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed  Fixed-Column MICS Science Allocation, dim 50 x 15            
;       hit_fixed   Fixed-Column HIT Science Allocation, dim 50 x 15           
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
         
;       Extract MUX data from fixed-column data           
mux = mics_fixed(*, 14)           
         
;      Get minor frame quality bits for last minor frame in sector         
qual_sect = transpose( reform( drh.mnfr_qua, 5, 50))   
qual_sect = qual_sect( *, 4)   
   
;       Find positions of ESA steps in MUX array           
jesa = where( ( sector eq 0B) and (qual_sect eq 0B))         
         
;               Check if any good values   
if jesa(0) ne -1L then begin   
   
;   Extract the data from those positions           
        esa_on = ishft( mux( jesa) and 32B, -5)           
        esa0 = mux( jesa) and 31B           
         
        esa = bytarr( 50)   
        es = bytarr( 50)   
        es( jesa) = esa_on   
        esa( jesa) =  esa0   
   
endif else begin   
   
        es = replicate( 255B, 50)   
        esa = replicate( 255B, 50)   
   
endelse   
         
;	Get Validity bits   
get_mvalid_LZ, valid   
jvalid = where( valid ne 0B, nvalid)   

;	Eliminate invalid sectors   
if nvalid gt 0 then begin

	esa = esa( jvalid)
	es = es( jvalid)

endif else begin

	esa = 255B
	es = 255B

endelse

end            
;------------------------------------------------------------------------------ 
pro get_mmag_LZ, sector, magaz, magel           
         
;       Extract MICS MagAz and MagEl from Level Zero record           
         
; Inputs:            
;       sector  Sector identification number, 0-31 to index MUX data           
         
; Outputs:           
;       magaz   Magnetic field azimuth           
;       magel   Magnetic field elevation           
         
; BLOCK: camLZ_data            
;       recnumb     Record number of data            
;       drh         Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom  Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15  
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
         
;       Extract MUX data from fixed-column data           
mux = mics_fixed(*, 14)           
         
;       Indexes for positions in MUX data           
index_az = 17           
index_el = 1           
         
;       Find data for those sectors in this major frame           
jmagaz = where( sector eq index_az)           
jmagel = where( sector eq index_el)           
         
;       Extract the MUX data for these positions           
if jmagaz(0) ne -1 then magaz = mux( jmagaz) else magaz = -1B          
if jmagel(0) ne -1 then magel = mux( jmagel) else magel = -1B           
         
end            
;------------------------------------------------------------------------------ 
pro get_mtotrts_LZ, ichannels, sector, times, delta, chans, esa, cnts           
         
;       Extract MICS non-sectored spin-averaged rate scaler compressed counts 
;       from a Level Zero record           
         
; Inputs:            
;       ichannels    Array of rate scaler channel numbers for non-sectored 
;                    spin-averaged rates only, 0-15 for MR0-15           
;       sector       Sector identification number, 0-31           
         
; Outputs:
;	times        Universal times of spin centers, long msec
;	delta	     Accumulation times in msec (1 spin period)
;       chans        Channel identification numbers
;	esa	     ESA step numbers, 0-31B
;       cnts         Compressed accumulated counts           
         
common camLZ_data, 	$ ; BLOCK for raw cammice lz data
	recnumb,	$ ; Record number of data
	drh,       	$ ; Data record header
	hk, 	   	$ ; Housekeeping bytes dim 250 x 2
	subcom,    	$ ; Subcom (Loose) bytes dim 250 x 2
	mics_fixed,	$ ; Fixed-Column MICS Science Allocation, dim 50 x 15
	hit_fixed  	  ; Fixed-Column HIT Science Allocation, dim 50 x 15
         
common last_mtot,   	$ ; BLOCK for selected items from previous record
	lstrec,    	$ ; record number of previous record accessed
	lstmstr,	$ ; Start time of master frame for previous record
	lstesas, 	$ ; ESA step number of all spins in previous record
	maxlstesa	  ; Last spin for which esa has been stored

;	Initialize common block first time through
if n_elements( lstrec) eq 0L then begin

	lstrec = -99L
	lstmstr = -1L
	lstesas = replicate( 255B, 20)
	maxlstesa = 0L

endif

;	Quality flags for last minor frame in every sector
qual_mux = transpose( reform( drh.mnfr_qua, 5, 50))
qual_mux = qual_mux( *, 4)

;       Extract MUX data from fixed-column data           
mux = mics_fixed(*, 14) * (qual_mux eq 0B) + 255B * (qual_mux ne 0B)          
         
;       Locations of spin-averaged rates in MUX array           
indx = indgen(4)           
sect_index = [indx+4, indx+12, indx+20, indx+28]           
         
;       Setup channel id array           
chans = replicate( -1B, 50)           
         
;       How many channels to extract           
nchans = n_elements( ichannels)           
         
;       Mark data one channel at a time           
for ichan = 0, nchans - 1 do begin           
        jchan = ichannels( ichan)           
        jindex = where( sector eq sect_index( jchan))           
        if jindex(0) ne -1 then chans( jindex) = jchan           
endfor           
         
;       Which data were marked?           
kchans = where( chans ne 255B, nkchans) 
if nkchans ne 0L then begin           
         
;       Extract the data which were marked           
        chans = chans( kchans)           
        cnts = mux( kchans)           

;	Obtain sector interval and first time
	get_mtimes, mfrm, tmaster, tfirst, tsector

;	Spin index for marked data 
	kspins = (kchans + mfrm*50L) / 32L

;	Universal times of spin centers delayed by one spin
	times = tmaster + long( 32L * tsector * (kspins - 0.5)) / 1000L

;	Accumulation times in milliseconds
	delta = replicate( 32L * tsector / 1000L, nkchans)

;       At beginning of master must calculate time with previous master time
        zspins = where( kspins lt 1L, nzspins)
        if nzspins gt 0L then begin
                if recnumb eq lstrec+1 then times( zspins) = lstmstr + $
                        long( 32L * tsector * (maxlstesa - 0.5) ) / 1000L $
			else times( zspins) = -1L
        endif

;	ESA step numbers are in sector zero
	jesa = where( sector eq 0B, nesa)

;	Check if any sector zeros found
	if (nesa gt 0L) and (mfrm ne 255B) then begin

;		Spin numbers of esa steps not delayed by one spin
;		First spin in master frame is number one
		espins = (jesa + mfrm*50L) / 32L + 1L

;		ESA step values
		esa_step = mux( jesa) and 31B
		esa_on = mux( jesa) and 32B
		esa0 = esa_step * (esa_on ne 0B) + 255B * (esa_on eq 0B)

;		Load ESA value for each possible spin number
		esa_spins = replicate( 255B, 21)
		esa_spins( espins) = esa0

;		Load step values into output buffer
		esa = esa_spins( kspins)

;		Find sectors with ESA step values from previous records
		prev_esa = where( kspins lt min( espins), nprev)
		if (nprev gt 0L) and (recnumb eq lstrec+1) then begin

			kprev = kspins( prev_esa)
			if max(kprev) lt 1 then kprev = kprev + maxlstesa
			esa( prev_esa) = lstesas( kprev)

		endif

		if (mfrm eq 0) or (recnumb ne lstrec+1) then $
			lstesas = replicate( 255B, 20)
		lstesas( espins) = esa0
		maxlstesa = max( where( lstesas ne 255B))
		lstmstr = tmaster
		lstrec = recnumb

	endif else esa = replicate( 255B, n_elements( kspins))

endif else begin
         
;       If no data marked then output -1's           
	times = -1L
	delta = -1L
        chans = 255B           
        esa = 255B
        cnts = 255B

endelse

end           
;------------------------------------------------------------------------------ 
pro get_mmux_LZ, mux_data
         
;       Extract MICS multiplex data from a Level Zero record           
         
; Inputs: None
         
; Outputs:           
;	mux_data	Data from multiplex channel in fixed column words
;			dim 50 bytes

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;      Get minor frame quality bits for last minor frame in sector
qual_sect = transpose( reform( drh.mnfr_qua, 5, 50))
qual_sect = qual_sect( *, 4)

;       Extract MUX data from fixed-column data           
mux = mics_fixed(*, 14) * (qual_sect eq 0B) + 255B * (qual_sect ne 0B)
         
;               Get Validity bits
get_mvalid_LZ, valid

;       Number of valid and invalid sectors
jvalid = where( valid ne 0B, nvalid)

;	Output only valid sectors
if nvalid ne 0 then mux_data = mux( jvalid) else mux_data = 255B

end           
;------------------------------------------------------------------------------ 
pro get_msing_LZ, sector, tspins, tsects, chans, esa, data   
         
;       Extract MICS selected-sector singles rate scaler compressed counts 
;       from a Level Zero record           
         
; Inputs:            
;       sector       Sector identification number, 0-31           
         
; Outputs:
;	tspins	Universal times of spin starts, long msec
;	tsects	Sector period, long MICROSEC
;       ids	Data identification numbers, 0-9B, in telemetry order:
;		0 = MS0-A TCR,
;		1 = MS0-B TCR,
;		2 = Sector-A
;		3 = MS1-A MSS, 
;		4 = MS1-B MSS,
;		5 = MS2-A P, 
;		6 = MS2-B P,
;		7 = Sector-B,
;		8 = MS3-A ALPHA,
;		9 = MS3-B ALPHA
;	esa	ESA step numbers, 0-31B
;	data	Compressed accumulated counts, 0-255B    

common camLZ_data, 	$ ; BLOCK for raw cammice lz data
	recnumb,	$ ; Record number of data
	drh,       	$ ; Data record header
	hk, 	   	$ ; Housekeeping bytes dim 250 x 2
	subcom,    	$ ; Subcom (Loose) bytes dim 250 x 2
	mics_fixed,	$ ; Fixed-Column MICS Science Allocation, dim 50 x 15
	hit_fixed  	  ; Fixed-Column HIT Science Allocation, dim 50 x 15
         
common last_msing,   	$ ; BLOCK for selected items from previous record
	lstrec,    	$ ; record number of previous record accessed
	lstmstr,	$ ; Start time of master frame for previous record
	lstesas, 	$ ; ESA step number of all spins in previous record
	maxlstesa	  ; Last spin for which esa has been stored

;	Initialize common block first time through
if n_elements( lstrec) eq 0L then begin

	lstrec = -99L
	lstmstr = -1L
	lstesas = replicate( 255B, 20)
	maxlstesa = 0L

endif

;	Quality flags for last minor frame in every sector
qual_mux = transpose( reform( drh.mnfr_qua, 5, 50))
qual_mux = qual_mux( *, 4)

;       Extract MUX data from fixed-column data           
mux = mics_fixed(*, 14) * (qual_mux eq 0B) + 255B * (qual_mux ne 0B)          
         
;       Locations of singles rates in MUX array           
sect_index = [2,3,9,10,11,18,19,25,26,27]
         
;       Setup channel id array           
chans = replicate( 255B, 50)           
         
;       Mark data one channel at a time           
for ichan = 0, 9 do begin           
        jindex = where( sector eq sect_index( ichan))           
        if jindex(0) ne -1 then chans( jindex) = ichan           
endfor           
         
;       Which data were marked?           
kchans = where( chans ne 255B, nkchans) 
if nkchans ne 0L then begin           
         
;       Extract the data which were marked           
        chans = chans( kchans)           
        data = mux( kchans)           

;	Obtain sector interval and first time
	get_mtimes, mfrm, tmaster, tfirst, tsector

;	Spin index for marked data delayed by one spin
;	First spin in master frame is number one
	kspins = (kchans + mfrm*50L) / 32L

;	Universal times of spin starts delayed by one spin
	tspins = tmaster + long( 32L * tsector * (kspins-1) ) / 1000L

;	At beginning of master must calculate time with previous master time
	zspins = where( kspins lt 1L, nzspins)
	if nzspins gt 0L then begin
		if recnumb eq lstrec+1 then tspins( zspins) = lstmstr + $
			long( 32L * tsector * (maxlstesa - 1)) / 1000L else $
			tspins( zspins) = -1L
	endif

;	ESA step numbers are in sector zero
	jesa = where( sector eq 0B, nesa)

;	Check if any sector zeros found
	if (nesa gt 0L) and (mfrm ne 255B) then begin

;		Spin numbers of esa steps not delayed by one spin
;		First spin in master frame is number one
		espins = (jesa + mfrm*50L) / 32L + 1L

;		ESA step values
		esa_step = mux( jesa) and 31B
		esa_on = mux( jesa) and 32B
		esa0 = esa_step * (esa_on ne 0B) + 255B * (esa_on eq 0B)

;		Load ESA value for each possible spin number
		esa_spins = replicate( 255B, 21)
		esa_spins( espins) = esa0

;		Load step values into output buffer
		esa = esa_spins( kspins)

;		Find sectors with ESA step values from previous records
		prev_esa = where( kspins lt min( espins), nprev)
		if (nprev gt 0L) and (recnumb eq lstrec+1) then begin

			kprev = kspins( prev_esa)
			if max(kprev) lt 1 then kprev = kprev + maxlstesa
			esa( prev_esa) = lstesas( kprev)

		endif

		if (mfrm eq 0) or (recnumb ne lstrec+1) then $
			lstesas = replicate( 255B, 20)
		lstesas( espins) = esa0
		maxlstesa = max( where( lstesas ne 255B))
		lstmstr = tmaster
		lstrec = recnumb

	endif else esa = replicate( 255B, nkchans)

;	Sector periods constant for entire record
	tsects = replicate( tsector, nkchans)

endif else begin

;       If no data marked then output flags
        chans = 255B           
        esa = 255B
        data = 255B
	tspins = -1L
	tsects = -1L

endelse

end           
;------------------------------------------------------------------------------ 
pro get_mtmstat_LZ, tmstat
         
;       Extract MICS DPU TM status only from a Level Zero record           
         
; Inputs: None.
         
; Outputs:
;       tmstat          MDPU TM status, 0-15

; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;       Minor frame quality flags
qual = drh.mnfr_qua

;       MICS DPU telemetry status and quality for word 12
q= qual(8)
tmstat = ishft( hk(8, 1) and 'F0'XB, -4)

end           
;------------------------------------------------------------------------------ 
pro get_mfstat_LZ, sector, mfstat
         
;       Extract MICS status bits from fixed column mux channels
         
; Inputs:            
;       sector          Sector identification number, 0-31           
         
; Outputs:           
;       mfstat          Three byte array of status bits
         
; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15 
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;       Extract MUX data from fixed-column data           
mux = mics_fixed(*, 14)           
         
;       Locations of status bytes in MUX array           
sect_index = (1 + indgen(3)) * 8           
         
;       Setup id array           
id = replicate( -1, 50)           
         
;       Mark data one channel at a time           
for istat = 0, 2 do begin           
        jindex = where( sector eq sect_index( istat))          
        if jindex(0) ne -1 then id( jindex) = istat
endfor           
         
;       Which data were marked?           
jstats = where( id ne -1)           
 
;       Check if any data available 
if jstats(0) ne -1 then begin           
         
;	Compute index numbers of marked bytes
	kindex = (jstats - jstats(0)) / 32
	index = kindex * 3 + istat

;	Setup output array
	mfstat = bytarr( 3, max(kindex) + 1)

;       Extract the status data which were marked           
        mfstat(index) = mux( jstats)           
         
endif else begin           
         
;       If no data marked then output one set of 255's
        mfstat = replicate( 255B, 3)           
         
endelse           
         
end           
;------------------------------------------------------------------------------ 
pro get_mword15_LZ, time, mdpu_stat15 
   
;       Extract contents of Word 15 from a Level Zero Record   
 
; Outputs: 
;	time		Universal time of major frame, long millisec
;       mdpu_stat15     Array of relevant bytes from Word 15 converted to 
;                       ints with bad data flagged as -1's 
   
; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15  
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;	Universal time of this Mf from header
time = drh.absolute_time(2)

;       Get minor frame quality bytes from header   
qual = drh.mnfr_qua   
   
;       Get subcom 0 from common block   
scom0 = subcom(*, 0)   
 
;       Index of relevant minor frames
index = [0, 1, 2, 3, 4, 5, 11, 12]
 
;       Extract and re-order data and quality bytes 
scom = scom0( index) 
q = qual( index) 
 
;       Flag bad minor frames with -1 in output array 
mdpu_stat15 = fix(scom) * (q eq 0B) - fix(q ne 0B) 
 
end   
;------------------------------------------------------------------------------ 
pro get_mword12_LZ, time, mdpu_hk 
   
;       Extract contents of Word 12 from a Level Zero Record   
 
; Outputs: 
;	time		Universal time of major frame, long millisec
;       mdpu_hk         Array of relevant bytes from Word 12 converted to 
;                       ints with bad data flagged as -1's 
   
; BLOCK: camLZ_data            
;       recnumb         Record number of data            
;       drh             Data record header            
;       hk              Housekeeping bytes dim 250 x 2           
;       subcom          Subcom (Loose) bytes dim 250 x 2           
;       mics_fixed      Fixed-Column MICS Science Allocation, dim 50 x 15 
;       hit_fixed       Fixed-Column HIT Science Allocation, dim 50 x 15  
         
common camLZ_data, recnumb, drh, hk, subcom, mics_fixed, hit_fixed           
           
;	Universal time of this Mf from header
time = drh.absolute_time(2)

;       Get minor frame quality bytes from header   
qual = drh.mnfr_qua   
   
;       Get subcom 0 from common block   
hk0 = hk(*, 1)   
 
;       Index of relevant minor frames; note re-ordering to group together 
;       like items, and put multi-byte integers into VAX order 
index = [8, 20, 44, 56, 116, 128, 140, 152, $ 
        32, 164, 176, 188, 68, 200, 104, 92, 80, 236, 224, 212, 248] 
 
;       Extract and re-order data and quality bytes 
hk = hk0( index) 
q = qual( index) 
 
;       Flag bad minor frames with -1 in output array 
mdpu_hk = fix(hk) * (q eq 0B) - fix(q ne 0B) 
 
end
