;------------------------------------------------------------------------------ 
; File: MICS_STA.PRO		CAMMICE MICS Status Routines             
; Revision: 02/12/96		J L Roeder             
;------------------------------------------------------------------------------ 
; Routines:             
;      def_mdpustat_strc	Create structure for MICS DPU Status
;      load_mdpustat		Loads structure with MICS housekeeping data 
;------------------------------------------------------------------------------ 
pro def_mdpustat_strc, strc, nbytes   
   
;       Define the structure to hold the MICS DPU Status bits   
   
strc = { MICS_Status,           $          
	MDPU_Status: 0B,        $ ; Status of MICS DPU from Word 12, 0-15 
	Subctr: 0B,             $ ; Subframe Counter, 0-31 
        VVPS_Mode: 0B,          $ ; VVPS Stepping Mode, 0-7 
        VVPS_Step: 0B,          $ ; VVPS Step, 0-31 
        VVPS_Monitor: 0.,       $ ; VVPS Voltage Monitor, kVolts 
        VV_HVPS_Status: 0,      $ ; Power Supply Status (S/C Generated) 
        MDPU_Pwr_Status: 0,     $ ; Analog Status (S/C Generated) 
        MDPU_Temp: 0.,          $ ; MDPU Temp (S/C Generated) 
        MICS_Temp: 0.,          $ ; MICS Sensor Temp (S/C Generated) 
        CMD: lonarr(2),         $ ; Last commands received 
        CMDCTR: intarr(2),      $ ; Command counter, 0-255 
	DUMP_ADDRESS: intarr(2),$ ; Memory dump addresses
	DUMP_VALUE: bytarr(2),	$ ; Memory dump values
        TM_Checksum: 0,         $ ; Checksum for previous major frame 
        RAL: 0B,                $ ; Rutherford VVPS Status Bit, 0-1 

        TM_status: 0B,          $ ; MDPU Status from Word 15   
        MFC:    0B,             $ ; Major Frame Counter, 0-10   
        VVPS_status: 0B,        $ ; ESA Subsystem Mode   
        RE:  0B,                $ ; SYSR Error Flag, 0=No error   
        TE:  0B,                $ ; SYST Error Flag, 0=No error   
        FPE: 0B,                $ ; SYSR Floating Point Error Flag, 0=No error
        REVOFF: 0B,             $ ; Sector Offset   
        SM: 0B,                 $ ; TM Synchronization Mode, 0=Spin, 1=PCM   
        AUXOFF: 0B,             $ ; Fine Tuning Offset   
        SDEP:   0B,             $ ; Software Direct Event Priority Buffer, 0-3
        HDEP:   0B,             $ ; Hardware Direct Event Priority Buffer, 0-3
        RSB:    0B,             $ ; Rate Scaler Select Buffer, 0-1   
        TBL:    0B,             $ ; Mass and M/Q Table Coeff Set, 0-7   
        Master_Frame: 0         $ ; Minor frame number of start of Master    
        }                         ; Period, 0-249   
   
nbytes = 56
   
end   
;------------------------------------------------------------------------------ 
pro load_mdpustat, w12, w15, mdpu_stat
 
;       Loads structure with MICS housekeeping data 
 
; Inputs: 
;	w12	Int array of data from TM Word 12 of LZ file 
;               bad data set to -1's 
;	w15	Int array of data from TM Word 15 of LZ file 
;               bad data set to -1's 
 
; Outputs: 
;	mdpu_stat	Structure filled with calibrated data 
 
;       Number of major frames to process
;	assume w12 array is nmajor x 21 
sz = size( w12) 
nmajor = sz( 1) 
 
;       Check that word 15 data has same numbers of major frames
;	assume w12 array is nmajor x 8
sz = size( w15) 
nmj15 = sz( 1) 
if nmajor ne nmj15 then stop,'Error load_mdpustat: wrong number of Mf'
 
;       Define output structure 
def_mdpustat_strc, struc, nbytes 
 
;       Create output array 
mdpu_stat = replicate( struc, nmajor) 
 
;       Load data into structure array 
;       MICS DPU Status 
tmp = w12(*, 0) 
status = ( tmp ne -1) * ishft( tmp and 'F0'X, -4) + 255 * ( tmp eq -1) 
mdpu_stat.MDPU_Status = byte( status) 
 
;       Subframe Counter 
tmp = w12(*, 1) 
sctr = ( tmp ne -1) * ( tmp and '1F'X) + 255 * ( tmp eq -1) 
mdpu_stat.SUBCTR = byte( sctr) 
 
;       RAL Bit 
rb = (tmp ne -1) * ishft( tmp and '20'X, -5) + 255 * ( tmp eq -1)  
mdpu_stat.RAL = byte( rb) 
 
;       VVPS Mode 
tmp = w12(*, 2) 
vm = (tmp ne -1) * ishft( tmp and 'E0'X, -5) + 255 * ( tmp eq -1)  
mdpu_stat.VVPS_Mode = byte( vm) 
 
;       VVPS Step 
vs = (tmp ne -1) * ( tmp and '1F'X) + 255 * ( tmp eq -1)  
mdpu_stat.VVPS_Step = byte( vs) 
 
;       VVPS Monitor 
tmp = w12(*, 3)
vvps_mon, tmp, kvolts
mdpu_stat.VVPS_Monitor = kvolts
 
;       VVPS/HVPS Power Status (SCG) 
mdpu_stat.VV_HVPS_Status = w12(*, 4) 
 
;       MDPU Power Status (SCG) 
mdpu_stat.MDPU_Pwr_Status = w12(*, 5) 
 
;       MDPU Temperature (SCG) 
counts = w12(*, 6:7)
mpass_temp, counts, degrees
mdpu_stat.MDPU_Temp = degrees(*, 0)
 
;       MICS Sensor Temperature (SCG) 
mdpu_stat.MICS_Temp = degrees(*, 1) 
 
;       Command Counter 
tmp = w12(*, 12:13) 
tmp = transpose( tmp) 
cmd_ctr = (tmp ne -1) * (tmp and '7F'X) - (tmp eq -1)
mdpu_stat.CMDCTR = cmd_ctr 
 
;	Extract two last commands for each major frame
cmdbytes = bytarr( nmajor, 8) 
cmdbytes( 0, 0) = w12(*, 14:16) 
cmdbytes( 0, 4) = w12(*, 17:19) 
cmdbytes = transpose( cmdbytes) 
cmds = long( cmdbytes, 0, 2, nmajor)

;	Swap bytes of longs on some platforms
swap_bytes, cmds

;       Last Commands valid only when msb of counter byte is zero
valid_cmd = (tmp and '80'X)
mdpu_stat.CMD = (valid_cmd eq 0) * cmds 

;	Dump data is output when commands not valid
mdpu_stat.DUMP_ADDRESS = (valid_cmd ne 0) * fix( cmds / 256L)
mdpu_stat.DUMP_VALUE = (valid_cmd ne 0) * byte( cmds and 'FF'XL)

;       Telemetry Checksum 
mdpu_stat.TM_Checksum = w12(*, 20)

;       Load data into structure array from Word 15
;       MICS DPU Status 
tmp = w15(*, 0) 
mdpu_stat.TM_status = (tmp ne -1) * ishft( tmp and 'F0'XB, -4) + $
	(tmp eq -1) * 255B
mdpu_stat.MFC = (tmp ne -1) * (tmp and '0F'XB) + (tmp eq -1) * 255
 
;	VVPS Status
tmp = w15(*, 1)
mdpu_stat.VVPS_status = (tmp ne -1) * tmp + (tmp eq -1) * 255B   
   
tmp = w15(*, 2)
mdpu_stat.RE = (tmp ne -1) * ishft( tmp and '80'XB, -7) + (tmp eq -1) * 255B
mdpu_stat.TE = (tmp ne -1) * ishft( tmp and '40'XB, -6) + (tmp eq -1) * 255B
mdpu_stat.FPE = (tmp ne -1) * ishft( tmp and '20'XB, -5) + (tmp eq -1) * 255B
mdpu_stat.REVOFF = (tmp ne -1) * (tmp and '1F'XB) + (tmp eq -1) * 255B
   
;	Synchronization Mode and Auxillary Offset
tmp = w15(*, 3)
mdpu_stat.SM = (tmp ne -1) * ishft( tmp and '80'XB, -7) + (tmp eq -1) * 255B
mdpu_stat.AUXOFF = (tmp ne -1) * (tmp and '7F'XB) + (tmp eq -1) * 255B
   
;	Table numbers
tmp = w15(*, 4)
mdpu_stat.SDEP = (tmp ne -1) * ishft( tmp and 'C0'XB, -6)   
mdpu_stat.HDEP = (tmp ne -1) * ishft( tmp and '30'XB, -4)   
mdpu_stat.RSB = (tmp ne -1) * ishft( tmp and '04'XB, -3)   
mdpu_stat.TBL = (tmp ne -1) * (tmp and '07'XB)
   
;	Minor frame number of start of master period
mdpu_stat.Master_frame = w15(*, 5)
   
end 
